% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/covidcast.R
\name{covidcast_signal}
\alias{covidcast_signal}
\title{Obtain a data frame for one COVIDcast signal}
\usage{
covidcast_signal(
  data_source,
  signal,
  start_day = NULL,
  end_day = NULL,
  geo_type = c("county", "hrr", "msa", "dma", "state", "hhs", "nation"),
  geo_values = "*",
  as_of = NULL,
  issues = NULL,
  lag = NULL,
  time_type = c("day", "week")
)
}
\arguments{
\item{data_source}{String identifying the data source to query. See
\url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast_signals.html}
for a list of available data sources.}

\item{signal}{String identifying the signal from that source to query. Again,
see \url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast_signals.html}
for a list of available signals.}

\item{start_day}{Query data beginning on this date. Date object, or string in
the form "YYYY-MM-DD". If \code{start_day} is \code{NULL}, defaults to first day data
is available for this signal.}

\item{end_day}{Query data up to this date, inclusive. Date object or string
in the form "YYYY-MM-DD". If \code{end_day} is \code{NULL}, defaults to the most
recent day data is available for this signal.}

\item{geo_type}{The geography type for which to request this data, such as
"county" or "state". Defaults to "county". See
\url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast_geography.html}
for details on which types are available.}

\item{geo_values}{Which geographies to return. The default, "*", fetches all
geographies. To fetch specific geographies, specify their IDs as a vector
or list of strings. See
\url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast_geography.html}
for details on how to specify these IDs.}

\item{as_of}{Fetch only data that was available on or before this date,
provided as a \code{Date} object or string in the form "YYYY-MM-DD". If \code{NULL},
the default, return the most recent available data. Note that only one of
\code{as_of}, \code{issues}, and \code{lag} should be provided; it does not make sense to
specify more than one. For more on data revisions, see
"Issue dates and revisions" below.}

\item{issues}{Fetch only data that was published or updated ("issued") on
these dates. Provided as either a single \code{Date} object (or string in the
form "YYYY-MM-DD"), indicating a single date to fetch data issued on, or a
vector specifying two dates, start and end. In this case, return all data
issued in this range. There may be multiple rows for each observation,
indicating several updates to its value. If \code{NULL}, the default, return the
most recently issued data.}

\item{lag}{Integer. If, for example, \code{lag = 3}, then we fetch only data that
was published or updated exactly 3 days after the date. For example, a row
with \code{time_value} of June 3 will only be included in the results if its
data was issued or updated on June 6. If \code{NULL}, the default, return the
most recently issued data regardless of its lag.}

\item{time_type}{The temporal resolution to request this data. Most signals
are available at the "day" resolution (the default); some are only
available at the "week" resolution, representing an MMWR week ("epiweek").}
}
\value{
\code{covidcast_signal} object with matching data. The object is a data
frame with additional metadata attached. Each row is one observation of one
signal on one day in one geographic location. Contains the following
columns:

\item{data_source}{Data source from which this observation was obtained.}
\item{signal}{Signal from which this observation was obtained.}
\item{geo_value}{String identifying the location, such as a state name or
county FIPS code.}
\item{time_value}{Date object identifying the date of this observation. For
data with \code{time_type = "week"}, this is the first day of the corresponding
epiweek.}
\item{issue}{Date object identifying the date this estimate was issued.
For example, an estimate with a \code{time_value} of June 3 might have been
issued on June 5, after the data for June 3rd was collected and ingested
into the API.}
\item{lag}{Integer giving the difference between \code{issue} and \code{time_value},
in days.}
\item{value}{Signal value being requested. For example, in a query for the
"confirmed_cumulative_num" signal from the "usa-facts" source, this would
be the cumulative number of confirmed cases in the area, as of the given
\code{time_value}.}
\item{stderr}{Associated standard error of the signal value, if available.}
\item{sample_size}{Integer indicating the sample size available in that
geography on that day; sample size may not be available for all signals,
due to privacy or other constraints, in which case it will be \code{NA}.}

Consult the signal documentation for more details on how values and
standard errors are calculated for specific signals.

The returned data frame has a \code{metadata} attribute containing metadata
about the signal contained within; see "Metadata" below for details.
}
\description{
Obtains data for selected date ranges for all geographic regions of the
United States. Available data sources and signals are documented in the
COVIDcast signal documentation.
Most (but not all) data sources are available at the county level, but the
API can also return data aggregated to metropolitan statistical areas,
hospital referral regions, or states, as desired, by using the \code{geo_type}
argument.
}
\details{
For data on counties, metropolitan statistical areas, and states, this
package provides the \code{\link{county_census}}, \code{\link{msa_census}}, and \code{\link{state_census}}
datasets. These include each area's unique identifier, used in the
\code{geo_values} argument to select specific areas, and basic information on
population and other Census data.

Downloading large amounts of data may be slow, so this function prints
messages for each chunk of data it downloads. To suppress these, use
\code{\link[base:message]{base::suppressMessages()}}, as in
\code{suppressMessages(covidcast_signal("fb-survey", ...))}.
}
\section{Metadata}{


The returned object has a \code{metadata} attribute attached containing basic
information about the signal. Use \code{attributes(x)$metadata} to access this
metadata. The metadata is stored as a data frame of one row, and contains the
same information that \code{covidcast_meta()} would return for a given signal.

Note that not all \code{covidcast_signal} objects may have all fields of metadata
attached; for example, an object created with \code{as.covidcast_signal()} using
data from another source may only contain the \code{geo_type} variable, along with
\code{data_source} and \code{signal}. Before using the metadata of a \code{covidcast_signal}
object, always check for the presence of the attributes you need.
}

\section{Issue dates and revisions}{


The COVIDcast API tracks updates and changes to its underlying data, and
records the first date each observation became available. For example, a data
source may report its estimate for a specific state on June 3rd on June 5th,
once records become available. This data is considered "issued" on June 5th.
Later, the data source may update its estimate for June 3rd based on revised
data, creating a new issue on June 8th. By default, \code{covidcast_signal()}
returns the most recent issue available for every observation. The \code{as_of},
\code{issues}, and \code{lag} parameters allow the user to select specific issues
instead, or to see all updates to observations. These options are mutually
exclusive, and you should only specify one; if you specify more than one, you
may get an error or confusing results.

Note that the API only tracks the initial value of an estimate and \emph{changes}
to that value. If a value was first issued on June 5th and never updated,
asking for data issued on June 6th (using \code{issues} or \code{lag}) would \emph{not}
return that value, though asking for data \code{as_of} June 6th would. See
\code{vignette("covidcast")} for examples.

Note also that the API enforces a maximum result row limit; results beyond
the maximum limit are truncated. This limit is sufficient to fetch
observations in all counties in the United States on one day. This client
automatically splits queries for multiple days across multiple API calls.
However, if data for one day has been issued many times, using the \code{issues}
argument may return more results than the query limit. A warning will be
issued in this case. To see all results, split your query across multiple
calls with different \code{issues} arguments.
}

\section{API keys}{


By default, \code{covidcast_signal()} submits queries to the API anonymously. All
the examples in the package documentation are compatible with anonymous use
of the API, but \href{https://cmu-delphi.github.io/delphi-epidata/api/api_keys.html}{there are some limits on anonymous queries},
including a rate limit. If you regularly query large amounts of data, please
consider \href{https://api.delphi.cmu.edu/epidata/admin/registration_form}{registering for a free API key}, which lifts
these limits. Even if your usage falls within the anonymous usage limits,
registration helps us understand who and how others are using the Delphi
Epidata API, which may in turn inform future research, data partnerships, and
funding.

If you have an API key, you can use it by setting the \code{covidcast.auth}
option once before calling \code{covidcast_signal()} or \code{covidcast_signals()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{options(covidcast.auth = "your_api_key")

cli <- covidcast_signal(data_source = "fb-survey", signal = "smoothed_cli",
                        start_day = "2020-05-01", end_day = "2020-05-07",
                        geo_type = "state")
}\if{html}{\out{</div>}}
}

\examples{
\dontrun{
## Fetch all counties from 2020-05-10 to the most recent available data
covidcast_signal("fb-survey", "smoothed_cli", start_day = "2020-05-10")
## Fetch all counties on just 2020-05-10 and no other days
covidcast_signal("fb-survey", "smoothed_cli", start_day = "2020-05-10",
                 end_day = "2020-05-10")
## Fetch all states on 2020-05-10, 2020-05-11, 2020-05-12
covidcast_signal("fb-survey", "smoothed_cli", start_day = "2020-05-10",
                 end_day = "2020-05-12", geo_type = "state")
## Fetch all available data for just Pennsylvania and New Jersey
covidcast_signal("fb-survey", "smoothed_cli", geo_type = "state",
                 geo_values = c("pa", "nj"))
## Fetch all available data in the Pittsburgh metropolitan area
covidcast_signal("fb-survey", "smoothed_cli", geo_type = "msa",
                 geo_values = name_to_cbsa("Pittsburgh"))
}

}
\references{
COVIDcast API documentation:
\url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast.html}

Documentation of all COVIDcast sources and signals:
\url{https://cmu-delphi.github.io/delphi-epidata/api/covidcast_signals.html}

COVIDcast public dashboard: \url{https://delphi.cmu.edu/covidcast/}
}
\seealso{
\code{\link[=plot.covidcast_signal]{plot.covidcast_signal()}}, \code{\link[=covidcast_signals]{covidcast_signals()}},
\code{\link[=as.covidcast_signal]{as.covidcast_signal()}}, \code{\link{county_census}}, \code{\link{msa_census}},
\code{\link{state_census}}
}
