% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pedigree_loglikelihood.R
\name{pedigree_loglikelihood}
\alias{pedigree_loglikelihood}
\title{Calculate the log-likelihoods of pedigrees}
\usage{
pedigree_loglikelihood(
  dat,
  geno_freq,
  trans,
  penet,
  sum_loglik = TRUE,
  ncores = 1
)
}
\arguments{
\item{dat}{A data frame with rows corresponding to people and columns
corresponding to the following variables (other variables can be included
but will be ignored), which will be coerced to \code{character} type:
\itemize{
\item \code{family} (optional), an identifier for each person's family, constant
within families.  If this variable is not supplied then \code{dat} will be
treated as a single pedigree.
\item \code{indiv}, an identifier for each individual person, with no duplicates
across the dataset.
\item \code{mother}, the individual identifier of each person's mother, or missing
(\code{NA}) for founders.
\item \code{father}, the individual identifier of each person's father, or missing
(\code{NA}) for founders.
}}

\item{geno_freq}{A vector of strictly positive numbers that sum to \code{1}.
The possible genotypes of the underlying genetic model are
\code{1:length(geno_freq)}, and \code{geno_freq[j]} is interpreted as the population
frequency of genotype \code{j}, so \code{geno_freq} is essentially the
function \code{Prior} of (Lange, 2002). For certain genetic models that often
occur in applications, these genotype frequencies can be calculated by
\code{\link{geno_freq_HWE}}.}

\item{trans}{An \code{ngeno^2} by \code{ngeno} matrix of non-negative numbers whose rows
all sum to \code{1}, where \code{ngeno = length(geno_freq)} is the number of possible
genotypes. The rows of \code{trans} correspond to joint parental genotypes and
the columns to offspring genotypes.  The element
\code{trans[ngeno * gm + gf - ngeno, go]} is interpreted as the conditional
probability that a person has genotype \code{go}, given that his or her
biological mother and father have genotypes \code{gm} and \code{gf}, respectively.
So \code{trans} is essentially the transmission function \code{Tran} of (Lange, 2002).
For certain genetic models that often occur in applications,
this transmission matrix can be calculated by \code{\link{trans_monogenic}}.}

\item{penet}{An \code{nrow(dat)} by \code{length(geno_freq)} matrix of non-negative
numbers. The element \code{penet[i,j]} is interpreted as the conditional
probability of the phenotypes of the person corresponding to row \code{i} of
\code{dat}, given that his or her genotype is \code{j}.  So \code{penet} is essentially the
penetrance function \code{Pen} of (Lange, 2002).
Note that genotype data can be incorporated into \code{penet} by regarding
known genotypes as phenotypes, i.e. by regarding known genotypes as
(possibly noisy) measurements of the underlying true genotypes.
If any row of \code{penet} consists entirely of zeroes then the likelihood is \code{0},
so the returned log-likelihood is \code{-Inf}.}

\item{sum_loglik}{A logical flag.  Return a named vector giving the
log-likelihood of each family if \code{sum_loglik} is \code{FALSE}, or return the sum
of these log-likelihoods if \code{sum_loglik} is \code{TRUE} (the default).}

\item{ncores}{The number of cores to be used, with \code{ncores = 1} (the
default) corresponding to non-parallel computing.  When \code{ncores > 1},
the \code{parallel} package is used to parallelize the calculation by dividing
the pedigrees among the different cores.}
}
\value{
Either a named vector giving the log-likelihood of each family
or the sum of these log-likelihoods, depending on \code{sum_loglik} (see above).
}
\description{
For one or more pedigrees, this function calculates the natural logarithm of
the pedigree likelihood on page 117 of (Lange, 2002), given inputs that
correspond to the terms in this formula.
}
\details{
This function provides a fast and general implementation of the
Elston-Stewart algorithm to calculate the log-likelihoods of large and
complex pedigrees without loops.  General references for the Elston-Stewart
algorithm are (Elston & Stewart, 1971), (Lange & Elston, 1975) and
(Cannings et al., 1978).

Each family within \code{dat} should be a complete pedigree, meaning that each
non-missing mother or father ID should correspond to a row, and each person
should either have both parental IDs missing (if a founder) or non-missing
(if a non-founder).  No family should contain pedigree loops, such as those
caused by inbreeding or by two sisters having children with two brothers
from a different family; see (Totir et al., 2009) for a precise definition.
The code is not currently suitable for families containing identical twins,
since these are not distinguished from full siblings.

In \code{geno_freq}, \code{trans} and \code{penet}, the order of the possible genotypes
must match (so that the same genotype corresponds to element \code{j} of \code{geno_freq}
and column \code{j} of \code{trans} and \code{penet}, for each \code{j} in \code{1:length(geno_freq)}).
}
\examples{
# Load pedigree files and penetrance matrices
data("dat_small", "penet_small", "dat_large", "penet_large")

# Settings for a biallelic locus in Hardy-Weinberg equilibrium and with a
# minor allele frequency of 10\%
geno_freq <- geno_freq_HWE(c(0.9, 0.1))
trans <- trans_monogenic(2)

# Calculate the log-likelihoods for 10 families, each with approximately 100 family members
pedigree_loglikelihood(
  dat_small, geno_freq, trans, penet_small, sum_loglik = FALSE, ncores = 2
)

# Calculate the log-likelihood for one family with approximately 10,000 family members
# Note:  this calculation takes approximately one minute on a standard desktop computer
# Note:  parallelization achieves nothing here because there is only one family
str(dat_large)
\donttest{
system.time(
  ll <- pedigree_loglikelihood(dat_large, geno_freq, trans, penet_large)
)
ll
}

}
\references{
Cannings C, Thompson E, Skolnick M. Probability functions
on complex pedigrees. Advances in Applied Probability, 1978;10(1):26-61.

Elston RC, Stewart J. A general model for the genetic analysis of pedigree
data. Hum Hered. 1971;21(6):523-542.

Lange K.  Mathematical and Statistical Methods for Genetic Analysis
(second edition). Springer, New York. 2002.

Lange K, Elston RC. Extensions to pedigree analysis I. Likehood calculations
for simple and complex pedigrees. Hum Hered. 1975;25(2):95-105.

Totir LR, Fernando RL, Abraham J. An efficient algorithm to compute marginal
posterior genotype probabilities for every member of a pedigree with loops.
Genet Sel Evol. 2009;41(1):52.
}
