% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cir.R
\name{quickIsotone}
\alias{quickIsotone}
\title{One-Stop-shop Forward point and interval estimation via CIR or IR}
\usage{
quickIsotone(
  y,
  x = NULL,
  wt = NULL,
  outx = NULL,
  dec = FALSE,
  estfun = cirPAVA,
  intfun = morrisCI,
  conf = 0.9,
  adaptiveShrink = FALSE,
  parabola = FALSE,
  ...
)
}
\arguments{
\item{y}{can be either of the following: y values (response rates), a 2-column matrix with positive/negative response counts by dose, a \code{\link{DRtrace}} object or a \code{\link{doseResponse}} object.}

\item{x}{dose levels (if not included in y). Note that the PAV algorithm doesn't really use them.}

\item{wt}{weights (if not included in y).}

\item{outx}{vector of x values for which predictions will be made. If \code{NULL} (default), this will be set to the set of unique values in the \code{x} argument (or equivalently in \code{y$x}).}

\item{dec}{logical, is the true function assumed to be monotone decreasing rather than increasing? Default \code{FALSE}.}

\item{estfun}{the function to be used for point estimation. Default \code{\link{cirPAVA}}.}

\item{intfun}{the function to be used for interval estimation. Default \code{\link{wilsonCI}} (see help on that function for additional options).}

\item{conf}{numeric, the interval's confidence level as a fraction in (0,1). Default 0.9.}

\item{adaptiveShrink}{logical, should the y-values be pre-shrunk towards an experiment's target? Recommended if data were obtained via an adaptive dose-finding design. If \code{TRUE}, then must also provide a \code{target} argument that will be passed via \code{...}.}

\item{parabola}{logical, should interpolated interval boundaries between observations be parabolically curved outwards to allow for more uncertainty? Default \code{FALSE}}

\item{...}{arguments passed on to other functions (constructor, point estimate and interval estimate).}
}
\value{
A data frame with 4 variables:  
\itemize{
\item {\code{x}} {either the input x values, or \code{outx} of specified;}
\item {\code{y}} {  The point estimates of x}
\item {\code{lowerPPconf,upperPPconf}  }  { the interval-boundary estimates for a 'PP'=\code{100*conf} confidence interval}
}
}
\description{
One-Stop-shop Forward point and confidence-interval estimation of a monotone response (y) as a function of dose (x), using centered-isotonic-regression (CIR, default) or isotonic regression. Input format is rather flexible.
This function calls \code{\link{cirPAVA}}, \code{\link{oldPAVA}}, \code{\link{iterCIR}} (speculatively), or a user-written function, for the point estimate, then \code{\link{isotInterval}} for the confidence interval. Vector input is allowed, but the preferred input format is a \code{\link{doseResponse}} object.
An analogous function for dose-finding (inverse estimation) is \code{\link{quickInverse}}.
}
\note{
You can obtain interpolated point estimates for x values between the observed data by specifying them via \code{outx}. However, for CIR, do NOT commit the error of generating estimates at observations, then interpolating using \code{\link{approx}}. If you need to retain a set of estimates for plotting the entire fitted curve, or for future interpolation at unknown points, call \code{\link{cirPAVA}} directly with \code{full=TRUE}, then use the returned \code{shrinkage} data frame for plotting and interpolation. See example code below.
}
\examples{
# Interesting run (#664) from a simulated up-and-down ensemble:
# (x will be auto-generated as dose levels 1:5)
dat=doseResponse(y=c(1/7,1/8,1/2,1/4,4/17),wt=c(7,24,20,12,17))
# CIR, using the default 'quick' function that also provides CIs (default 90\%).
quick1=quickIsotone(dat)
quick1
# Use 'estfun' argument to operate the same function with old PAVA as the estimator
quick0=quickIsotone(dat,estfun=oldPAVA)
quick0

### Showing the data and the fits
par(mar=c(3,3,1,1),mgp=c(2,.5,0),tcl=-0.25)
plot(dat,ylim=c(0.05,0.55),refsize=1,las=1) # uses plot.doseResponse()
# The IR fit: a straightforward interpolation
lines(quick0$y,lty=2) 

# With CIR, 'quickIsotone' cannot show us the true underlying interpolation; 
# it only provides the estimates at requested points.  Interpolation should be done between 
# shrinkage points, not the original design points. So we must call the full 'cirPAVA' function:

slow1=cirPAVA(dat,full=TRUE)
# Now, compare these 3 (the first one is wrong, b/c it interpolates from design points):
midpts=1:4+0.5
approx(1:5,quick1$y,xout=midpts)$y
quickIsotone(dat,outx=midpts) # instead, you can just call 'quickIsotone' and specify 'outx'
approx(slow1$shrinkage$x,slow1$shrinkage$y,xout=midpts)$y # Or use 'cirPAVA'

# Ok... finally plotting the CIR curve
# Both flat intervals are shrunk, because neither are at y=0 or y=1
lines(slow1$shrinkage$x,slow1$shrinkage$y)

# Last but not least, here's the true response function
lines(seq(1,5,0.1),pweibull(seq(1,5,0.1),shape=1.1615,scale=8.4839),col=2)
legend('topleft',pch=c(NA,'X',NA,NA),lty=c(1,NA,2,1),col=c(2,1,1,1),
	legend=c('True Curve','Observations','IR','CIR'),bty='n')
}
\seealso{
\code{\link{cirPAVA}},\code{\link{oldPAVA}},\code{\link{isotInterval}},\code{\link{quickInverse}},\code{\link{doseResponse}}
}
\author{
Assaf P. Oron \code{<aoron.at.idmod.org>}
}
