% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analIntervals.r
\name{isotInterval}
\alias{isotInterval}
\title{Returns analytical interval estimates, given isotonic-regression (centered or not) point estimates}
\usage{
isotInterval(
  isotPoint,
  outx = isotPoint$x,
  conf = 0.9,
  intfun = morrisCI,
  parabola = FALSE,
  ...
)
}
\arguments{
\item{isotPoint}{a \code{\link{doseResponse}} object with the x values and isotonic point estimates at design points.}

\item{outx}{vector of x values for which estimates will be made. If \code{NULL} (default), this will be set to the set of unique values in isotPoint$x argument (or equivalently in y$x).}

\item{conf}{numeric, the interval's confidence level as a fraction in (0,1). Default 0.9.}

\item{intfun}{the function to be used for interval estimation. Default \code{\link{morrisCI}} (see help on that function for additional options).}

\item{parabola}{logical: should the confidence-interval's interpolation between points with observations follow a parabola (\code{TRUE}) creating broader intervals between observations, or a straight line (\code{FALSE}, default)?}

\item{...}{additional arguments passed on to \code{intfun}}
}
\value{
a data frame with two variables \code{ciLow, ciHigh} containing the estimated lower and upper confidence bounds, respectively.
}
\description{
For confidence intervals at design points ($x$ values with obesrvations), this function calls \code{intfun} to do the work. In addition, CIs for any $x$ value are calculated using linear interpolation between design points (note that for CIR, this differs from the interpolation of point estimates which is carried out between shrinkage points, as explained in \code{\link{quickIsotone}})
}
\note{
All provided algorithm and formulae are for Binomial data only. For other data, write your own \code{intfun}, returning a two-column matrix. The interval estimation method is presented and discussed by Oron and Flournoy (2017).
}
\examples{
# Interesting run (#664) from a simulated up-and-down ensemble:
# (x will be auto-generated as dose levels 1:5)
dat=doseResponse(y=c(1/7,1/8,1/2,1/4,4/17),wt=c(7,24,20,12,17))
# The experiment's goal is to find the 30th percentile
slow1=cirPAVA(dat,full=TRUE)
# Default interval (Morris+Wilson); same as you get by directly calling 'quickIsotone'
int1=isotInterval(slow1$output)
# Morris without Wilson; the 'narrower=FALSE' argument is passed on to 'morrisCI'
int1_0=isotInterval(slow1$output,narrower=FALSE)
# Wilson without Morris
int2=isotInterval(slow1$output,intfun=wilsonCI)
# Agresti=Coull (the often-used "plus 2")
int3=isotInterval(slow1$output,intfun=agcouCI)
# Jeffrys (Bayesian-inspired) is also available
int4=isotInterval(slow1$output,intfun=jeffCI)

### Showing the data and the intervals
par(mar=c(3,3,4,1),mgp=c(2,.5,0),tcl=-0.25)
plot(dat,ylim=c(0,0.65),refsize=4,las=1,main="Forward-Estimation CIs") # uses plot.doseResponse()

# The true response function; true target is where it crosses the y=0.3 line
lines(seq(0,7,0.1),pweibull(seq(0,7,0.1),shape=1.1615,scale=8.4839),col=4)

lines(int1$ciLow,lty=2,col=2,lwd=2) 
lines(int1$ciHigh,lty=2,col=2,lwd=2) 

lines(int1_0$ciLow,lty=2) 
lines(int1_0$ciHigh,lty=2) 

lines(int2$ciLow,lty=2,col=3) 
lines(int2$ciHigh,lty=2,col=3) 
# Plotting the remaining 2 is skipped, as they are very similar to Wilson.

# Note how the default (red) boundaries take the tighter of the two options everywhere, 
# except for one place (dose 1 upper bound) where they go even tighter thanks to monotonicity 
# enforcement. This can often happen when sample size is uneven; since bounds tend to be 
# conservative it is rather safe to do.

legend('topleft',pch=c(NA,'X',NA,NA,NA),lty=c(1,NA,2,2,2),col=c(4,1,2,1,3),lwd=c(1,1,2,1,1),legend
=c('True Curve','Observations','Morris+Wilson (default)','Morris only','Wilson only'),bty='n')

}
\references{
Oron, A.P. and Flournoy, N., 2017. Centered Isotonic Regression: Point and Interval Estimation for Dose-Response Studies. Statistics in Biopharmaceutical Research, In Press (author's public version available on arxiv.org).
}
\seealso{
\code{\link{quickIsotone}},\code{\link{quickInverse}},\code{\link{morrisCI}},
}
\author{
Assaf P. Oron \code{<aoron.at.idmod.org>}
}
