% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cfr_static.R
\name{cfr_static}
\alias{cfr_static}
\title{Estimate a static disease severity measure}
\usage{
cfr_static(data, delay_density = NULL, poisson_threshold = 100)
}
\arguments{
\item{data}{A \verb{<data.frame>} containing the outbreak data. A daily time
series with dates or some other absolute indicator of time (e.g. epiday or
epiweek) and the numbers of new cases and new deaths at each time point.
Note that the required columns are "date" (for the date), "cases" (for the
number of reported cases), and "deaths" (for the number of reported deaths)
on each day of the outbreak.

Note that the \verb{<data.frame>} is required to have an unbroken sequence of
dates with no missing dates in between. The "date" column must be of class
\code{Date} (see \code{\link[=as.Date]{as.Date()}}).

Note also that the total number of cases must be greater than the total
number of reported deaths.}

\item{delay_density}{An optional argument that controls whether delay
correction is applied in the severity estimation.
May be \code{NULL}, for no delay correction, or a function that returns the
density function of a distribution to evaluate
density at user-specified values, e.g.
\code{function(x) stats::dgamma(x = x, shape = 5, scale = 1)}.}

\item{poisson_threshold}{The case count above which to use Poisson
approximation. Set to 100 by default. Must be > 0.}
}
\value{
A \verb{<data.frame>} with the maximum likelihood estimate and 95\%
confidence interval of the severity estimates, named "severity_mean",
"severity_low", and "severity_high".
}
\description{
Calculates the severity of a disease, while optionally
correcting for reporting delays using an epidemiological delay distribution
of the time between symptom onset and death (onset-to-death).

Other delay distributions may be passed to calculate different disease
severity measures such as the hospitalisation fatality risk.
}
\section{Details: Adjusting for delays between two time series}{
The method used in \code{cfr_static()} follows Nishiura et al.
(2009).
The function calculates a quantity \eqn{u_t} for each day within the input
data, which represents the proportion of cases estimated to have
a known outcome on day \eqn{t}.
Following Nishiura et al., \eqn{u_t} is calculated as:
\deqn{u_t = \dfrac{\sum_{i = 0}^t
        \sum_{j = 0}^\infty c_i f_{j - i}}{\sum_{i = 0} c_i}}
where \eqn{f_t} is the value of the probability mass function at time \eqn{t}
and \eqn{c_t}, \eqn{d_t} are the number of new cases and new deaths at time
\eqn{t}, (respectively).
We then use \eqn{u_t} at the end of the outbreak in the following likelihood
function to estimate the severity of the disease in question.
\deqn{{\sf L}({\theta \mid y}) = \log{\dbinom{u_tC}{D}} + D \log{\theta} +
  (u_tC - D)\log{(1.0 - \theta)}}
\eqn{C} and \eqn{D} are the cumulative number of cases and deaths
(respectively) up until time \eqn{t}.
\eqn{\theta} is the parameter we wish to estimate, the severity of the
disease. We estimate \eqn{\theta} using simple maximum-likelihood methods,
allowing the functions within this package to be quick and easy tools to use.

The precise severity measure — CFR, IFR, HFR, etc — that \eqn{\theta}
represents depends upon the input data given by the user.

The epidemiological delay-distribution density function passed to
\code{delay_density} is used to evaluate the probability mass function
parameterised by time; i.e. \eqn{f(t)} which
gives the probability that a case has a known outcome (usually death) at time
\eqn{t}, parameterised with disease-specific parameters before it is supplied
here.
}

\examples{
# load package data
data("ebola1976")

# estimate severity without correcting for delays
cfr_static(ebola1976)

# estimate severity for each day while correcting for delays
# obtain onset-to-death delay distribution parameters from Barry et al. 2018
# The Lancet. <https://doi.org/10.1016/S0140-6736(18)31387-4>
cfr_static(
  ebola1976,
  delay_density = function(x) dgamma(x, shape = 2.40, scale = 3.33)
)

}
\references{
Nishiura, H., Klinkenberg, D., Roberts, M., & Heesterbeek, J. A. P. (2009).
Early Epidemiological Assessment of the Virulence of Emerging Infectious
Diseases: A Case Study of an Influenza Pandemic. PLOS ONE, 4(8), e6852.
\doi{10.1371/journal.pone.0006852}
}
