#' Install a cffr GitHub Action
#'
#' @description
#'
#' This function would install a GitHub Action on your repo. The action
#' will update your `CITATION.cff` when any of these events occur:
#' - You publish a new release of the package.
#' - Your DESCRIPTION or inst/CITATION are modified.
#' - The action can be run also manually.
#'
#' @param path Project directory
#' @param overwrite If already present, do you want to overwrite your action?
#'
#' @return Invisible, this function is called by its side effects.
#'
#' @details
#'
#' Triggers on your action can be modified, see
#' [Events that trigger workflows](https://docs.github.com/en/actions/learn-github-actions/events-that-trigger-workflows).
#'
#' @examples
#' \dontrun{
#' cff_gha_update()
#' }
#' @export
cff_gha_update <- function(path = ".",
                           overwrite = FALSE) {
  destdir <- file.path(path, ".github", "workflows")
  checkdir <- dir.exists(destdir)

  if (isFALSE(checkdir)) {
    message(crayon::blue("Creating directory '", destdir, "'"))
    dir.create(destdir, recursive = TRUE, showWarnings = FALSE)
  }

  newfile <- file.path(destdir, "update-citation-cff.yaml")

  if (!file.exists(newfile) | isTRUE(overwrite)) {
    message(crayon::blue("Installing update-citation-cff.yaml"))
    file.copy(system.file("yaml/update-citation-cff.yaml",
      package = "cffr"
    ),
    newfile,
    overwrite = TRUE
    )
  } else {
    message(crayon::blue(
      "File update-citation-cff.yaml already installed.",
      "Use overwrite = TRUE for overwrite"
    ))
  }

  if (file.exists(file.path(path, ".Rbuildignore"))) {
    ignore <- readLines(file.path(path, ".Rbuildignore"))

    # If not already
    if (!("^\\.github$" %in% ignore)) {
      ignore <- c(ignore, "^\\.github$")
      ignore <- unique(ignore)

      message(crayon::blue(
        "Adding .github",
        "to .Rbuildignore"
      ))

      writeLines(ignore, ".Rbuildignore")
    }
  }

  return(invisible())
}
