% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derivative.R
\name{derivative}
\alias{derivative}
\title{Numerical and Symbolic Derivatives}
\usage{
derivative(
  f,
  var,
  params = list(),
  order = 1,
  accuracy = 4,
  stepsize = NULL,
  drop = TRUE,
  deparse = TRUE
)
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See details.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{order}{integer vector, giving the differentiation order for each variable. See details.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}

\item{drop}{if \code{TRUE}, return the array of derivatives without adding a dummy dimension when \code{order} is of length 1.}

\item{deparse}{if \code{TRUE}, return \code{character} instead of \code{expression}.}
}
\value{
\code{array}.
}
\description{
Computes symbolic derivatives based on the \code{\link[stats]{D}} function, or numerical derivatives based on finite differences.
}
\details{
The function behaves differently depending on the arguents \code{order}, 
the order of differentiation, and \code{var}, the variable names with respect to 
which the derivatives are computed.

When multiple variables are provided and \code{order} is a single integer \eqn{n}, 
then the \eqn{n}-th order derivative is computed for each element of \code{f} 
with respect to each variable:

\deqn{D = \partial^{(n)} \otimes F}

that is:

\deqn{D_{i,\dots,j,k} = \partial^{(n)}_{k} F_{i,\dots,j}}

where \eqn{F} is the array of functions and \eqn{\partial_k^{(n)}} denotes the 
\eqn{n}-th order partial derivative with respect to the \eqn{k}-th variable.
    
When \code{order} matches the length of \code{var}, it is assumed that the 
differentiation order is provided for each variable. In this case, each element 
is derived \eqn{n_k} times with respect to the \eqn{k}-th variable, for each 
of the \eqn{m} variables.
    
\deqn{D_{i,\dots,j} = \partial^{(n_1)}_1\cdots\partial^{(n_m)}_m F_{i,\dots,j}}
    
The same applies when \code{order} is a named vector giving the differentiation 
order for each variable. For example, \code{order = c(x=1, y=2)} differentiates 
once with respect to \eqn{x} and twice with respect to \eqn{y}. A call with 
\code{order = c(x=1, y=0)} is equivalent to \code{order = c(x=1)}. 
    
To compute numerical derivatives or to evaluate symbolic derivatives at a point, 
the function accepts a named vector for the argument \code{var}; e.g. 
\code{var = c(x=1, y=2)} evaluates the derivatives in \eqn{x=1} and \eqn{y=2}. 
For \code{functions} where the first argument is used as a parameter vector, 
\code{var} should be a \code{numeric} vector indicating the point at which the 
derivatives are to be calculated.
}
\examples{
### symbolic derivatives
derivative(f = "sin(x)", var = "x")

### numerical derivatives
f <- function(x) sin(x)
derivative(f = f, var = c(x=0))

### higher order derivatives
f <- function(x) sin(x)
derivative(f = f, var = c(x=0), order = 3)

### multivariate functions 
##  - derive once with respect to x
##  - derive twice with respect to y
##  - evaluate in x=0 and y=0
f <- function(x, y) y^2*sin(x)
derivative(f = f, var = c(x=0, y=0), order = c(1,2))

### vector-valued functions
##  - derive each element twice with respect to each variable
##  - evaluate in x=0 and y=0
f <- function(x, y) c(x^2, y^2)
derivative(f, var = c(x=0, y=0), order = 2)

### vectorized interface
f <- function(x) c(sum(x), prod(x))
derivative(f, var = c(0,0,0), order = 1)

}
\references{
Guidotti, E. (2020). "calculus: High dimensional numerical and symbolic calculus in R". \url{https://arxiv.org/abs/2101.00086}
}
\seealso{
Other derivatives: 
\code{\link{taylor}()}

Other differential operators: 
\code{\link{curl}()},
\code{\link{divergence}()},
\code{\link{gradient}()},
\code{\link{hessian}()},
\code{\link{jacobian}()},
\code{\link{laplacian}()}
}
\concept{derivatives}
\concept{differential operators}
