% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sea.R
\name{sea}
\alias{sea}
\title{Perform superposed epoch analysis}
\usage{
sea(x, event, nbefore = 6, nafter = 4, event_range = TRUE,
  n_iter = 1000)
}
\arguments{
\item{x}{A data frame climate reconstruction or tree-ring series with row
names as years, and one numeric variable.}

\item{event}{An numeric vector of event years for superposed epoch, such as
fire years, or an \code{fhx} object with a single series as produced by
\code{\link[=composite]{composite()}}.}

\item{nbefore}{The number of lag years prior to the event year.}

\item{nafter}{The number of lag years following the event year.}

\item{event_range}{Logical. Constrain the time series to the time period of
key events within the range of the \code{x} series. \code{FALSE} uses the entire
series, ignoring the period of key events.}

\item{n_iter}{The number of iterations for bootstrap resampling.}
}
\value{
A \code{sea} object containing. This contains:
\itemize{
\item "event_years": a numeric vector of event years.
\item "actual": a \code{data.frame} summary of the actual events.
\item "random": a \code{data.frame} summary of the bootstrapped events.
\item "departure": a \code{data.frame} summary of the departures of actual from
bootstrapped events.
\item "simulated": a full 2D \code{matrix} of the bootstrapped-values across lags.
\item "observed": a ful 2D \code{matrix} of "actual" events across lags.
}
}
\description{
Perform superposed epoch analysis
}
\details{
Superposed epoch analysis (SEA) helps to evaluate fire-climate
relationships in studies of tree-ring fire history. It works by compositing
the values of an annual time series or climate reconstruction for the fire
years provided (\code{event}) and both positive and negative lag years.
Bootstrap resampling of the time series is performed to evaluate the
statistical significance of each year's mean value. Users interpret the
departure of the actual event year means from the simulated event year means.
Note that there is no rescaling of the climate time series \code{x}.

The significance of lag-year departures from the average climate condition
was first noted by Baisan and Swetnam (1990) and used in an organized SEA by
Swetnam (1993). Since then, the procedure has been commonly applied in fire
history studies. The FORTRAN program EVENT.exe was written by Richard Holmes
and Thomas Swetnam (Holmes and Swetnam 1994) to perform SEA for fire history
specifically. EVENT was incorporated in the FHX2 software by Henri
Grissino-Mayer. Further information about SEA can be found in the FHAES
user's manual, http://help.fhaes.org/.

\code{\link[=sea]{sea()}} was originally designed to replicate EVENT as closely as possible. We
have tried to stay true to their implementation of SEA, although multiple
versions of the analysis exist in the climate literature and for fire
history. The outcome of EVENT and sea should only differ slightly in the
values of the simulated events and the departures, because random draws are
used. The event year and lag significance levels should match, at least in
the general pattern.

Our SEA implementation borrowed from \code{dplR::sea()} function in how it
performs the bootstrap procedure, but differs in the kind of output provided
for the user.
}
\examples{
\dontrun{
# Read in the Cook and Krusic (2004; The North American Drought Atlas)
# reconstruction of Palmer Drought Severity Index (PDSI) for the Jemez
# Mountains area (gridpoint 133).
target_url <- paste0(
  "http://iridl.ldeo.columbia.edu",
  "/SOURCES/.LDEO/.TRL/.NADA2004",
  "/pdsiatlashtml/pdsiwebdata/1050w_350n_133.txt"
)
pdsi <- read.table(target_url, header = TRUE, row.names = 1)
pdsi <- subset(pdsi, select = "RECON")

# Run SEA on Peggy Mesa (pgm) data
data(pgm)
pgm_comp <- composite(pgm)

pgm_sea <- sea(pdsi, pgm_comp)

# See basic results:
print(pgm_sea)

# Basic plot:
plot(pgm_sea)
}

}
\references{
Baisan and Swetnam 1990, Fire history on desert mountain range:
Rincon Mountain Wilderness, Arizona, U.S.A. Canadian Journal of Forest
Research 20:1559-1569.

Bunn 2008, A dendrochronology program library in R (dplR),
Dendrochronologia 26:115-124

Holmes and Swetnam 1994, EVENT program description

Swetnam 1993, Fire history and climate change in giant sequoia
groves, Science 262:885-889.
}
\seealso{
\itemize{
\item \code{\link[=plot_sealags]{plot_sealags()}} plots \code{sea} lags and their statistical significance.
\item \code{\link[=print.sea]{print.sea()}} prints a pretty summary of \code{sea} objects.
\item \code{\link[=composite]{composite()}} creates fire composites, a common input to \code{\link[=sea]{sea()}}.
}
}
