% Generated by roxygen2 (4.0.0): do not edit by hand
\name{boost}
\alias{boost}
\alias{boost.function}
\alias{boost.list}
\title{Boost an Estimation Procedure with a Reweighter and an Aggregator.}
\usage{
boost(x, B, reweighter, aggregator, data, .procArgs = NULL, metadata = NULL,
  initialWeights = rep.int(1, nrow(data))/nrow(data),
  analyzePerformance = defaultOOBPerformanceAnalysis,
  .boostBackendArgs = NULL)

\method{boost}{list}(x, B, reweighter, aggregator, data, .procArgs = NULL,
  metadata = NULL, initialWeights = rep.int(1, nrow(data))/nrow(data),
  analyzePerformance = defaultOOBPerformanceAnalysis,
  .boostBackendArgs = NULL)

\method{boost}{function}(x, B, reweighter, aggregator, data, .procArgs = NULL,
  metadata = NULL, initialWeights = rep.int(1, nrow(data))/nrow(data),
  analyzePerformance = defaultOOBPerformanceAnalysis,
  .boostBackendArgs = NULL)
}
\arguments{
\item{B}{number of iterations of boost to perform.}

\item{x}{a list with entries '\code{train}' and '\code{predict}' or a
function that satisfies the definition of an estimation procedure given
below. The list input will invoke a call to
\code{\link{buildEstimationProcedure}}. Function input will invoke a call to
\code{\link{wrapProcedure}}, unless the function inherits from
'\code{estimationProcedure}'. In either event, metadata may be required to
properly wrap \code{x}. See the appropriate help documentation.}

\item{reweighter}{A reweighter, as defined below. If the function does not
inherit from '\code{reweighter}', a call to \code{\link{wrapReweighter}}
will be made. See \code{\link{wrapReweighter}} to determine what metadata,
if any, you may need to pass for the wrapper to be \code{boostr} compatible}

\item{aggregator}{An aggregator, as defined below. If the function does not
inherit from '\code{aggregator}' a call to \code{\link{wrapAggregator}}
will be made to build a boostr compatible wrapper. See
\code{\link{wrapAggregator}} to determine if any  metadata needs to be
passed in for this to be successful.}

\item{data}{a data.frame of matrix to act as the learning set. The columns
are assumed to be ordered such that the response variable in the first
column and the remaining columns as the predictors. As a convenience,
\code{\link{boostBackend}} comes with a switch, \code{.formatData}
(defaulted to \code{TRUE}) which will look for  an argument named
\code{formula} inside \code{.procArgs} and use the value of
\code{formula} to format \code{data}. If you don't want this to happen,
or if the data is already properly formatted, include
\code{.formatData=FALSE} in \code{metadata}.}

\item{.procArgs}{a named list of arguments to pass to the estimation
procedure.
If \code{x} is a list, \code{.procArgs} is a named list of lists with
entries \code{.trainArgs} and \code{.predictArgs} and each list is a
named list of arguments to pass to \code{x$train} and \code{x$predict},
respectively. If \code{x} is a function, \code{.procArgs} is a named list
of arguments to pass to \code{x}, in addition to \code{data} and
\code{weights}. See 'Examples' below.}

\item{initialWeights}{a vector of weights used for the first iteration of
the ensemble building phase of Boost.}

\item{analyzePerformance}{a function which accepts an estimator's
predictions and the true responses to said predictions (among other
arguments) and returns a list of values. If no function is provided,
\code{\link{defaultOOBPerformanceAnalysis}} is used.
 See \code{\link{wrapPerformanceAnalyzer}} for metadata that may
need to be passed to make \code{analyzePerformance} compatible with the
boostr framework.}

\item{metadata}{a named list of arguments to be passed to
\code{\link{wrapProcedure}}, \code{\link{buildEstimationProcedure}},
 \code{\link{wrapReweighter}}, \code{\link{wrapAggregator}}, and/or
 \code{\link{wrapPerformanceAnalyzer}}.}

\item{.boostBackendArgs}{a named list of additional arguments to pass to
\code{\link{boostBackend}}.}
}
\value{
a '\code{boostr}' object which is returned from \code{\link{boostBackend}}.
 This object is a function of a single input
 \item{newdata}{a data.frame or matrix whose columns should probably be in
  the same order as the columns of the data each of the constituent
  estimators was trained on.}
 The return value of this function is a prediction for each row in
 \code{newdata}.

 See \code{\link{boostBackend}} for more details on "\code{boostr}" objects.
}
\description{
Boost an \emph{estimation procedure} and analyze individual
estimator performance using a \emph{reweighter}, \emph{aggregator}, and
some \emph{performance analyzer}.
}
\details{
This function is a designed to be an interface between the user
and \code{\link{boostBackend}} when \code{x}, \code{reweighter},
\code{aggregator} and/or \code{analyzePerformance} are valid input to
the Boost algorithm, but do not have \code{boostr} compatible signatures.
Hence, \code{boost} calls the appropriate wrapper function (with the
relevant information from \code{metadata}) to convert user supplied
functions into \code{boostr} compatible functions.
}
\examples{
### Demonstrate simple call with just list(train=svm)

library(foreach)
library(iterators)
library(e1071)

svmArgs <- list(formula=Species~., cost=100)
boost(x=list(train=svm),
      reweighter=arcfsReweighter,
      aggregator=arcfsAggregator,
      data=iris,
      .procArgs=list(.trainArgs=svmArgs),
      B=2)

### Demonstrate call with train and predict and custom 
### reweighters and aggregators

df <- within(iris, {
  Setosa <- as.factor(2*as.numeric(Species == "setosa")-1)
  Species <- NULL
})

# custom predict function
newPred <- function(obj, new) {
  predict(obj, new)
}

predMetadata <- c(modelName="obj", predictionSet="new")

# custom reweighter
testReweighterMetadata <- list(
                            reweighterInputWts="w",
                            reweighterInputResponse="truth",
                            reweighterInputPreds="preds",
                            reweighterOutputWts="w")

testReweighter <- function(preds, truth, w) {
  
  wrongPreds <- (preds != truth)
  err <- mean(wrongPreds)
  if (err != 0) {
    new_w <- w / err^(!wrongPreds)
  } else {
    new_w <- runif(n=length(w), min=0, max=1)
  }
  
  
  list(w=new_w, alpha=rnorm(1))
}

# custom aggregator
testAggregatorMetadata <- c(.inputEnsemble="ensemble")

testAggregator <- function(ensemble) {
  weights <- runif(min=0, max=1, n=length(ensemble))
  function(x) {
    preds <- foreach(estimator = iter(ensemble),
                     .combine = rbind) \%do\% {
                       matrix(as.character(estimator(x)), nrow=1)
                     }
    
    as.factor(predictClassFromWeightedVote(preds, weights))
  }
}

# collect all the relevant metadata
metadata <- c(predMetadata, testReweighterMetadata, testAggregatorMetadata)

# set additional procedure arguments
procArgs <- list(
              .trainArgs=list(
                formula=Setosa ~ .,
                cost=100)
              )

#test boost when irrelevant metadata is passed in.
boostedSVM <- boost(list(train=svm, predict=newPred),
                    B=3,
                    reweighter=testReweighter,
                    aggregator=testAggregator,
                    data=df,
                    metadata=metadata,
                    .procArgs=procArgs,
                    .boostBackendArgs=list(
                      .reweighterArgs=list(fakeStuff=77))
                    )

### Demonstrate customizing 'metadata' for estimation procedure
library(class)

testkNNProcMetadata <- list(learningSet="traindata", predictionSet="testdata")

testkNNProc <- function(formula, traindata, k) {  
  df <- model.frame(formula=formula, data=traindata)
  function(testdata, prob=FALSE) {
    df2 <- tryCatch(model.frame(formula=formula, data=testdata)[, -1],
                    error = function(e) testdata 
    )
    knn(train=df[, -1], test=df2, cl=df[, 1], prob=prob, k=k) 
  }
}

testKNNProcArgs <- list(formula=Setosa ~ ., k = 5)

metadata <- testkNNProcMetadata
boostBackendArgs <- list(.reweighterArgs=list(m=0))

boostedKNN <- boost(x=testkNNProc, B=3,
      reweighter=arcx4Reweighter,
      aggregator=arcx4Aggregator,
      data=df, 
      metadata=metadata,
      .boostBackendArgs=boostBackendArgs,
      .procArgs=testKNNProcArgs)

### Demonstrate using an alternative performance analyzer

testPerfAnalyzer2 <- function(pred, truth, oob, zeta) {
  list(e=mean(pred != truth), z=zeta)
}

testPerfAnalyzer2Metadata <- list(analyzerInputPreds="pred",
                                  analyzerInputResponse="truth",
                                  analyzerInputOObObs="oob")

metadata <- c(metadata, testPerfAnalyzer2Metadata)

boostedkNN <- boost(testkNNProc,
                    B=3,
                    reweighter=vanillaBagger,
                    aggregator=vanillaAggregator,
                    data=df,
                    .procArgs=testKNNProcArgs,
                    metadata=metadata,
                    .boostBackendArgs = list(
                      .analyzePerformanceArgs = list(zeta="77"),
                      .reweighterArgs=list(fakeStuff=77)),
                    analyzePerformance=testPerfAnalyzer2)

}
\seealso{
Other aggregators: \code{\link{adaboostAggregator}};
  \code{\link{arcfsAggregator}};
  \code{\link{arcx4Aggregator}},
  \code{\link{vanillaAggregator}},
  \code{\link{weightedAggregator}}

Other performance analyzers: \code{\link{defaultOOBPerformanceAnalysis}}

Other reweighters: \code{\link{adaboostReweighter}};
  \code{\link{arcfsReweighter}};
  \code{\link{arcx4Reweighter}};
  \code{\link{vanillaBagger}}
}

