#'@title Anytime-valid sequential estimation of the p-value of a Monte-Carlo
#'  test
#'
#'@description `avseqmc()` performs anytime-valid sequential estimation of the
#'  p-value of a Monte-Carlo test as described in Stoepker and Castro (2024,
#'  Definition 6). Subsequent references to equations and sections in this
#'  section of the reference manual pertain to this paper. The sequential
#'  p-value estimate is based on the construction of Definition 6 (i.e. through
#'  the confidence sequence by Robbins (1970)).
#'
#'  For first-time usage, it may be helpful to follow the examples in the
#'  package vignette.
#'
#'@param sample_G Either: a function (without arguments) that draws one (or a
#'  batch of) zero/one samples from the distribution G as in Equation (5), where
#'  the function returns a vector of zeroes and ones; or an object from class
#'  'avseqmc_progress' containing earlier progress from anytime-valid estimation
#'  of the p-value. The function `avseqmc()` returns such an object, or the
#'  object can be constructed manually using function
#'  `init_avseqmc_progress()`).
#'@param epsilon The desired risk of overestimated significance. Ignored if
#'  sample_G is an object of class `avseqmc_progress` and mandatory otherwise.
#'@param stopcrit The desired stopping criterion. Can use one of the two
#'  pre-defined stopping criteria from Section 4.1 as follows (with respect to
#'  the notation used in that section):
#' - `list("type" = "futility", "param" = alpha)`
#' - `list("type" = "convergence", "param" = c(gamma, n_0))`
#'  Alternatively, a custom function may be provided that takes an
#'  `avseqmc_progress` object as input and returns `FALSE` if sampling should
#'  continue for another batch.
#'@param min_samples Minimum number of Monte-Carlo samples before returning the
#'  current p-value estimate. Defaults to 0.
#'@param max_samples Maximum number of Monte-Carlo samples before returning the
#'  current p-value estimate. Defaults to `max(1000, min_samples)`.
#'@param compute_lower Boolean; if `TRUE`, the lower confidence sequence limit
#'  with significance level epsilon is computed after each batch of Monte-Carlo
#'  samples, based on the construction by Robbins (1970). Since it is used in
#'  the evaluation of the futility stopping criterion (i.e. `stopcrit =
#'  list("type"="futility","param"=...)`) it is automatically computed when this
#'  stopping criterion is selected.
#'
#'@return An object of class `avseqmc_progress` containing the progress of the
#'  sequentially estimated p-value. The object is a list containing the
#'  following elements:
#'
#'  - `$epsilon`: risk of overestimated significance used in the sequential estimation.
#'  - `$sample_G`: function that samples (batches) from the Monte-Carlo distribution $G^*(X)$ as in Equation (5).
#'  - `$ptilde`: sequence of sequential $p$-value estimates. The final value in this sequence is the most recent estimate of the $p$-value.
#'  - `$Ltilde`: sequence of lower bounds of the confidence sequence based on the construction by Robbins (1970). Contains NA values if these were
#'  not computed by default through `stopcrit =
#'  list("type"="futility","param"=...)` or requested using
#'  `compute_lower=TRUE`.
#'  - `$n`: total number of samples drawn from the MC sampler.
#'  - `$S`: total number of ones observed from the MC sampler.
#'  - `$B`: sequence of number of ones observed at each sampling timepoint (which can be greater than 1 if `sample_G` samples in batches)
#'  - `$Bn`: sequence of number of samples drawn from MC sampler at each timepoint (which can be greater than 1 if `sample_G` samples in batches)
#'
#' @examples
#' # Minimal example using defaults:
#' set.seed(123)
#' library(avseqmc)
#' G1 <- function(){runif(1) < 0.01} # A mock MC function to demonstrate functionality
#' R1 <- avseqmc(sample_G = G1, epsilon = 0.001)
#' print(R1)
#'
#' # Minimal example to resuming earlier estimation:
#' G2 <- function(){runif(1) < 0.03}
#' R2a <- avseqmc(sample_G = G2, epsilon = 0.001)
#' print(R2a)
#' R2b <- avseqmc(R2a)
#' print(R2b)
#'
#' # Using built-in convergence stopping time:
#' G3 <- function(){runif(1) < 0.04}
#' R3 <- avseqmc(sample_G = G3,
#'               epsilon = 0.001,
#'               stopcrit = list("type" = "convergence", param = c(1e-5, 100)))
#'
#' # Batch sampling example (drawing batches of size 50)
#' G4 <- function(){runif(50) < 0.04}
#' R4 <- avseqmc(sample_G = G4, epsilon = 0.001)
#' print(R4)
#'
#'@seealso \code{\link{init_avseqmc_progress}} which can be used if one wishes
#'  to resume progress based on earlier reported p-values estimated by
#'  Monte-Carlo simulation.
#'
#'@references Stoepker, I. V., and R. M. Castro. 2024. Inference with Sequential
#'  Monte-Carlo Computation of p-Values: Fast and Valid Approaches.
#'  https://doi.org/10.48550/arXiv.2409.18908.
#'
#'  Robbins, H. (1970). Statistical Methods Related to the Law of the Iterated
#'  Logarithm. The Annals of Mathematical Statistics, 41(5):1397–1409.
#'  http://dx.doi.org/10.1214/aoms/1177696786
#'@export
avseqmc <- function(sample_G, epsilon = NULL, stopcrit = list("type" = "futility", "param" = 0.05), min_samples = 0, max_samples = max(1000,min_samples), compute_lower = FALSE) {

  ##############################################################################

  # Check if sample_G and epsilon arguments are specified correctly and
  # (if needed) initializes an avseqmc_progress object.
  if (is.function(sample_G)) {
    # Check if sample_G is a function that takes no arguments.
    if (length(formals(sample_G)) != 0) {
      stop("If sample_G is a function, it must not take any arguments.")
    }
    # Check if epsilon is specified correctly.
    if (is.null(epsilon) || !is.numeric(epsilon) || length(epsilon) != 1 || epsilon <= 0 || epsilon >= 1) {
      stop("If sample_G is a function, epsilon must be specified as a single numeric value strictly between 0 and 1.")
    }
    R <- init_avseqmc_progress(sample_G, epsilon)
  } else if (inherits(sample_G, "avseqmc_progress")) {
    R <- sample_G  # The value of sample_G is an avseqmc_progress object which we will subsequently use.
    # Warn user that specified epsilon is not used if a avseqnmc.progress object
    # is provided (which includes a value of epsilon)
    if (!is.null(epsilon)) {
      warning("epsilon is ignored when sample_G is of class 'avseqmc_progress'.")
    }
  } else {
    stop("sample_G must be either a function with no arguments, or an object of class 'avseqmc_progress'.")
  }
  # Now, R is an object of class avseqmc_progress which we will subsequently use.

  # Check if min_samples is input correctly
  if(!missing(min_samples)){
    if (!is.numeric(min_samples) || length(min_samples) != 1 || min_samples < 0 || min_samples != floor(min_samples) || min_samples > max_samples) {
      stop("min_samples must be a single integer greater or equal to 0 and smaller than max_samples.")
    }
  }

  # Check if max_samples is input correctly
  if(!missing(max_samples)){
    if (!is.numeric(max_samples) || length(max_samples) != 1 || max_samples < 1 || max_samples != floor(max_samples)) {
      stop("max_samples must be a single integer greater or equal to 1.")
    }
  }

  # Check if compute_lower is input correctly
  if( !is.logical(compute_lower) & length(compute_lower) > 1 & !is.na(compute_lower)){
    stop("compute_lower should be either TRUE or FALSE.")
  }

  ##############################################################################

  starting_iterations <- R$n

  # Check what stopping criterion is specified, and keep sampling until
  # criterion is reached

  # Check if custom stopping criterion is chosen (in the form of a function
  # with a single parameter)
  if (is.function(stopcrit)) {
    if (length(formals(stopcrit)) != 1){
      stop("Custom stopping criterion must be specified through a function taking a single argument with class avseqmc_progress.")
    } else {
      if(compute_lower == FALSE){
        message("A custom stopping criterion is used with compute_lower=FALSE. Note that the lower bound of the confidence sequence R$Ltilde is only available for the evaluation of the stopping criterion if requested via compute_lower=TRUE.")
      }
      while ( (!stopcrit(R) && R$n - starting_iterations < max_samples) || R$n - starting_iterations < min_samples) {
        R <- nextsample(R, compute_lower = compute_lower)
      }
    }

  # Check if built-in stopping time is chosen (through supplying formatted list)
  }else if (is.list(stopcrit)) {
    if (stopcrit$type == "futility") {

      # Check if parameters are correctly specified
      if (!is.numeric(stopcrit$param) || length(stopcrit$param) != 1 || stopcrit$param <= 0 || stopcrit$param >= 1) {
        stop("For the 'futility' stopping criterion, a single parameter (corresponding to the significance level against which futility is evaluated) should be between 0 and 1.")
      }

      # Check if the avseqmc_progress object does not contain any samples yet,
      # since then we must first draw a sample before evaluating stopping
      # criterion
      if (length(R$ptilde) == 0) {
        R <- nextsample(R, compute_lower = TRUE)
      }

      # Check if the avseqmc_progress object already contains the most recent
      # value for the lower bound, since we require it to evaluate the stopping
      # criterion
      if (is.na(R$Ltilde[length(R$Ltilde)])) {
        # Compute Ltilde for the latest value
        R$Ltilde[length(R$Ltilde)] <- max(c(R$Ltilde, Ln_tilde(R$S, R$n, R$epsilon)), na.rm = TRUE)
      }

      # Keep sampling until futility stopping criterion is reached
      while ( (R$ptilde[length(R$ptilde)] > stopcrit$param &&
             R$Ltilde[length(R$Ltilde)] < stopcrit$param &&
             R$n - starting_iterations < max_samples) || R$n - starting_iterations < min_samples) {
        R <- nextsample(R, compute_lower = TRUE)
      }

    } else if (stopcrit$type == "convergence") {

      # Check if parameters are correctly specified
      if (!is.numeric(stopcrit$param) || length(stopcrit$param) != 2 ||
          stopcrit$param[1] <= 0 || stopcrit$param[2] < 1 || stopcrit$param[2] != floor(stopcrit$param[2]) || stopcrit$param[2] >= max_samples) {
        stop("For the 'convergence' stopping criterion, two parameters should be specified. The first parameter corresponds to the desired rate gamma and should be positive, and the second parameter corresponds to the window size n_0 over which the rate is computed, and should be a positive integer smaller than the maximum number of samples.")
      }

      # Check if the avseqmc_progress object contains sufficient samples to be
      # able to evaluate the stopping criterion.
      while ( (R$n <= stopcrit$param[2] && R$n - starting_iterations < max_samples) || R$n - starting_iterations < min_samples) {
        R <- nextsample(R, compute_lower = compute_lower)
      }

      # Keep sampling until convergence stopping criterion is reached Note:
      # which(cumsum(rev(R$Bn)) >= stopcrit$param[2])[1] + 1 finds the index in
      # R$Bn such that the number of underlying MC samples is at least
      # stopcrit$param[2]
      while (
          (
            (R$ptilde[length(R$Bn) - which(cumsum(rev(R$Bn)) >= stopcrit$param[2])[1] + 1] - R$ptilde[length(R$ptilde)]) / stopcrit$param[2] > stopcrit$param[1]
            && R$n - starting_iterations < max_samples
          ) || R$n - starting_iterations < min_samples
        ) {
        R <- nextsample(R, compute_lower = compute_lower)
      }

    } else {
      stop("Invalid stopping criterion specified.")
    }
  } else {
    stop("Invalid stopping criterion specified.")
  }

  return(R)
}
