\name{single_cover_value}
\alias{single_cover_value}
\title{Vegetation cover by height or growth form using TERN AusPlots data.
}
\description{Calculates a single vegetation cover value per site based on the plot-based point-intercept data from AusPlots (as generated by \code{\link{get_ausplots}}). Cover can be subsetted to vegetation that has a specified minimum or maximum height and/or by plant growth forms, allowing user defined definitions of e.g. grass or tree cover. For example, by default, forest cover is calculated per plot for tree growth forms of 5 metres or higher.
}
\usage{
single_cover_value(veg.PI, in_canopy_sky=FALSE, 
by.growth_form=TRUE, min.height=5, max.height=NULL, 
my.growth_forms=c("Tree/Palm", "Tree Mallee"))
}
\arguments{
  \item{veg.PI}{The input raw point intercept data ($veg.PI) as generated in the \code{\link{get_ausplots}} function.
}
  \item{in_canopy_sky}{Logical, whether to include in canopy sky hits to calculate 'opaque canopy cover' instead of 'projected foliage cover'. Set to \code{TRUE} to calculate opaque canopy cover (e.g., suitable for some applications such as comparison to remotely sensed cover products).
}
  \item{by.growth_form}{Logical, whether to subset by plant growth form. If set to FALSE then only height subsetting occurs. Set to TRUE to calculate cover subsetted to specified growth forms.
}
\item{min.height}{Numeric, sets the height in m below which PI hits are ignored. This argument is used to calculate e.g. cover of trees 5 m high or higher. Can be set to zero to ignore height and include any plant hit. If you set min.height to a negative number you will get a nonsensical output!
}
\item{max.height}{Numeric, sets the height in m above which PI hits are ignored. This argument can be used to calculate ground cover of desired growth forms, for example. 
}
\item{my.growth_forms}{A user-supplied character vector specifying growth forms to include in the cover calculation. Defaults to trees. Applies only when \code{by.growth_form=TRUE}, otherwise this argument is ignored and only the height rule is applied. For allowable growth forms, see \code{unique(veg.PI$growth_form)}.
}
}
\details{All included point intercept hits are merged to give a single percent cover value with a maximum of 100. Cover partitioned by multiple plant growth forms can be calculated using \code{growth_form_table}, by species using \code{species_table} and fractional (green, brown or bare) cover using \code{fractional_cover}. Note, when min.height is set to zero and by.growth_form is set to FALSE, the output is nearly the same as the green cover fraction returned from \code{fractional_cover}. The values can differ, however, because \code{fractional_cover} applies a height rule in which the highest intercept at a given point is taken, whereas \code{single_cover_value} finds any green cover, leading to differences, most obviously, for example, when dead trees overhang green understorey. For such general cover purposes, we recommend \code{fractional_cover}. \code{single_cover_value} is best suited to cover subset by height and growth form. 
}
\value{Returns a data.frame in which plots are rows identified by a unique site and visit identifier ('site_unique') and the returned cover value is given in the percentCover column.
}
\author{Greg Guerin
}

\seealso{
\code{\link{get_ausplots}}
\cr
\cr
\code{\link{fractional_cover}}
}
\examples{
\dontrun{
my.dat <- get_ausplots(bounding_box=c(140, 142, -20, -18), veg.vouchers=FALSE, site_info=FALSE)

#Default settings, where cover of trees 5 m or higher is calculated:
tree.cover <- single_cover_value(my.dat$veg.PI)
tree.cover

#Tussock grasses of any height (minimum height set to zero):
grass.cover <- single_cover_value(my.dat$veg.PI, 
my.growth_forms=c("Tussock grass"), min.height=0) 
grass.cover

#Any green veg at least 2 m in height:
veg.height.eg <- single_cover_value(my.dat$veg.PI, by.growth_form=FALSE, 
min.height=2)
veg.height.eg

#Any green veg up to 1 m high:
ground.cover <- single_cover_value(my.dat$veg.PI, by.growth_form=FALSE, 
min.height=0, max.height=1)
ground.cover
}
}
