% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multigons.R
\name{multigons}
\alias{multigons}
\title{Draws several polygons}
\usage{
multigons(
  i,
  x,
  y,
  j = unique(i),
  forget = NULL,
  front = NULL,
  back = NULL,
  density = 0,
  angle = 45,
  border = "black",
  col = NA,
  lty = par("lty"),
  lwd = par("lwd"),
  scol = border,
  slty = lty,
  slwd = lwd,
  lend = 0,
  ljoin = 0,
  lmitre = 10
)
}
\arguments{
\item{i}{a polygon id for each x and y coordinate, i.e. the name of
each polygon. If you want to give each polygon a different aspect you should
provide a vector of n elements (if you have three polygons "A1", "A2" and
"A3" with "A2" that should be blue you should provide the colours of all
three: e.g. \code{col = c("white", "blue", "white")})}

\item{x, y}{numeric vectors of x and y coordinates}

\item{j}{a list of the ids (names) in the order used for the
graphical parameters (e.g. colour, shading, etc...). By default they are in
their order of appearance in \code{i}}

\item{forget}{the polygons that should not be drawn, by their id or
index (i.e. name or number of appearance).}

\item{front, back}{the polygons to be put in front and back position,
by their id or index (i.e. name or number of appearance). By default the
order is the one defined by \code{j}, and if \code{j} is absent by the order
in \code{i}.}

\item{density}{the density of shading lines, in lines per inch. The
default value of 0 means that no shading lines are drawn.}

\item{angle}{the slope of shading lines, given as an angle in degrees
(counter-clockwise).}

\item{border}{the colour to draw the border. The default is black. Use
border = NA to omit borders.}

\item{col}{the colour for filling the polygon. The default, NA, is to
leave polygons unfilled.}

\item{lty, lwd}{the border line type and width, see ?par for details.}

\item{scol, slty, slwd}{the colour, type and width of the shading lines.}

\item{lend, ljoin, lmitre}{additional graphical parameters, see ?par
for details.}
}
\description{
Draws several polygons. This function expands on the polygon()
function from base R graphics. The difference is that several polygons can
be drawn in one line by providing a polygon id: i. To each polygon you can
provide different graphical parameters (i.e. colour, shading, etc). On the
contrary of the polygon() function the graphical parameters of the shading
lines can be independent of the border lines.
}
\details{
In the case you want shading this function will draw three
overlapping polygons: one for the background, one for the shading lines and
one for the border. \code{multigons} shares similarities with
\code{\link{centresvg}} and \code{\link{framesvg}}, but allows more advanced
control of each element.
}
\examples{
# Simple use:

i <- c(rep("A1",6), rep("A2",6), rep("A3",6)) # Polygon ids
x <- c(1,2,3,3,2,1,2,3,4,4,3,2,3,4,5,5,4,3)   # x coordinates
y <- c(1,2,3,4,5,6,1,2,3,4,5,6,1,2,3,4,5,6)   # y coordinates

plot(c(-1,7), c(-2,9), type = "n", xlab = "", ylab = "", main = "Simple use")

multigons(i, x, y,
          front = "A2", # This gets the polygon A2 in front of all others
          density = c(NA, 5, 10),  # Different shading density
          scol = "darkred", # Same shading colour
          col = c("black", "grey", "white"), # Different background colour
          lwd = 2, # Width of border lines for all polygons
          slty = 2, # Shading lines type, same for all polygons
          slwd = 1) # Shading lines width, same for all polygons

# Advanced use:
# Lets first create more polygons

i2 <- c(i, rep("A4",6), rep("A5",6), rep("A6",6))
x2 <- rep(x,2)
y2 <- c(y, y - 4)

# Then lets attribute a group to each of them: lets say blue and red polygons

groups <- data.frame(j = c("A1", "A2", "A3", "A4", "A5","A6"),
                      group = c("blue", "red", "blue", "red", "red", "blue"),
                      stringsAsFactors = FALSE)

# Then lets attribute different graphical parameters for each group

legend <- data.frame(group = c("red", "blue"),
                     col = c("red", "blue"),
                     density = c(10,20),
                     scol = c("darkred", "darkblue"),
                      stringsAsFactors = FALSE)

 # Now that you have a data frame saying which polygon is in which group,
 # and one providing distinct graphical parameters for each group, you can
 # join the two with help of the dplyr package:

 library(dplyr)

 parameters <- left_join(groups, legend, by = "group")

 # Then simply apply them to multigons:

 plot(c(0,12), c(-3,7), type = "n", xlab = "", ylab = "",
      main = "Advanced use")

 multigons(i2,x2,y2,
           forget = c("A1"),     # If you want to avoid drawing one polygon
           front = c("A2","A3"), # Puts A2 in front and A3 right behind
           col = parameters$col,
           density = parameters$density,
           scol = parameters$scol,
           lwd = 2)

# Another way (more advanced, but with interesting programming applications)
# to code this:

all.parameters <- merge_list(list(i = i2, x = x2 + 6, y = y2),
                             as.list(parameters),
                             list(lwd = 3, slwd = 2, slty = 2))

all.parameters <- all.parameters[!names(all.parameters) == "group"]

do.call(multigons, all.parameters)

}
\seealso{
Similar functions: \code{\link{multilines}}, \code{\link{infobar}}

Complementary function: \code{\link{shift}}

Uses \code{\link{ignore}} to avoid drawing unnecessary objects
}
