#' Orthogonalized spillover index
#' 
#' @description
#' Computes the orthogonalized spillover index proposed in Diebold and Yilmaz (2009) 
#' which is based on the Orthogonalized Forecast Error Variance Decompositon. 
#' 
#' @param x Object of class \sQuote{\code{varest}} generated by \code{VAR()} from vars package.
#' @param n.ahead Integer specifying the steps ahead.
#' @param ortho.type A character string indicating the type of orthogonalized index is required. \code{"single"} takes the original ordering of variables in VAR model and applies Cholesky decomposition for the fevd. Whereas \code{"partial"} takes a random sample out of all the possible combinations generated for the Cholesky decomposition, while \code{"total"} uses all the combinations, therefore it takes more time to finish.
#' Both, \code{"partial"} and \code{"total"} provide average results.
#' @param standardized A logical value indicating whether the values should be divided by the number of columns
#' to get a percentage. 
#'  
#' @details
#' This function computes the Orthogonalized Directional Spillover Table which has as its
#' \eqn{ij^{th}} entry the estimated contribution \emph{to} the forecast error variance of
#' variable \emph{i} coming \emph{from} innovations to variable \emph{j}. The off-diagonal 
#' column sums are the \emph{Contributions to Others}, while the row sums represent 
#' \emph{Contributions from Others}, when these are totaled across  countries then we have 
#' the numerator of the Spillover Index. Similarly, the columns sums or rows sums (including 
#' diagonal), when totaled across countries, give the denominator of the Spillover Index, which 
#' is 100\%.
#' 
#' \code{O.spillover} is based upon the Orthogonalized (using Cholesky orthogonalization) Forecast
#' Error Variance Decompositon (see Lutkepohl, 2006) and its explicit formulation can be found 
#' in Diebold and Yilmaz (2009).
#' 
#' Since \code{O.spillover} is based on orthogonalized FEVD, then the result is as many indeces
#' as combinations is allowed according to the number of variables in the VAR model, this is 
#' exactly equal to \eqn{K!}{K!}, then \code{output} has three options: \code{table}, \code{summary}
#' and \code{all.ind}. \code{table} produces a \code{data.frame} holding the (orthogonalized) 
#' directional  mean spillover indices.
#' 
#' When \code{output="table"}, a \code{data.frame} is generated consisting of either mean or 
#' median directional spillover indeces, this because for each possible order of the variables 
#' the \code{o.fevd} is computed and over this result a spillover index is generated and this
#' procedure repeats until reaching the last order (this means all the possible combinations
#' given by \eqn{K!}). When \code{output="table"} a mean directional spillover table is generated,
#' but this can be changed using \code{stat="median"} for a median directional spillover to be
#' genereated. Note that \code{stat} argument only affects the results of \code{output="table"}.
#' 
#' When \code{output="summary"} an vector is generated,
#' this contains  \code{Mean, Min, Max}.
#' 
#' This is a user-frendly version of \code{fastSOM::sot_avg_exact()} function.
#' 
#' @return
#' When \code{output="table"}, a \code{data.frame} consisting of the spillover index. 
#' 
#' When \code{output="summary"}, a \code{summary} of all spillover indeces.
#' 
#' 
#' @references
#' Diebold, F. X. & Yilmaz, K. (2009). \emph{Measuring Financial Asset Return and Volatility Spillovers, 
#' with Application to Global Equity Markets}. The Economic Journal, 119, 158-171
#' 
#' Lutkepohl, H. (2006), \emph{New Introduction to Multiple Time Series Analysis}, Springer, New York.  
#' 
#' @author Jilber Urbina
#' @seealso  \code{\link{G.spillover}}
#' @export
#' @import vars 
#' @examples 
#' library(vars)
#' data(stock.prices)
#' stocks <- stock.prices[,1:2]
#' VAR.1 <- VAR(stocks)
#' O.spillover(VAR.1, n.ahead=5)
#' O.spillover(VAR.1, n.ahead=5, ortho.type  = "partial")
#' O.spillover(VAR.1, n.ahead=5, ortho.type  = "total")
#' 
#' # Replicating Table 3, Diebold and Yilmaz (2009)
#' \donttest{
#' data(dy2009)
#' VAR.2 <- VAR(dy2009[,-1], p=2)
#' O.spillover(VAR.2, ortho.type  = "single", standardized = FALSE) 
#' O.spillover(VAR.2, ortho.type  = "partial" ) 
#' }
#' 
#' @keywords 
#' o.fevd
#' forecast error variance decomposition
#' Orthogonal spillover index


O.spillover <- function (x, n.ahead = 10,
                         ortho.type = c("single", "partial", "total"), 
                         standardized=TRUE) 
{
  if (!(class(x) == "varest")) {
    stop("\nPlease provide an object of class 'varest', generated by 'VAR()'.\n")
  }
  
  n.ahead <- abs(as.integer(n.ahead))
  K <- x$K
  index <- function(x) {(sum(x)-sum(diag(x)))/nrow(x)}
  #------------------------------------------------------
  table <-  switch(match.arg(ortho.type), 
                            single = {t(sapply(vars::fevd(x, n.ahead = n.ahead), function(x) x[n.ahead, ])) * 100},
                            partial = {fastSOM::sot_avg_exact(summary(x)$covres , Phi(x, n.ahead - 1))$Average}, 
                            total = {fastSOM::sot_avg_est(summary(x)$covres , Phi(x, n.ahead - 1))$Average})
  
  
  CFO <- rowSums(table-diag(diag(table)))   # C. from Others
  CTO <- colSums(table)-diag(table)         # C. to Others (spillover)
  CTOT <- colSums(table)                    # C. to others including own
  table <- rbind(table, "C. to others (spillover)"= CTO, "C. to others including own" = CTOT)
  if(standardized){
    table <- cbind(table, `C. from others` =  c(CFO, sum(CFO), sum(CTOT) )) / K
  } else{
    table <- cbind(table, `C. from others` = c(CFO, index(table[1:K, ]), sum(CTOT)))
  }
  
  return(table)
}

