## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2017  Stefan Engblom
## Copyright (C) 2015 - 2017  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

##' Class \code{"SimInf_mparse"}
##'
##' Class to handle the SimInf mparse data
##' @section Slots:
##' \describe{
##'   \item{C_code}{
##'     Character vector with model C code.
##'   }
##'   \item{G}{
##'     Dependency graph that indicates the transition rates that need
##'     to be updated after a given state transition has occured.
##'     A non-zero entry in element \code{G[i, i]} indicates that transition
##'     rate \code{i} needs to be recalculated if the state transition
##'     \code{j} occurs. Sparse matrix (\eqn{Nt \times Nt}) of object class
##'     \code{"\linkS4class{dgCMatrix}"}.
##'   }
##'   \item{S}{
##'     Each column corresponds to a state transition, and execution
##'     of state transition \code{j} amounts to adding the \code{S[,
##'     j]} column to the state vector \code{u[, i]} of node \emph{i}
##'     where the transition occurred. Sparse matrix (\eqn{Nc \times
##'     Nt}) of object class \code{"\linkS4class{dgCMatrix}"}.
##'   }
##' }
##' @keywords methods
##' @export
##' @import Matrix
setClass("SimInf_mparse",
         slots = c(C_code = "character",
                   G      = "dgCMatrix",
                   S      = "dgCMatrix"),
         validity = function(object) {
             errors <- character()

             ## Check C code
             if (nchar(paste0(object@C_code, collapse = "\n")) == 0L) {
                 errors <- c(errors, "'C_code' is empty.")
             }

             ## Check S.
             if (identical(dim(object@S), c(0L, 0L))) {
                 errors <- c(errors, "'S' is empty.")
             } else if (!all(is_wholenumber(object@S@x))) {
                 errors <- c(errors,
                             "'S' matrix must be an integer matrix.")
             }

             ## Check G.
             Nt <- dim(object@S)[2]
             if (!identical(dim(object@G), c(Nt, Nt))) {
                 errors <- c(errors,
                             "Wrong size of dependency graph 'G'.")
             }

             if (length(errors) == 0) TRUE else errors
         }
)

## C code: heading
C_heading <- function() {
    c(sprintf("/* Generated by SimInf (v%s) %s */",
            packageVersion("SimInf"),
            format(Sys.time(), "%Y-%m-%d %H:%M")),
      "")
}

## C code: include directives
C_include <- function() {
    c("#include <R_ext/Rdynload.h>",
      "#include \"SimInf.h\"",
      "")
}

## C code: compartments
C_compartments <- function(compartments) {
    lines <- character(0)
    if (length(compartments)) {
        lines <- c("/* Compartments */",
                   paste0("enum{",
                          paste0(compartments, collapse = ", "),
                          "};"),
                   "")
    }
    lines
}

## C code: rate constants
C_rate_constants <- function(rates) {
    lines <- character(0)
    if (length(rates)) {
        lines <- "/* Rate constants */"
        for (i in seq_len(length(rates))) {
            lines <- c(lines, sprintf("const double %s = %g;",
                                      names(rates)[i],
                                      rates[[i]]))
        }
        lines <- c(lines, "")
    }
    lines
}

## C code: transition rate functions.
C_trFun <- function(transitions) {
    lines <- character(0)
    for (i in seq_len(length(transitions))) {
        lines <- c(lines,
                   sprintf("double trFun%i(", i),
                   "    const int *u,",
                   "    const double *v,",
                   "    const double *ldata,",
                   "    const double *gdata,",
                   "    double t)",
                   "{",
                   sprintf("    return %s;", transitions[[i]]$propensity),
                   "}",
                   "")
    }
    lines
}

## C code: post time step function.
C_ptsFun <- function() {
    c("int ptsFun(",
      "    double *v_new,",
      "    const int *u,",
      "    const double *v,",
      "    const double *ldata,",
      "    const double *gdata,",
      "    int node,",
      "    double t)",
      "{",
      "    return 0;",
      "}",
      "")
}

## C code: SimInf model run function.
C_run <- function(transitions) {
    c("SEXP SimInf_model_run(SEXP model, SEXP threads, SEXP seed)",
      "{",
      sprintf("    TRFun tr_fun[] = {%s};",
              paste0("&trFun", seq_len(length(transitions)), collapse = ", ")),
      "    DL_FUNC SimInf_run = R_GetCCallable(\"SimInf\", \"SimInf_run\");",
      "    return SimInf_run(model, threads, seed, tr_fun, &ptsFun);",
      "}",
      "")
}

## C code: Generate C code for mparse.
C_code_mparse <- function(transitions, rates, compartments) {
    c(C_heading(),
      C_include(),
      C_compartments(compartments),
      C_rate_constants(rates),
      C_trFun(transitions),
      C_ptsFun(),
      C_run(transitions))
}

## Rewrite propensity
##
## Rewrite the propensity by replacing all compartments by
## \code{u[compartments[j]]} where \code{j} is the numbering in
## compartments. On return, 'depends' contains all compartments upon
## which the propensity depends.
rewriteprop <- function(propensity, compartments) {
    ## Switch to an intermediate representation using '###', replacing
    ## larger strings first in order to avoid changing e.g., 'BA' with
    ## '###[1]A' whenever 'B' and 'BA' are two different compartments.
    depends <- integer(length(compartments))
    symbols <- order(nchar(compartments), decreasing = TRUE)
    for (i in symbols) {
        if (length(grep(compartments[i], propensity))) {
            depends[i] <- 1
            propensity <- gsub(compartments[i],
                               sprintf("###[%i]", i),
                               propensity)
        }
    }

    ## Final replace
    for (i in symbols) {
        propensity <- gsub(sprintf("###[%i]", i),
                           sprintf("u[%s]", compartments[i]),
                           propensity, fixed = TRUE)
    }

    list(propensity = propensity, depends = depends)
}

##' Model parser
##'
##' @param transitions character vector containing transitions on the
##'     form \code{"X -> ... -> Y"}. The left (right) side is the
##'     initial (final) state and the propensity is written in between
##'     the \code{->}-signs. The special symbol \code{@} is reserved
##'     for the empty set. For example, \code{transitions =
##'     c("S -> k1*S*I -> I", "I -> k2*I -> R")} expresses a SIR
##'     model.
##' @param compartments contains the names of the involved
##'     compartments, for example, \code{compartments = c("S", "I",
##'     "R")}.
##' @param ... rate-constants for the model.
##' @return \linkS4class{SimInf_mparse}
##' @export
##' @importFrom utils packageVersion
##' @examples
##' \dontrun{
##' ## Use the model parser to create a 'SimInf_mparse' object
##' m <- mparse(c("S -> k1*S*I/(S+I+R) -> I", "I -> k2*I -> R"),
##'             c("S", "I", "R"), k1 = 0.16, k2 = 0.077)
##'
##' ## Initialize a 'SimInf_model' from the 'SimInf_mparse' object
##' u0 <- data.frame(S = 100, I = 1, R = 0)
##' model <- init(m, u0 = u0, tspan = 1:100)
##'
##' ## Run and plot the result
##' plot(run(model))
##' }
mparse <- function(transitions = NULL, compartments = NULL, ...)
{
    rates <- list(...)
    stopifnot(all(sapply(rates, class) == "numeric"))

    if (is.null(transitions))
        stop("'transitions' must be specified.")
    if(!is.character(transitions))
        stop("'transitions' must be specified in a character vector.")
    if (is.null(compartments))
        stop("'compartments' must be specified.")
    if(!is.character(compartments))
        stop("'compartments' must be specified in a character vector.")
    if (!all(identical(length(compartments), length(unique(compartments))),
             identical(length(names(rates)), length(unique(names(rates))))))
        stop("'compartments' and 'rates' must consist of unique names.")

    reserved = c("v_new", "u", "v", "ldata", "gdata", "node", "t", "rng")
    if (length(intersect(compartments, reserved)))
        stop(paste("Invalid compartment names:",
                   paste0(intersect(compartments, reserved), collapse = ", ")))
    if (length(intersect(names(rates), reserved)))
        stop(paste("Invalid rate names:",
                   paste0(intersect(names(rates), reserved), collapse = ", ")))

    transitions <- lapply(strsplit(transitions, "->"), function(x) {
        if (!identical(length(x), 3L))
            stop(paste0("Invalid transition: '", paste0(x, collapse = "->"), "'"))

        ## Remove spaces and the empty set
        from <- gsub(" ", "", gsub("@", "", x[1]))
        propensity <- gsub(" ", "", x[2])
        dest <- gsub(" ", "", gsub("@", "", x[3]))

        ## Split from and dest into 'compartment1 + compartment2 + ..'
        from <- unlist(strsplit(from, "+", fixed = TRUE))
        dest <- unlist(strsplit(dest, "+", fixed = TRUE))

        ## Assign each compartment into its number according to the
        ## ordering in compartments
        ifrom <- match(from, compartments)
        if (any(is.na(ifrom)))
            stop(sprintf("Unknown compartment: '%s'.", from[is.na(ifrom)]))
        idest <- match(dest, compartments)
        if (any(is.na(idest)))
            stop(sprintf("Unknown compartment: '%s'.", dest[is.na(idest)]))

        ## The corresponding column in the state change matrix S is
        ## now known.
        S <- integer(length(compartments))
        S[ifrom] <- -1
        S[idest] <- 1

        propensity <- rewriteprop(propensity, compartments)

        list(from       = from,
             dest       = dest,
             propensity = propensity$propensity,
             depends    = propensity$depends,
             S          = S)
    })

    S <- as(do.call("cbind", lapply(transitions, function(x) x$S)), "dgCMatrix")
    depends <- do.call("rbind", lapply(transitions, function(x) x$depends))
    G <- as(((depends %*% abs(S)) > 0) * 1, "dgCMatrix")

    colnames(G) <- as.character(seq_len(dim(G)[2]))
    rownames(G) <- sapply(transitions, function(x) {
        paste(ifelse(length(x$from), x$from, "@"),
              "->",
              ifelse(length(x$dest), x$dest, "@"))
    })
    colnames(S) <- as.character(seq_len(dim(S)[2]))
    rownames(S) <- compartments

    new("SimInf_mparse",
        C_code = C_code_mparse(transitions, rates, compartments),
        G = G,
        S = S)
}

##' @rdname init-methods
##' @export
setMethod("init",
          signature(model = "SimInf_mparse"),
          function(model, u0, tspan)
          {
              compartments <- rownames(model@S)

              ## Check u0
              if (!is.data.frame(u0))
                  stop("'u0' must be a data.frame")
              if (!all(compartments %in% names(u0)))
                  stop("Missing columns in u0")
              u0 <- u0[, compartments, drop = FALSE]

              E <- as(matrix(integer(0), nrow = 0, ncol = 0), "dgCMatrix")

              N <- matrix(integer(0), nrow = 0, ncol = 0)

              v0 <- matrix(numeric(0), nrow  = 0, ncol = nrow(u0))
              storage.mode(v0) <- "double"

              ldata <- matrix(numeric(0), nrow = 0, ncol = nrow(u0))
              storage.mode(ldata) <- "double"

              SimInf_model(G      = model@G,
                           S      = model@S,
                           E      = E,
                           N      = N,
                           tspan  = tspan,
                           events = NULL,
                           ldata  = ldata,
                           gdata  = numeric(),
                           u0     = u0,
                           v0     = v0,
                           C_code = model@C_code)
          }
)
