#' Plots to visualize SMLE screening
#'
#' This function returns two plot windows. By default, the first contains 4 plots to assess convergence:
#' 1) log-likelihood,  
#' 2) Euclidean distance between the current and the previous coefficient estimates,   
#' 3) the number of tries in u-search (see details of \code{\link{SMLE}()}),  
#' and 4) the number of features changed in the current active set.
#' By default, the second plot shows the solution path (estimated coefficient by iteration step) for the retained features.
#'
#' @param x A \code{'smle'} object as the output from \code{\link{SMLE}()}.
#' 
#' @param num_path The number of top coefficients to be shown.
#' Default is equal to the number of features retained in the model.
#' @param which_path A vector to control which features are shown in addition to the paths for the most significant coefficients.
#' @param out_plot A number from 1 to 5 indicating which plot is to be shown in the separate window; the default for solution path plot is "5".
#' See Description for plot labels 1-4.
#' @param ... Additional arguments passed to the second plot.
#' @return
#' No return value.
#' @export
#' @importFrom graphics plot.new
#' @method plot smle
#' @examples
#' \donttest{
#' set.seed(1)
#' Data <- Gen_Data(correlation = "CS")
#' fit <- SMLE(Y = Data$Y,X = Data$X, k = 20, family = "gaussian")
#' plot(fit)
#'}

plot.smle<-function(x,num_path=NULL,which_path=NULL,out_plot=5,...){
  
  ## new plot
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar),add = TRUE)

  #-------------------check
  nsteps<-x$steps

  Feature_path<-x$path_retained
  
  if(!is.null(x$data)){
    
    X <- x$X
    x$ID_retained<- sort((1:length(colnames(X)))[colnames(X) %in% x$iteration_data$feature_name])
  
    }

  #--------------------plot-------------
  if(out_plot==5){

  par(mfrow=c(2,2))

  plot(y=x$likelihood_iter,x=1:nsteps,xlab="steps",ylab="log-likelihood",type="b")

  title("Likelihood convergence")

  plot(x$coef_dist, x=1:length(x$coef_dist),xlab="steps",ylab="L2-dist b/w beta updates",
       main=" Coefficient convergence",type="b" )

  plot(y=x$Usearch,x=1:nsteps,xlab="steps",ylab="No. of tries")

  title("U-search")

  plot(y=x$iteration_data$FD,x=1:nsteps,xlab="steps",ylab="No. of changes")

  title("Retained feature change")

  dev.new()

  if(is.null(num_path)){num_path=length(x$ID_retained)}

  TOP_index<- x$ID_retained[sort(abs(x$coef_retained),decreasing = T,index.return=T)$ix][1:num_path]

  TOP_index<-unique(c(TOP_index,which_path))

  Feature_path<-Feature_path[TOP_index,]

  plot(NULL, xlim=c(1,nsteps+1), ylim=c(floor(min(Feature_path[,nsteps])),ceiling(max(Feature_path[,nsteps]))),xlab="steps",ylab="coefficients",cex.lab = 1.5,cex.axis = 1.5 ,...)

  title("Solution path")

  lines(rep(0,nsteps+1),lty=1,lwd=1,col="black")
  
  A<-lapply(1:length(TOP_index),function(i){lines(Feature_path[i,],lty=i,col=rainbow(length(TOP_index))[i])})

  legend("topright",lty=1:length(TOP_index),cex=1,col=rainbow(length(TOP_index)),
           legend=TOP_index,bty="n")
    
  }else if(out_plot==1){

    par(mfrow=c(2,2))
    
    if(is.null(num_path)){num_path=length(x$ID_retained)}
    
    TOP_value<-rep(0,num_path)
    
    TOP_value<-sort(abs(Feature_path[,ncol(Feature_path)]),decreasing = T)[1:num_path]
    
    TOP_index<-(1:dim(x$X)[2])[Feature_path[,ncol(Feature_path)]%in% unique(c(TOP_value,-TOP_value))]
    
    TOP_index<-unique(c(TOP_index,which_path))
    
    Feature_path<-Feature_path[TOP_index,]
    
    plot(NULL, xlim=c(1,nsteps+1), ylim=c(floor(min(Feature_path[,nsteps])),ceiling(max(Feature_path[,nsteps]))),xlab="steps",ylab="coefficients",...)
    
    title("Solution path")
    
    lines(rep(0,nsteps),lty=1,lwd=1,col="black")
    
    A<-lapply(1:length(TOP_index),function(i){lines(Feature_path[i,],lty=i,col=rainbow(length(TOP_index))[i])})
    
    legend("topright",lty=1:length(TOP_index),cex=0.5,col=rainbow(length(TOP_index)),
           legend=TOP_index,bty="n")
    
    plot(x$coef_dist, x=1:length(x$coef_dist),xlab="steps",ylab="L2-dist b/w beta updates",
         main=" Coefficient convergence",type="b" )
    title(' Coefficient convergence')
    plot(y=x$Usearch,x=1:nsteps,xlab="steps",ylab="No. of tries")
    title("U-search")
    #
    plot(y=x$iteration_data$FD,x=1:nsteps,xlab="steps",ylab="No. of changes")
    title("Retained feature change")

  dev.new()
  plot(y=x$likelihood_iter,x=1:nsteps,xlab="steps",ylab="Soulution path",type="b",...)
  title("Likelihood convergence")
  }else if(out_plot==2){
  
    par(mfrow=c(2,2))
    
    if(is.null(num_path)){num_path= length(x$ID_retained)}
    
    TOP_value<-rep(0,num_path)
    
    TOP_value<-sort(abs(Feature_path[,ncol(Feature_path)]),decreasing = T)[1:num_path]
    
    TOP_index<-(1:dim(x$X)[2])[Feature_path[,ncol(Feature_path)]%in% unique(c(TOP_value,-TOP_value))]
    
    TOP_index<-unique(c(TOP_index,which_path))
    
    Feature_path<-Feature_path[TOP_index,]
    
    plot(NULL, xlim=c(1,nsteps+1), ylim=c(floor(min(Feature_path[,nsteps])),ceiling(max(Feature_path[,nsteps]))),xlab="steps",ylab="coefficients",...)
    
    title("Solution path")
    
    lines(rep(0,nsteps),lty=1,lwd=1,col="black")
    
    A<-lapply(1:length(TOP_index),function(i){lines(Feature_path[i,],lty=i,col=rainbow(length(TOP_index))[i])})
    
    legend("topright",lty=1:length(TOP_index),cex=0.5,col=rainbow(length(TOP_index)),
           legend=TOP_index,bty="n")
    
    plot(y=x$likelihood_iter,x=1:nsteps,xlab="steps",ylab="log-likelihood",type="b")
    title("Likelihood convergence")
    plot(y=x$Usearch,x=1:nsteps,xlab="steps",ylab="No. of tries")
    title("U-search")
    plot(y=x$iteration_data$FD,x=1:nsteps,xlab="steps",ylab="No. of changes")
    title("Retained feature change")

    dev.new()
    plot(x$coef_dist, x=1:length(x$coef_dist),xlab="steps",ylab="L2-dist b/w beta updates",
         main=" Coefficient convergence",type="b" )
    title(' Coefficient convergence')
  
    }else if(out_plot==3){
    
    par(mfrow=c(2,2))
      
      if(is.null(num_path)){num_path= length(x$ID_retained)}
      
      TOP_value<-rep(0,num_path)
      
      TOP_value<-sort(abs(Feature_path[,ncol(Feature_path)]),decreasing = T)[1:num_path]
      
      TOP_index<-(1:dim(x$X)[2])[Feature_path[,ncol(Feature_path)]%in% unique(c(TOP_value,-TOP_value))]
      
      TOP_index<-unique(c(TOP_index,which_path))
      
      Feature_path<-Feature_path[TOP_index,]
      
      plot(NULL, xlim=c(1,nsteps+1), ylim=c(floor(min(Feature_path[,nsteps])),ceiling(max(Feature_path[,nsteps]))),xlab="steps",ylab="coefficients",...)
      
      title("Solution path")
      
      lines(rep(0,nsteps),lty=1,lwd=1,col="black")
      
      A<-lapply(1:length(TOP_index),function(i){lines(Feature_path[i,],lty=i,col=rainbow(length(TOP_index))[i])})
      
      legend("topright",lty=1:length(TOP_index),cex=0.5,col=rainbow(length(TOP_index)),
             legend=TOP_index,bty="n")
      title('Coefficient convergence')
    plot(y=x$iteration_data$FD,x=1:nsteps,xlab="steps",ylab="No. of changes")
    title("Retained feature change")
    dev.new()
    plot(y=x$Usearch,x=1:nsteps,xlab="steps",ylab="Soulution path",...)
    title("U-search")
  }else if(out_plot==4){
    par(mfrow=c(2,2))
    
    if(is.null(num_path)){num_path= length(x$ID_retained)}
    
    TOP_value<-rep(0,num_path)
    
    TOP_value<-sort(abs(Feature_path[,ncol(Feature_path)]),decreasing = T)[1:num_path]
    
    TOP_index<-(1:dim(x$X)[2])[Feature_path[,ncol(Feature_path)]%in% unique(c(TOP_value,-TOP_value))]
    
    TOP_index<-unique(c(TOP_index,which_path))
    
    Feature_path<-Feature_path[TOP_index,]
    
    plot(NULL, xlim=c(1,nsteps+1), ylim=c(floor(min(Feature_path[,nsteps])),ceiling(max(Feature_path[,nsteps]))),xlab="steps",ylab="coefficients",...)
    
    title("Solution path")
    
    lines(rep(0,nsteps),lty=1,lwd=1,col="black")
    
    A<-lapply(1:length(TOP_index),function(i){lines(Feature_path[i,],lty=i,col=rainbow(length(TOP_index))[i])})
    
    legend("topright",lty=1:length(TOP_index),cex=0.5,col=rainbow(length(TOP_index)),
           legend=TOP_index,bty="n")
    

    plot(y=x$likelihood_iter,x=1:nsteps,xlab="steps",ylab="log-likelihood",type="b")
    title("Likelihood convergence")
    plot(x$coef_dist, x=1:length(x$coef_dist),xlab="steps",ylab="L2-dist b/w beta updates",
         main=" Coefficient convergence",type="b" )
    title(' Coefficient convergence')
    plot(y=x$Usearch,x=1:nsteps,xlab="steps",ylab="No. of tries")
    title("U-search")
    dev.new()
    plot(y=x$iteration_data$FD,x=1:nsteps,xlab="steps",ylab="No. of changes")
    title("Retained feature change")
  }
}

