\name{limitedDiversity}

\alias{limitedDiversity}

\title{Analyze QCA Solution Behaviour under Limited Empirical Diversity}

\description{
This evaluation function computes all solutions and unique models that result 
when all \emph{n}-tuples of minterms are systematically eliminated from a 
truth table. It has initially been programmed for \href{https://www.researchgate.net/publication/280555425_Often_Trusted_But_Never_Properly_Tested_Evaluating_Qualitative_Comparative_Analysis}{Baumgartner and Thiem (2015)} to test 
the correctness of QCA's three search strategies (conservative/complex, intermediate, 
parsimonious).
}

\usage{
limitedDiversity(truth.tab, outcome = "", exo.facs = c(""), sol.type = "ps", 
                 dir.exp = c(), n.drop = 1, c.minterms = FALSE)
}

\arguments{
  \item{truth.tab}{A truth table (either in plain format or a truth table object
                   of class "tt" generated by the \code{truthTable} function).}
  \item{outcome}{A character vector with the name of the outcome.}
  \item{exo.facs}{A character vector with the names of the exogenous factors.}
  \item{sol.type}{A character scalar specifying the QCA solution type that should 
        be applied; either "ps" (parsimonious solution), "ps+" (parsimonious 
        solution including both positive and contradiction minterms), "cs" (
        conservative solution) or "cs+" (conservative solution including both 
        positive and contradiction minterms).}
  \item{dir.exp}{A vector of directional expectations for deriving intermediate 
        solutions; can only be used in conjunction with \code{sol.type = "ps"}
        or \code{sol.type = "ps+"}.}
  \item{n.drop}{The number of minterms to be dropped from the truth table.}
  \item{c.minterms}{Logical, should contradictions be treated as positive minterms.}  
}

\details{
This function computes all solutions and unique models that result when all 
n-tuples of observed minterms are systematically dropped from a truth table.
It has been programmed for \href{https://www.researchgate.net/publication/280555425_Often_Trusted_But_Never_Properly_Tested_Evaluating_Qualitative_Comparative_Analysis}{Baumgartner and Thiem (2015)} to test the correctness of QCA's three search strategies (conservative/complex, intermediate, parsimonious) in conjunction with the \code{\link{submodels}} function.

The argument \code{truth.tab} specifies the truth table from which minterms are
to be dropped. The truth table can either be in plain format or be a truth table 
object of class "tt" generated by the \code{\link{truthTable}} function. If it 
is a truth table object, the arguments \code{outcome} and \code{exo.facs} need 
not be specified. The main difference between a truth table in plain format (as 
also used by Coincidence Analysis, for example (\href{http://dx.doi.org/10.1177/0049124109339369}{Baumgartner 2009})), is that each minterm includes only cases that have identical values on the exogenous factors and the endogenous factor. A QCA truth table object, in contrast, consists of minterms that include both cases with the outcome being analyzed as well as cases with the negation of this outcome. The ratio between these cases is used as the basis for the output function value. Thus, dropping minterms from plain truth tables will drop all cases that are identical with respect to all factors in the factor frame, whereas dropping minterms from QCA truth table objects will drop all cases that are identical with respect to all exogenous factors in the factor frame.

The argument \code{n.drop} specifies the size of the tuples of minterms to be 
dropped for generating limited empirical diversity. For example, if the truth 
table has 16 observed minterms, \code{n.drop = 2} creates 120 2-tuples, 
\code{n.drop = 3} creates 560 3-tuples, and so on.

The argument \code{c.minterms} specifies whether contradictions should be 
treated as positive minterms (\code{TRUE}) or negative minterms (\code{FALSE}).
}

\value{A list with the following three components:\cr
  \item{model.shares}{All unique models for all \emph{n}-tuples of dropped minterms 
                      and their occurrence shares.}
  \item{solutions}{The solutions for all \emph{n}-tuples of eliminated minterms.}
  \item{tt}{The truth table.}
}

\author{
Alrik Thiem (\href{http://www.alrik-thiem.net}{Personal Website}; \href{https://www.researchgate.net/profile/Alrik_Thiem}{ResearchGate Website})
}

\section{Contributors}{\tabular{ll}{
Dusa, Adrian\tab: programming\cr
Thiem, Alrik\tab: development, documentation, programming, testing
}}

\references{
Baumgartner, Michael. 2009. \dQuote{Inferring Causal Complexity.} 
\emph{Sociological Methods & Research} \strong{38} (1):71-101.
DOI: \href{http://dx.doi.org/10.1177/0049124109339369}{10.1177/0049124109339369}.

Baumgartner, Michael, and Alrik Thiem. 2015. \emph{Often Trusted but Never 
(Properly) Tested: Evaluating Qualitative Comparative Analysis}. Paper presented 
at the 12th Conference of the European Sociological Association, 25-28 August, 
Czech Technical University, Prague (Czech Republic). \href{https://www.researchgate.net/publication/280555425_Often_Trusted_But_Never_Properly_Tested_Evaluating_Qualitative_Comparative_Analysis}{Link}.
}

\seealso{\code{\link{eQMC}}, \code{\link{submodels}}, \code{\link{truthTable}}}

\examples{
\dontrun{
# number (n) of minterms (mt) and levels (lv) for each factor (exogenous 
# and endogenous)
n.mt <- 2^5 
n.lv <- rep(2, 5) 

# expand to unevaluated truth table and assign case/factor labels
tt.unev <- data.frame(mintermMatrix(n.lv))
dimnames(tt.unev) <- list(1:n.mt, c(LETTERS[1:4], "Z"))

# cull rows from tt.unev that are compatible with aB + Bc + D <=> Z 
# to produce evaluated truth table tt.ev
tt.ev <- tt.unev[pmax(tt.unev$D, pmin(1 - tt.unev$A, tt.unev$B),
                 pmin(tt.unev$B, 1 - tt.unev$C)) == tt.unev$Z, ]

# conservative solutions for all 1-tuples (16)
limitedDiversity(tt.ev, outcome = "Z", sol.type = "cs")$model.shares

# using a truth table object of class 'tt' created by eQMC function
#------------------------------------------------------------------
data(d.represent)
tt <- truthTable(d.represent, outcome = "WNP")

# with objects of class 'tt', exogenous factors and the outcome need not be 
# specified again
limitedDiversity(tt)

# proof that the conservative/complex solution type of QCA is incorrect, 
# (see Baumgartner and Thiem (2015) for more details)
#-----------------------------------------------------------------------

# 1. build truth table on the basis of reference model aB + Bc + D
tt <- data.frame(mintermMatrix(rep(2, 5)))
dimnames(tt) <- list(as.character(1:32), c(LETTERS[1:4], "OUT"))
tt <- tt[pmax(pmin(1 - tt$A, tt$B), pmin(tt$B, 1 - tt$C), tt$D) == tt$OUT, ]

# 2. generate all conservative/complex solutions for all 16 + 120 scenarios 
# of one/two dropped minterm/s
sollist.cs <- vector("list", 2)
sollist.cs <- lapply(1:2, function (x) {
  limitedDiversity(tt, outcome = "OUT", sol.type = "cs", n.drop = x)
  }
)

# 3. compute in how many scenarios a correctness-preserving submodel of 
# the reference model was part of the solution (43.75\% for one dropped 
# minterm and  16.67\% for two dropped minterms)
cs.correct <- numeric(2)
cs.correct <- sapply(1:2, function (x) {round((sum(unlist(lapply(
  sollist.cs[[x]][[2]], function (y) {any(
    submodels("aB + Bc + D")$submodels \%in\% y)}
  ))) / choose(16, x))*100, 2)}
)
cs.correct
}
}

\keyword{functions}
