library(testthat)
#library(MetaUtility)
library(devtools)
library(metafor)
library(dplyr)

# helper fn for simulating meta-analysis with publication bias

# # p: row of parameters dataframe
sim_data = function(p) {

  N = p$k * p$per.cluster

  # generate cluster random intercepts
  gam1 = rnorm( n = p$k, mean = 0, sd = sqrt( p$V.gam ) )
  gam1i = rep( gam1, each = p$per.cluster )

  # generate individual-study random intercepts
  gam2i = rnorm( n = N, mean = 0, sd = sqrt( p$V - p$V.gam ) )

  # individual study means
  mui = p$mu + gam1i + gam2i
  sei = runif( n = N, min = p$sei.min, max = p$sei.max )
  yi = rnorm( n = N, mean = mui, sd = sei )

  d = data.frame( cluster = rep(1:p$k, each = p$per.cluster),
                  Study.name = 1:N,
                  yi = yi,
                  sei = sei,
                  vi = sei^2,
                  pval = 2 * ( 1 - pnorm( abs(yi) / sei ) ) )

  # 1-tailed publication bias
  signif = d$pval < 0.05 & d$yi > 0
  publish = rep( 1, nrow(d) )
  publish[ signif == FALSE ] = rbinom( n = sum(signif == FALSE), size = 1, prob = 1/p$eta )

  d$weight = 1
  d$weight[ signif == 0 ] = p$eta
  d = d[ publish == 1, ]

  return(d)
}

# ##### Sanity Check #####
# d = sim_data( data.frame( k = 5,
#                           per.cluster = 20,
#                           mu = .5,
#                           V = 1,
#                           V.gam = .5,
#                           sei.min = 0.01,
#                           sei.max = 0.01,
#                           eta = 1 ) )


# does corrected_meta agree with regular meta-analysis fns when there
#  is no selection?
test_that("corrected_meta #1", {

  ##### Recover Regular FE model With Eta = 1 #####
  # when using z-based inference and eta = 1,
  # should match metafor
  dat = escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
  FE.plain = rma( yi, vi, data = dat, method = "FE" )
  # should match corrected with eta = 1

  FE.adj = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = 1,
                             model = "fixed",
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = FALSE )

  expect_equal( FE.adj$est, as.numeric(FE.plain$b), tol = 0.001 )
  expect_equal( FE.adj$se, as.numeric(FE.plain$se), tol = 0.001 )
  expect_equal( FE.adj$lo, as.numeric(FE.plain$ci.lb), tol = 0.001 )
  expect_equal( FE.adj$hi, as.numeric(FE.plain$ci.ub), tol = 0.001 )
  expect_equal( FE.adj$pval, as.numeric(FE.plain$pval), tol = 0.001 )

  ##### Recover Regular Robust Indepenent Model With Eta = 1 #####
  for ( .small in c(TRUE, FALSE) ) {
    dat$clustervar = 1:length(dat$yi)

    meta.re = rma.uni( yi = dat$yi,
                       vi = dat$vi)
    t2hat.naive = meta.re$tau2

    RI.robust = robu( yi ~ 1,
                      studynum = clustervar,
                      data = dat,
                      userweights = 1 / (vi + t2hat.naive),
                      var.eff.size = vi,
                      small = .small )


    RI.adj = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = 1,
                             model = "robust",
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = .small )

    expect_equal( RI.adj$est, as.numeric( as.numeric(RI.robust$b.r) ), tol = 0.001 )
    expect_equal( RI.adj$se, as.numeric( RI.robust$reg_table$SE ), tol = 0.001 )
    expect_equal( RI.adj$lo, as.numeric( RI.robust$reg_table$CI.L ), tol = 0.001 )
    expect_equal( RI.adj$hi, as.numeric( RI.robust$reg_table$CI.U ), tol = 0.001 )
    expect_equal( RI.adj$pval, as.numeric( RI.robust$reg_table$prob ), tol = 0.001 )
  }


  ##### Recover Regular Robust Clustered Model With Eta = 1 #####
  for ( .small in c(TRUE, FALSE) ) {
    dat$clustervar = dat$author

    meta.re = rma.uni( yi = dat$yi,
                       vi = dat$vi)
    t2hat.naive = meta.re$tau2

    RI.robust = robu( yi ~ 1,
                      studynum = clustervar,
                      data = dat,
                      userweights = 1 / (vi + t2hat.naive),
                      var.eff.size = vi,
                      small = .small )


    RI.adj = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = 1,
                             model = "robust",
                             clustervar = clustervar,
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = .small )

    expect_equal( RI.adj$est, as.numeric( as.numeric(RI.robust$b.r) ), tol = 0.001 )
    expect_equal( RI.adj$se, as.numeric( RI.robust$reg_table$SE ), tol = 0.001 )
    expect_equal( RI.adj$lo, as.numeric( RI.robust$reg_table$CI.L ), tol = 0.001 )
    expect_equal( RI.adj$hi, as.numeric( RI.robust$reg_table$CI.U ), tol = 0.001 )
    expect_equal( RI.adj$pval, as.numeric( RI.robust$reg_table$prob ), tol = 0.001 )
  }

})


# do svalue and corrected_meta agree?
test_that("svalue #1", {

  dat = escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)

  for ( .CI.level in c(.8, .95) ) {
    for ( .small in c(TRUE, FALSE) ) {

      svals = svalue( yi = dat$yi,
                      vi = dat$vi,
                      q = 0,
                      model = "fixed",
                      CI.level = 0.95,
                      small = .small )$svals

      # CI upper limit should be exactly 0 when eta = sval.ci
      meta = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = as.numeric(svals["sval.ci"]),
                             model = "fixed",
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = .small )

      expect_equal( meta$hi, 0 )

    }
  }
} )

# bookmark
# do svalue and corrected_meta agree? (#2)
test_that("svalue #1", {

  dat = sim_data( data.frame( k = 50,
                            per.cluster = 5,
                            mu = -0.5,
                            V = 0.25,
                            V.gam = 0,
                            sei.min = 0.1,
                            sei.max = 1,
                            eta = 2 ) )

  # uncorrected meta
  m0 = corrected_meta( yi = dat$yi,
                  vi = dat$vi,
                  eta = 1,
                  model = "robust" )

  # choose q to be between worst-case CI limit and actual CI limit so that
  #  both s-values will be well-defined
  meta.worst =  corrected_meta( yi = dat$yi,  # not flipped
                                vi = dat$vi,
                                eta = 2,
                                model = "robust" )

  # use upper limit since this meta has negative point estimate
  q = mean( c(m0$hi, meta.worst$hi) )



  for ( .CI.level in c(.8, .95) ) {
    for ( .small in c(TRUE, FALSE) ) {

      svals = svalue( yi = dat$yi,
                      vi = dat$vi,
                      q = q,
                      model = "robust",
                      CI.level = 0.95,
                      small = .small )$svals

      # estimate should be exactly equal to q when eta = sval.est
      meta = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = as.numeric(svals["sval.est"]),
                             model = "robust",
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = .small )
      expect_equal( meta$est, q, tol = 0.001 )

      # and CI upper limit should be exactly 0 when eta = sval.ci
      meta = corrected_meta( yi = dat$yi,
                             vi = dat$vi,
                             eta = as.numeric(svals["sval.ci"]),
                             model = "robust",
                             selection.tails = 1,
                             CI.level = 0.95,
                             small = .small )

      expect_equal( meta$hi, q, tol = 0.001 )

    }
  }
} )



# # is worst-case meta correct for both 1-tailed and 2-tailed selection?
# test_that("svalue #2", {
#
#   dat = escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)
#   # doctor one data point so that 2-tailed will be different from 1-tailed
#   dat$yi[2] = -dat$yi[2]
#
#   # first get worst-case meta-analysis using package
#   m2 = svalue( yi = dat$yi,
#                vi = dat$vi,
#                q = 0,
#                model = "fixed",
#                CI.level = 0.95,
#                small = FALSE )$meta.bd
#
#   # flip signs
#   rma.uni( dat$yi,
#            dat$vi,
#            method = "FE" )$b
#   dat$yi = -dat$yi
#
#   # calculate p-values
#   Z = abs( dat$yi / sqrt(dat$vi) )
#   pval = 2 * ( 1 - pnorm(Z) )
#
#   # affirmative indicators
#   A.1tail = (pval < .05) & (dat$yi > 0)
#   A.2tail = (pval < 0.05)
#
#   table(A.1tail)
#   table(A.2tail)
#
#   # note sign flip back to original
#   m1 = rma.uni( -dat$yi[ A.1tail == FALSE ],
#                  dat$vi[ A.1tail == FALSE ],
#                  method = "FE" )
#
#   expect_equal( as.numeric( m1$b ),
#                 as.numeric( m2$b ) )
# } )



# does it reject bad choices of q?
test_that( "svalue #3", {
  dat = escalc(measure="RR", ai=tpos, bi=tneg, ci=cpos, di=cneg, data=dat.bcg)

  expect_error( svalue( yi = dat$yi,
          vi = dat$vi,
          q = -3,
          model = "fixed",
          CI.level = 0.95,
          small = FALSE ) )

  # reverse signs
  expect_error( svalue( yi = -dat$yi,
                        vi = dat$vi,
                        q = .8,
                        model = "fixed",
                        CI.level = 0.95,
                        small = FALSE ) )
} )


# does svalue give correct results when the s-value is greater than the highest
#  value in eta grid?
test_that( "svalue #4", {

  eta = 3

  d = sim_data( data.frame( k = 50,
                            per.cluster = 5,
                            mu = 0.5,
                            V = 0,
                            V.gam = 0,
                            sei.min = 0.1,
                            sei.max = 1,
                            eta = eta ) )

  svals = svalue( yi = d$yi,
          vi = d$vi,
          q = 0,
          model = "robust",
          eta.grid = seq(1,10,1),
          CI.level = 0.95,
          small = FALSE )

} )


# does significance_funnel give error if no non-affirmative studies?
test_that( "significance_funnel #1", {
  eta = 50

  d = sim_data( data.frame( k = 50,
                            per.cluster = 1,
                            mu = 0.5,
                            V = 0.1,
                            V.gam = 0,
                            # sei.min = 0.2,
                            # sei.max = .9,
                            sei.min = .1,
                            sei.max=.1,
                            eta = eta ) )


  # also see if significance_funnel works
  expect_error( significance_funnel( yi = d$yi,
                       vi = d$vi ) )
} )



# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#
#                                     MANUAL TESTS                                    #
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#


# ##### Visually Test Significance Funnel on Simulated Data #####
# eta = 5
#
# d = sim_data( data.frame( k = 500,
#                           per.cluster = 1,
#                           mu = -0.1,
#                           V = 0.1,
#                           V.gam = 0,
#                           sei.min = 0.01,
#                           sei.max = 0.05,
#                           eta = eta ) )
#
#
# # also see if significance_funnel works
# significance_funnel( yi = d$yi,
#                      vi = d$vi )


# ##### Reproduce Manuscript Stats for Boehm Data #####
#
# setwd("~/Dropbox/Personal computer/Independent studies/Sensitivity analysis for publication bias (SAPB)/Linked to OSF (SAPB)/Applied examples/Data/Boehm data")
#
# d = read.csv("boehm_prepped.csv")
#
# significance_funnel( yi = d$yi,
#                      vi = d$vi )
#
# ### fixed effects
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = 0,
#         model = "fixed",
#         small = TRUE)
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = r_to_z(0.1),
#         model = "fixed",
#         small = TRUE)
#
#
# ### robust independent
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = 0,
#         model = "robust" )
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = r_to_z(0.1),
#         model = "robust" )
#
# ### robust clusters
# svalue( yi = d$yi,
#         vi = d$vi,
#         clustervar = d$study,
#         q = 0,
#         model = "robust" )
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         clustervar = d$study,
#         q = r_to_z(0.1),
#         model = "robust" )
#
#
#
# ##### Reproduce Manuscript Stats for Anderson Data #####
#
# setwd("~/Dropbox/Personal computer/Independent studies/Sensitivity analysis for publication bias (SAPB)/Private data component/Anderson data")
#
# d = read.csv("anderson_prepped.csv")
#
# significance_funnel( yi = d$yi,
#                      vi = d$vi )
#
# ### fixed effects
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = 0,
#         model = "fixed",
#         small = TRUE)
# # matches :)
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = r_to_z(0.1),
#         model = "fixed",
#         small = TRUE)
#
#
# ### robust independent
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = 0,
#         model = "robust" )
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         q = r_to_z(0.1),
#         model = "robust" )
# # should move CI to 0.10
# corrected_meta( yi = d$yi,
#                 vi = d$vi,
#                 eta = 5.25,
#                 model = "robust" )
# # yes :)
#
# ### robust clusters
# svalue( yi = d$yi,
#         vi = d$vi,
#         clustervar = d$cluster,
#         q = 0,
#         model = "robust" )
#
# svalue( yi = d$yi,
#         vi = d$vi,
#         clustervar = d$cluster,
#         q = r_to_z(0.1),
#         model = "robust" )
# # should move CI to 0.10
# corrected_meta( yi = d$yi,
#                 vi = d$vi,
#                 eta = 3.5,
#                 model = "robust",
#                 clustervar = d$cluster )






