% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_cv.R
\name{get_rse}
\alias{get_rse}
\title{Compute the expected parameter relative standard errors}
\usage{
get_rse(fmf, poped.db, bpop = poped.db$parameters$bpop[, 2, drop = F],
  d = poped.db$parameters$d[, 2, drop = F], docc = poped.db$parameters$docc,
  sigma = poped.db$parameters$sigma, use_percent = T,
  fim.calc.type = poped.db$settings$iFIMCalculationType)
}
\arguments{
\item{fmf}{The initial value of the FIM. If set to zero then it is computed.}

\item{poped.db}{A PopED database.}

\item{bpop}{Matrix defining the fixed effects, per row (row number = parameter_number) we should have:
\itemize{
\item column 1 the type of the distribution for E-family designs (0 = Fixed, 1 = Normal, 2 = Uniform,
 3 = User Defined Distribution, 4 = lognormal and 5 = truncated normal)
\item column 2  defines the mean.
\item column 3 defines the variance of the distribution (or length of uniform distribution).
}
Can also just supply the parameter values as a vector \code{c()} if no uncertainty around the 
parameter value is to be used.}

\item{d}{Matrix defining the diagnonals of the IIV (same logic as for the fixed efects 
matrix bpop to define uncertainty). One can also just supply the parameter values as a \code{c()}.}

\item{docc}{Matrix defining the IOV, the IOV variances and the IOV distribution as for d and bpop.}

\item{sigma}{Matrix defining the variances can covariances of the residual variability terms of the model.
can also just supply the diagnonal parameter values (variances) as a \code{c()}.}

\item{use_percent}{Should RSE be reported as percent or not?}

\item{fim.calc.type}{The method used for calculating the FIM. Potential values:
\itemize{
\item 0 = Full FIM.  No assumption that fixed and random effects are uncorrelated.  See \code{\link{mftot0}}.
\item 1 = Reduced FIM. Assume that there is no correlation in the FIM between the fixed and random effects, and set these elements in 
the FIM to zero. See \code{\link{mftot1}}.
\item 2 = weighted models (placeholder).
\item 3 = Not currently used.
\item 4 = Reduced FIM and computing all derivatives with respect to the standard deviation of the residual unexplained variation (sqrt(SIGMA) in NONMEM). 
This matches what is done in PFIM, and assumes that the standard deviation of the residual unexplained variation is the estimated parameter
(NOTE: NONMEM estimates the variance of the resudual unexplained variation by default). See \code{\link{mftot4}}.
\item 5 = Full FIM parameterized with A,B,C matrices & derivative of variance. See \code{\link{mftot5}}.
\item 6 = Calculate one model switch at a time, good for large matrices. See \code{\link{mftot6}}.
\item 7 = Reduced FIM parameterized with A,B,C matrices & derivative of variance See \code{\link{mftot7}}.
}}
}
\value{
A named list of RSE values.
}
\description{
This function  computes the expected relative standard errors of a model given a design and a previously computed
FIM.
}
\examples{
## Warfarin example from software comparison in:
## Nyberg et al., "Methods and software tools for design evaluation 
##   for population pharmacokinetics-pharmacodynamics studies", 
##   Br. J. Clin. Pharm., 2014. 

library(PopED)

## find the parameters that are needed to define from the structural model
ff.PK.1.comp.oral.md.CL

## -- parameter definition function 
## -- names match parameters in function ff
sfg <- function(x,a,bpop,b,bocc){
  parameters=c(CL=bpop[1]*exp(b[1]),
               V=bpop[2]*exp(b[2]),
               KA=bpop[3]*exp(b[3]),
               Favail=bpop[4],
               DOSE=a[1])
    return(parameters) 
}

## -- Define initial design  and design space
poped.db <- create.poped.database(ff_file="ff.PK.1.comp.oral.sd.CL",
                                  fg_file="sfg",
                                  fError_file="feps.prop",
                                  bpop=c(CL=0.15, V=8, KA=1.0, Favail=1), 
                                  notfixed_bpop=c(1,1,1,0),
                                  d=c(CL=0.07, V=0.02, KA=0.6), 
                                  sigma=0.01,
                                  groupsize=32,
                                  xt=c( 0.5,1,2,6,24,36,72,120),
                                  minxt=0,
                                  maxxt=120,
                                  a=70)


## evaluate initial design with the reduced FIM
FIM.1 <- evaluate.fim(poped.db) 
FIM.1
det(FIM.1)
get_rse(FIM.1,poped.db)

## evaluate initial design with the full FIM
FIM.0 <- evaluate.fim(poped.db,fim.calc.type=0) 
FIM.0
det(FIM.0)
get_rse(FIM.0,poped.db,fim.calc.type=0)

## evaluate initial design with the reduced FIM 
## computing all derivatives with respect to the 
## standard deviation of the residual unexplained variation 
FIM.4 <- evaluate.fim(poped.db,fim.calc.type=4) 
FIM.4
det(FIM.4)
get_rse(FIM.4,poped.db,fim.calc.type=4)

## evaluate initial design with the full FIM with A,B,C matricies
## should give same answer as fim.calc.type=0
FIM.5 <- evaluate.fim(poped.db,fim.calc.type=5) 
FIM.5
det(FIM.5)
get_rse(FIM.5,poped.db,fim.calc.type=5)

## evaluate initial design with the reduced FIM with 
## A,B,C matricies and derivative of variance
## should give same answer as fim.calc.type=1 (default)
FIM.7 <- evaluate.fim(poped.db,fim.calc.type=7) 
FIM.7
det(FIM.7)
get_rse(FIM.7,poped.db,fim.calc.type=7)

}
\seealso{
Other evaluate_design: \code{\link{evaluate.fim}},
  \code{\link{evaluate_design}},
  \code{\link{model_prediction}},
  \code{\link{plot_efficiency_of_windows}},
  \code{\link{plot_model_prediction}}
}

