% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/friedmanTest.R
\name{friedmanTest}
\alias{friedmanTest}
\alias{friedmanTest.default}
\title{Friedman Rank Sum Test}
\usage{
friedmanTest(y, ...)

\method{friedmanTest}{default}(y, groups, blocks, dist = c("Chisquare", "FDist"), ...)
}
\arguments{
\item{y}{a numeric vector of data values, or a list of numeric data
vectors.}

\item{groups}{a vector or factor object giving the group for the
corresponding elements of \code{"x"}. Ignored with a warning if \code{"x"} is a list.}

\item{blocks}{a vector or factor object giving the block for the
corresponding elements of \code{"x"}.
Ignored with a warning if \code{"x"} is a list.}

\item{dist}{the test distribution. Defaults to \code{Chisquare}.}

\item{\ldots}{further arguments to be passed to or from methods.}
}
\value{
A list with class \code{"htest"} containing the following components:
\describe{
 \item{method}{a character string indicating what type of test was performed.}
 \item{data.name}{a character string giving the name(s) of the data.}
 \item{statistic}{the estimated quantile of the test statistic.}
 \item{p.value}{the p-value for the test.}
 \item{parameter}{the parameters of the test statistic, if any.}
 \item{alternative}{a character string describing the alternative hypothesis.}
\item{estimates}{the estimates, if any.}
\item{null.value}{the estimate under the null hypothesis, if any.}
}
}
\description{
Performs a Friedman rank sum test. The null hypothesis
H\eqn{_0: \theta_i = \theta_j~~(i \ne j)} is tested against the
alternative H\eqn{_{\mathrm{A}}: \theta_i \ne \theta_j}, with at least
one inequality beeing strict.
}
\details{
The function has implemented Friedman's test as well as
the extension of Conover anf Iman (1981). Friedman's
test statistic is assymptotically chi-squared distributed.
Consequently, the default test distribution is \code{dist = "Chisquare"}.

If \code{dist = "FDist"} is selected, than the approach of
Conover and Imam (1981) is performed.
The Friedman Test using the \eqn{F}-distribution leads to
the same results as doing an two-way Analysis of Variance without
interaction on rank transformed data.
}
\examples{
## Hollander & Wolfe (1973), p. 140ff.
## Comparison of three methods ("round out", "narrow angle", and
##  "wide angle") for rounding first base.  For each of 18 players
##  and the three method, the average time of two runs from a point on
##  the first base line 35ft from home plate to a point 15ft short of
##  second base is recorded.
RoundingTimes <-
matrix(c(5.40, 5.50, 5.55,
        5.85, 5.70, 5.75,
        5.20, 5.60, 5.50,
        5.55, 5.50, 5.40,
        5.90, 5.85, 5.70,
        5.45, 5.55, 5.60,
        5.40, 5.40, 5.35,
        5.45, 5.50, 5.35,
        5.25, 5.15, 5.00,
        5.85, 5.80, 5.70,
        5.25, 5.20, 5.10,
        5.65, 5.55, 5.45,
        5.60, 5.35, 5.45,
        5.05, 5.00, 4.95,
        5.50, 5.50, 5.40,
        5.45, 5.55, 5.50,
        5.55, 5.55, 5.35,
        5.45, 5.50, 5.55,
        5.50, 5.45, 5.25,
        5.65, 5.60, 5.40,
        5.70, 5.65, 5.55,
        6.30, 6.30, 6.25),
      nrow = 22,
      byrow = TRUE,
      dimnames = list(1 : 22,
                      c("Round Out", "Narrow Angle", "Wide Angle")))

## Chisquare distribution
friedmanTest(RoundingTimes)

## check with friedman.test from R stats
friedman.test(RoundingTimes)

## F-distribution
friedmanTest(RoundingTimes, dist = "FDist")

## Check with One-way repeated measure ANOVA
rmat <- RoundingTimes
for (i in 1:length(RoundingTimes[,1])) rmat[i,] <- rank(rmat[i,])
dataf <- data.frame(
    y = y <- as.vector(rmat),
    g = g <- factor(c(col(RoundingTimes))),
    b = b <- factor(c(row(RoundingTimes))))
summary(aov(y ~ g + Error(b), data = dataf))

}
\references{
Conover, W.J., Iman, R.L. (1981) Rank Transformations as a Bridge
 Between Parametric and Nonparametric Statistics.
 \emph{Am Stat} \bold{35}, 124--129.

Sachs, L. (1997) \emph{Angewandte Statistik}. Berlin: Springer.
}
\seealso{
\code{\link{friedman.test}}
}
\keyword{htest}
\keyword{nonparametric}
