#' @title This function inserts missingness (Missing at Random - MAR) into the given data sets.
#' @description
#' Missing values (MAR) will be added to the generated data sets (Generated by sim.skewed() or sim.normal() functions).
#' Under MAR, the missingness was associated with the values of the variable in the data set except itself. If baseV parameter was not given, two different and random variables in the data set are selected, and the missing values are assigned based on the mean of the two variables on the selected item.
#' For example, if the data has 8 items and the second item will be assigned MAR values, two items among the item 1, 3, 4, 5, 6, 7, and 8 were selected randomly, let’s say items 5 and 7. The mean of the items was then calculated and the values were sorted.
#' Then, based on the given percent of missingness, 90 percent of the missing values were selected from the top.  The remaining 10 percent of missing values were assigned from the rest of the variable.
#' For example, let’s say the sample size was 300, and 20 percent of missingness was wanted (missing count: 300x20%=60). The mean of the random variables was sorted by decreasing. Missing values were added randomly to 54 of the top 60 values (60x90%=54). The remaining 6 values (60-54, 10%) were randomly assigned missing among the 240 values (300-60=240).
#' The missing values are shown as "NA" in the data files. The new data sets which have missing values will be saved as a different data file.
#' In each data file, the first column shows sample numbers. The second and the other columns show actual data sets for each item.
#' There will be a file named "MAR_List.dat". The file includes the names of the data sets which has missing values in it. Besides, a file named “Model_MAR_relations.dat” shows which item was associated with which random items that were used for the MAR calculation.
#'
#' @author Fatih Orcan
#' @importFrom utils read.table write.table
#' @param misg A vector of 0s and 1s for each item. 0 indicates non-missing and 1 indicates items which have missing values. If misg is not indicated all items are considered as missing.
#' @param baseV A list of  items that MAR will be calculated based on. It has to be match with the misg parameter. If it is not given, two random items (except the variable itself) will be selected and used to get MAR values for the given items.
#' @param perct The percent of missingness. The default is 10 percent.
#' @param dataList List of the names of data sets generated earlier either with the package functions or any other software.
#' @param f.loc File location. It indicates where the simulated data sets and "dataList" are located.
#' @export
#' @examples
#'
#' #   Data needed to be generated at the first step.
#'
#' fc<-fcors.value(nf=3, cors=c(1,.5,.6,.5,1,.4,.6,.4,1))
#' fl<-loading.value(nf=3, fl.loads=c(.5,.5,.5,0,0,0,0,0,0,0,0,.6,.6,.6,0,0,0,0,0,0,0,0,.4,.4))
#' floc<-tempdir()
#' sim.normal(nd=10, ss=100, fcors=fc, loading<-fl,  f.loc=floc)
#'
#'  #  Missing values were added at the second step.
#'
#' mis.items<-c(1,0,1,1,0,0,0,0)
#' bV<-list(c(0,0,0,0,0,0,1,1),NA,c(0,0,0,0,0,1,1,0),c(0,0,0,0,0,1,1,1), NA,NA,NA,NA)
#' dl<-"Data_List.dat"  # should be located in the working directory.
#' MAR.data(misg = mis.items, baseV=bV, perct = 20, dataList = dl, f.loc=floc )

MAR.data<-function(misg=NULL,baseV=NULL, perct=10, dataList="Data_List.dat", f.loc){

  data.names<-as.matrix(read.table(paste(f.loc, "/", dataList,sep=""), header = FALSE))
  misg.names<-data.names
  veri<-read.table(paste(f.loc,"/", data.names[1,],sep=""))
  colnames(veri)<-c("ID", paste("x",seq(1:(dim(veri)[2]-1)),sep=""))

  nd<-dim(data.names)[1]
  nitem<-dim(veri)[2]-1
  ss<-dim(veri)[1]

  n.vec<-c(1:nitem)

  basR<-list(NA)    # Empty list to save randomly selected item for the MAR
  for(bs in 1:nitem){
    r.i<-sample(n.vec[-bs],2)      # Two random items that MAR was based on for the given item.
    aa<-rep(0,nitem)
    aa[r.i]<-1
    ifelse(misg[bs]==1, basR[[bs]]<-aa, basR[[bs]]<-NA)
      }


  for(i in 1:nd){
    veri<-read.table(paste(f.loc,"/", data.names[i,],sep=""))
    misg.names[i,]<-c(paste("MAR_",i,".dat", sep=""))

    MAR.data<-veri
    for(j in 1:nitem){
      if(misg[j]==0){
        MAR.data[,j+1]<-veri[,j+1]}
      else if(misg[j]==1){
        mis.ss<-(perct/100)*ss

      if(is.null(baseV)==TRUE){

        c.v<-apply (veri[,which(basR[[j]]==1)+1], 1, mean, na.rm=T)
        } # Calculated values that MAR will be based on

        if(is.null(baseV)==FALSE){
        baseV[j]
        c.v<-apply (veri[,which(baseV[[j]]==1)+1], 1, mean, na.rm=T) # Calculated values that MAR will be based on
        }

        n.veri<-cbind(veri,c.v)

        MAR.data[sample(order(n.veri[,10],decreasing = T)[1:mis.ss],size=mis.ss*.9),j+1]<-NA
        MAR.data[sample(order(n.veri[,10],decreasing = T)[(mis.ss+1):ss],size=mis.ss*.1),j+1]<-NA

        message(paste("MAR_",i,".dat was completed", sep=""))
        }


      else {stop("Please use only 0s or 1s to indicated missingness")}}
    write.table(MAR.data, file= paste(f.loc, "/MAR_",i,".dat", sep=""), sep = "\t",
                col.names = FALSE, row.names = FALSE,quote = FALSE)
  }
  write.table(misg.names,file=paste(f.loc,"/MAR_List.dat", sep = ""),
              col.names = FALSE, row.names = FALSE, quote = FALSE)
  message("Done!...")

  if(is.null(baseV)==TRUE){
  sink(paste(f.loc,"/Model_MAR_relations.dat", sep = ""))
  print("Two random items that MAR was based on:")
  print(basR)
  sink()}

  if(is.null(baseV)==FALSE){
    sink(paste(f.loc,"/Model_MAR_relations.dat", sep = ""))
    print("The items that MAR was based on:")
    print(baseV)
    sink()}


}
