% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MRGmerge.R
\name{MRGmerge}
\alias{MRGmerge}
\title{Merge two or more multi-resolution grids to a common resolution}
\usage{
MRGmerge(himg1, himg2, vars1, vars2, na.rm = TRUE, postProcess = FALSE, ...)
}
\arguments{
\item{himg1}{Either a grid resulting from a call to multiResGrid, or a list of such grids}

\item{himg2}{A grid resulting from a call to multiResGrid}

\item{vars1}{Variable(s) of interest that should be merged from the first grid, or a list of variables,
one for each grid in the list \code{himg1}}

\item{vars2}{Variable(s) of interest that should be merged from the second grid}

\item{na.rm}{Should NA values be removed when summing values (essentially
treating them equal to zero)}

\item{postProcess}{Logical; should the postprocessing be done as part
of creation of the multiresolution grid (TRUE), or be done in a separate 
step afterwards (FALSE). The second option is useful when wanting
to check the confidential grid cells of the final map}

\item{...}{Additional grids (himg3, himg4, ...) and variables (vars3, vars4, ...) to be merged. 
Additional grids and variables must be named.}
}
\description{
Merge two or more multi-resolution grids to a common resolution
}
\details{
This function can merge different multi-resolution grids to a common 
resolution, i.e., it will select the grid cells with the lowest resolution,
as these are the ones defining the restrictions.  

The function will merge the variable names in \code{vars1, vars2, ...}
if they exist. If they are missing, the function will look for variable
names in the attributes of the grids (\code{attr(himg, "vars")}). These
are added by \code{\link{multiResGrid}}, but will often disappear if the
grid has been manipulated, or has been exported to another format for transmission.

If the variables are not given as \code{vars} or attributes, the function
will try to guess them from the column names. Typical column names used by
MRG (mostly temporary variables such as \code{small}, \code{confidential} etc)
will be ignored. If variable names partly coincide with any of these names,
or with \code{count}, \code{res}, \code{geometry}, it is necessary to specify vars.

The multi-resolution grids must be passed as named parameters if more than two 
are given. 

Common variable names in different grids should be avoided.

The default of the function is to treat NA-values as zeroes when merging 
(through \code{na.rm} in sums).
It will therefore not be possible to separate restricted grid cells
from grid cells with zero observations after merging, except for the ones that
have been left as they were. The alternative would
be a much higher number of NA-values in the merged grids.

The resulting grid will most likely not have exactly the same values as a 
multi-resolution grid produced 
directly from the microdata. If the input-grids have been post-processed
(the normal situation when not having access to the microdata), the 
grid cell values have usually been rounded, and some might have been 
suppressed. As these rounded and potentially suppressed values are summed,
their values are likely to deviate from those that are computed directly
from the microdata through a joint gridding process.
}
\examples{
\donttest{
library(sf)

# These are SYNTHETIC agricultural FSS data 
data(ifs_dk) # Census data
ifs_weight = ifs_dk \%>\% dplyr::filter(Sample == 1) # Extract weighted subsample

# Create spatial data
ifg = fssgeo(ifs_dk, locAdj = "LL")
fsg = fssgeo(ifs_weight, locAdj = "LL")

# We use the numeric part of the farmtype to create a third variable. This 
# is done for the an example, the value does not have any meaning when treated 
# like this
ifg$ft = as.numeric(substr(ifg$FARMTYPE, 3, 4))^2

ress = c(1,5,10,20,40, 80, 160)*1000
# Create regular grid of the variables
ifl = gridData(ifg, vars = c("UAA", "UAAXK0000_ORG", "ft"), res = ress)

# Create the different multi-resolution grids
himg1 = multiResGrid(ifl, vars = "UAA", ifg = ifg, postProcess = FALSE)
himg2 = multiResGrid(ifl, vars = "UAAXK0000_ORG", ifg = ifg, postProcess = FALSE)
himg3 = multiResGrid(ifl, vars = "ft", ifg = ifg, postProcess = FALSE)

# The grids have different number of polygons
dim(himg1)
dim(himg2)
dim(himg3)

hh1 = MRGmerge(himg1, himg2, himg3 = himg3)
dim(hh1)
# Postprocessing can also be done on the merged object
hh11 = MRGmerge(himg1, himg2, himg3 = himg3, postProcess = TRUE, rounding = -1)
dim(hh11)
summary(hh1$UAA-hh11$UAA)

# Try to avoid having the same variable names in two different multi-resolution grids,
# although it works in this case

himg21 = multiResGrid(ifl, vars = c("UAA", "UAAXK0000_ORG"), ifg = ifg, postProcess = FALSE)
hh3 = MRGmerge(himg1, himg21, himg3 = himg3)
summary(hh3[, c("UAA", "UAA.1")])

himg4 = multiResGrid(ifl, vars = c("UAA", "ft", "UAAXK0000_ORG"), ifg = ifg, postProcess = FALSE)
summary(hh1[, c("UAA", "UAAXK0000_ORG", "ft")])
summary(himg4[, c("UAA", "UAAXK0000_ORG", "ft")])
}
           
           
}
