% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LassoGEE.R
\name{LassoGEE}
\alias{LassoGEE}
\title{Function to fit penalized GEE by I-CGD algorithm.}
\usage{
LassoGEE(
  X,
  y,
  id,
  family = binomial("probit"),
  lambda,
  corstr = "independence",
  method = c("CGD", "RWL"),
  beta.ini = NULL,
  R = NULL,
  scale.fix = TRUE,
  scale.value = 1,
  maxiter = 50,
  tol = 0.001,
  silent = TRUE,
  Mv = NULL,
  verbose = TRUE
)
}
\arguments{
\item{X}{A design matrix of dimension  \code{(nm) * p}.}

\item{y}{A response vector of length  \code{m * n}.}

\item{id}{A vector for identifying subjects/clusters.}

\item{family}{A family object  representing one of the built-in families.
Families supported here are the same as in \pkg{PGEE}, e.g, \code{binomial},
\code{gaussian}, \code{gamma} and \code{poisson}, and the corresponding
link functions are supported, e.g, \code{identity}, and \code{probit}.}

\item{lambda}{A user supplied value for the penalization parameter.}

\item{corstr}{A character string that indicates the correlation structure among
the repeated measurements of a subject. Structures supported in \code{LassoGEE}
are "AR1", "exchangeable", "unstructured", and "independence". The default
\code{corstr} type is "independence".}

\item{method}{The algorithms that are available. \code{"CGD"} represents the
I-CGD algorithm, and \code{"RWL"} represents re-weighted least square algorithm.}

\item{beta.ini}{User specified initial values for regression parameters.
The default value is \code{NULL}.}

\item{R}{User specified correlation matrix. The default value is \code{NULL}.}

\item{scale.fix}{A logical variable. The default value is \code{TRUE},
then the value of the scale parameter is fixed to \code{scale.value}.}

\item{scale.value}{If  \code{scale.fix = TRUE}, a numeric value will be assigned
to the fixed scale parameter. The default value is 1.}

\item{maxiter}{The maximum number of iterations used in the algorithm.
The default value is 50.}

\item{tol}{The tolerance level used in the algorithm. The default value is \code{1e-3}.}

\item{silent}{A logical variable; if false, the iteration counts
at each iteration of CGD are printed. The default value is TRUE.}

\item{Mv}{If either "stat_M_dep", or "non_stat_M_dep" is specified in corstr,
then this assigns a numeric value for Mv. Otherwise, the default value is NULL.}

\item{verbose}{A logical variable; Print the out loop iteration counts. The default value is TRUE.}
}
\value{
A list containing the following components:
  \item{betaest}{return final estimation}
  \item{beta_all_step}{return estimate in each iteration}
  \item{inner.count}{iterative count in each stage}
  \item{outer.iter}{iterate number of outer loop}
}
\description{
This function fits a \eqn{L_1} penalized GEE model to longitudinal
 data by I-CGD algorithm or re-weighted least square algorithm.
}
\examples{
# required R package
library(mvtnorm)
library(SimCorMultRes)
#
set.seed(123)
p <- 200
s <- ceiling(p^{1/3})
n <- ceiling(10 * s * log(p))
m <- 4
# covariance matrix of p number of continuous covariates
X.sigma <- matrix(0, p, p)
{
  for (i in 1:p)
    X.sigma[i,] <- 0.5^(abs((1:p)-i))
}

# generate matrix of covariates
X <- as.matrix(rmvnorm(n*m, mean = rep(0,p), X.sigma))

# true regression parameter associated with the covariate
bt <- runif(s, 0.05, 0.5) # = rep(1/s,s)
beta.true <- c(bt,rep(0,p-s))
# intercept
beta_intercepts <- 0
# unstructure
tt <- runif(m*m,-1,1)
Rtmp <- t(matrix(tt, m,m))\%*\%matrix(tt, m,m)+diag(1,4)
R_tr <- diag(diag(Rtmp)^{-1/2})\%*\%Rtmp\%*\%diag(diag(Rtmp)^{-1/2})
diag(R_tr) = round(diag(R_tr))

# library(SimCorMultRes)
# simulation of clustered binary responses
simulated_binary_dataset <- rbin(clsize = m, intercepts = beta_intercepts,
                                 betas = beta.true, xformula = ~X, cor.matrix = R_tr,
                                 link = "probit")
lambda <- 0.2* s *sqrt(log(p)/n)
data = simulated_binary_dataset$simdata
y = data$y
X = data$X
id = data$id

ptm <- proc.time()
nCGDfit = LassoGEE(X = X, y = y, id = id, family = binomial("probit"),
                 lambda = lambda, corstr = "unstructured")
proc.time() - ptm
betaest <- nCGDfit$betaest

}
\references{
Li, Y., Gao, X., and Xu, W. (2020). Statistical consistency for
generalized estimating equation with \eqn{L_1} regularization.
}
\seealso{
cv.LassoGEE
}
