# This file was generated by Rcpp::compileAttributes
# Generator token: 10BE3573-1514-4C36-9D1C-5A225CD40393

#' Calculate the Drift and Diffusion of one-dimensional stochastic processes
#'
#' \code{Langevin1D} calculates the Drift and Diffusion vectors (with errors)
#' for a given time series.
#'
#'
#' @param data a vector containing the time series.
#' @param bins a scalar denoting the number of \code{bins} to calculate the
#' conditional moments on.
#' @param steps a vector giving the \eqn{\tau} steps to calculate the
#' conditional moments (in samples (=\eqn{\tau * sf})).
#' @param sf a scalar denoting the sampling frequency.
#' @param bin_min a scalar denoting the minimal number of events per \code{bin}.
#' Defaults to \code{100}.
#' @param reqThreads a scalar denoting how many threads to use. Defaults to
#' \code{-1} which means all available cores.
#'
#' @return \code{Langevin1D} returns a list with thirteen components:
#' @return \item{D1}{a vector of the Drift coefficient for each \code{bin}.}
#' @return \item{eD1}{a vector of the error of the Drift coefficient for each
#' \code{bin}.}
#' @return \item{D2}{a vector of the Diffusion coefficient for each \code{bin}.}
#' @return \item{eD2}{a vector of the error of the Driffusion coefficient for
#' each \code{bin}.}
#' @return \item{D4}{a vector of the fourth Kramers-Moyal coefficient for each
#' \code{bin}.}
#' @return \item{mean_bin}{a vector of the mean value per \code{bin}.}
#' @return \item{density}{a vector of the number of events per \code{bin}.}
#' @return \item{M1}{a matrix of the first conditional moment for each
#' \eqn{\tau}. Rows corespond to \code{bin}, columns to \eqn{\tau}.}
#' @return \item{eM1}{a matrix of the error of the first conditional moment
#' for each \eqn{\tau}. Rows corespond to \code{bin}, columns to \eqn{\tau}.}
#' @return \item{M2}{a matrix of the second conditional moment for each
#' \eqn{\tau}. Rows corespond to \code{bin}, columns to \eqn{\tau}.}
#' @return \item{eM2}{a matrix of the error of the second conditional moment
#' for each \eqn{\tau}. Rows corespond to \code{bin}, columns to \eqn{\tau}.}
#' @return \item{M4}{a matrix of the fourth conditional moment for each
#' \eqn{\tau}. Rows corespond to \code{bin}, columns to \eqn{\tau}.}
#' @return \item{U}{a vector of the \code{bin} borders.}
#'
#' @author Philip Rinn
#' @seealso \code{\link{Langevin2D}}
#' @examples
#'
#' # Set number of bins, steps and the sampling frequency
#' bins <- 20;
#' steps <- c(1:5);
#' sf <- 1000;
#'
#' #### Linear drift, constant diffusion
#'
#' # Generate a time series with linear D^1 = -x and constant D^2 = 1
#' x <- timeseries1D(N=1e6, d11=-1, d20=1, sf=sf);
#' # Do the analysis
#' est <- Langevin1D(x, bins, steps, sf, reqThreads=2);
#' # Plot the result and add the theoretical expectation as red line
#' plot(est$mean_bin, est$D1);
#' lines(est$mean_bin, -est$mean_bin, col='red');
#' plot(est$mean_bin, est$D2);
#' abline(h=1, col='red');
#'
#' #### Cubic drift, constant diffusion
#'
#' # Generate a time series with cubic D^1 = x - x^3 and constant D^2 = 1
#' x <- timeseries1D(N=1e6, d13=-1, d11=1, d20=1, sf=sf);
#' # Do the analysis
#' est <- Langevin1D(x, bins, steps, sf, reqThreads=2);
#' # Plot the result and add the theoretical expectation as red line
#' plot(est$mean_bin, est$D1);
#' lines(est$mean_bin, est$mean_bin - est$mean_bin^3, col='red');
#' plot(est$mean_bin, est$D2);
#' abline(h=1, col='red');
#' @import Rcpp
#' @useDynLib Langevin
#' @export
Langevin1D <- function(data, bins, steps, sf = 1, bin_min = 100L, reqThreads = -1L) {
    .Call('Langevin_Langevin1D', PACKAGE = 'Langevin', data, bins, steps, sf, bin_min, reqThreads)
}

#' Calculate the Drift and Diffusion of two-dimensional stochastic processes
#'
#' \code{Langevin2D} calculates the Drift (with error) and Diffusion matrices
#' for given time series.
#'
#'
#' @param data a matrix containing the time series as rows.
#' @param bins a scalar denoting the number of \code{bins} to calculate Drift
#' and Diffusion on.
#' @param steps a vector giving the \eqn{\tau} steps to calculate the moments
#' (in samples).
#' @param sf a scalar denoting the sampling frequency.
#' @param bin_min a scalar denoting the minimal number of events per \code{bin}.
#' Defaults to \code{100}.
#' @param reqThreads a scalar denoting how many threads to use. Defaults to
#' \code{-1} which means all available cores.
#'
#' @return \code{Langevin2D} returns a list with nine components:
#' @return \item{D1}{a tensor with all values of the drift coefficient.
#' Dimension is \code{bins} x \code{bins} x 2. The first
#' \code{bins} x \code{bins} elements define the drift \eqn{D^{(1)}_{1}}
#' for the first variable and the rest define the drift \eqn{D^{(1)}_{2}}
#' for the second variable.}
#' @return \item{eD1}{a tensor with all estimated errors of the drift
#' coefficient. Dimension is \code{bins} x \code{bins} x 2. Same expression as
#' above.}
#' @return \item{D2}{a tensor with all values of the diffusion coefficient.
#' Dimension is \code{bins} x \code{bins} x 3. The first
#' \code{bins} x \code{bins} elements define the diffusion \eqn{D^{(2)}_{11}},
#' the second \code{bins} x \code{bins} elements define the diffusion
#' \eqn{D^{(2)}_{22}} and the rest define the diffusion
#' \eqn{D^{(2)}_{12} = D^{(2)}_{21}}.}
#' @return \item{eD2}{a tensor with all estimated errors of the driffusion
#' coefficient. Dimension is \code{bins} x \code{bins} x 3. Same expression as
#' above.}
#' @return \item{mean_bin}{a matrix of the mean value per \code{bin}.
#' Dimension is \code{bins} x \code{bins} x 2. The first
#' \code{bins} x \code{bins} elements define the mean for the first variable
#' and the rest for the second variable.}
#' @return \item{density}{a matrix of the number of events per \code{bin}.
#' Rows label the \code{bin} of the first variable and columns the second
#' variable.}
#' @return \item{M1}{a tensor of the first moment for each \code{bin} (line
#' label) and  each \eqn{\tau} step (column label). Dimension is
#' \code{bins} x \code{bins} x 2\code{length(steps)}.}
#' @return \item{eM1}{a tensor of the standard deviation of the first
#' moment for each bin (line label) and  each \eqn{\tau} step (column label).
#' Dimension is \code{bins} x \code{bins} x 2\code{length(steps)}.}
#' @return \item{M2}{a tensor of the second moment for each bin (line
#' label) and  each \eqn{\tau} step (column label). Dimension is
#' \code{bins} x \code{bins} x 3\code{length(steps)}.}
#' @return \item{U}{a matrix of the \code{bin} borders}
#'
#' @author Philip Rinn
#' @seealso \code{\link{Langevin1D}}
#' @import Rcpp
#' @useDynLib Langevin
#' @export
Langevin2D <- function(data, bins, steps, sf = 1, bin_min = 100L, reqThreads = -1L) {
    .Call('Langevin_Langevin2D', PACKAGE = 'Langevin', data, bins, steps, sf, bin_min, reqThreads)
}

#' Generate a 1D Langevin process
#'
#' \code{timeseries1D} generates a one-dimensional Langevin process using a
#' simple Euler integration. The drift function is a cubic polynomial, the
#' diffusion funcation a quadratic.
#'
#'
#' @param N a scalar denoting the length of the time series to generate.
#' @param startpoint a scalar denoting the starting point of the time series.
#' @param d13,d12,d11,d10 scalars denoting the coefficients for the drift polynomial.
#' @param d22,d21,d20 scalars denoting the coefficients for the diffusion polynomial.
#' @param sf a scalar denoting the sampling frequency.
#' @param dt a scalar denoting the maximal time step of integration. Default
#' \code{dt=0} yields \code{dt=1/sf}.
#'
#' @return \code{timeseries1D} returns a vector of length \code{N} with the
#' generated time series.
#'
#' @author Philip Rinn
#' @seealso \code{\link{timeseries2D}}
#' @examples
#' # Generate standardized Ornstein-Uhlenbeck-Process (d11=-1, d20=1)
#' # with integration time step 0.01 and sampling frequency 1
#' s <- timeseries1D(N=1e4, sf=1, dt=0.01);
#' t <- 1:1e4;
#' plot(t, s, t="l", main=paste("mean:", mean(s), " var:", var(s)));
#' @import Rcpp
#' @useDynLib Langevin
#' @export
timeseries1D <- function(N, startpoint = 0, d13 = 0, d12 = 0, d11 = -1, d10 = 0, d22 = 0, d21 = 0, d20 = 1, sf = 1000, dt = 0) {
    .Call('Langevin_timeseries1D', PACKAGE = 'Langevin', N, startpoint, d13, d12, d11, d10, d22, d21, d20, sf, dt)
}

.timeseries2D <- function(N, startpointx, startpointy, D1_1, D1_2, g_11, g_12, g_21, g_22, sf, dt) {
    .Call('Langevin_timeseries2D', PACKAGE = 'Langevin', N, startpointx, startpointy, D1_1, D1_2, g_11, g_12, g_21, g_22, sf, dt)
}

