% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Utilities.R
\name{pick_MappedGeneSymbol}
\alias{pick_MappedGeneSymbol}
\title{pick_MappedGeneSymbol}
\usage{
pick_MappedGeneSymbol(.from, .to)
}
\arguments{
\item{.from}{Character vector containing concatenated, triple-slash delimited gene symbols/identifiers
(e.g. "LOC101055758///LOC100041903///Gm2666///Gm7609///Csprs")}

\item{.to}{Character vector containing gene symbols to be matched (e.g. "Gm2666")}
}
\value{
A vector containing the matched symbols.
}
\description{
Function for matching values in \code{.from} vector derived from \code{Gene symbol} field
from GEO feature data (e.g. "LOC101055758///LOC100041903///Gm2666///Gm7609///Csprs") with the first
match in \code{.to} vector. The point of this is for a given differentially expressed feature, match
the corresponding gene symbols to gene symbols present in a gene set collection. This (hopefully)
leads to mapping more features in a GEO dataset to more gene symbols in a gene set collection to be
searched. Symbol matches are done in a case independent way, and the value returned is the value in
the .to vector (with its particular capitalization), such that pathways analysis can be easily performed.
}
\examples{
library(GSNA)
# These gene symbols correspond to the `Gene Symbol` field from a GEO dataset:
gene_symbols.from <- c( "BNS///CSMH///DDS1///THC8///BKRNS///BRWS1///PS1TP5BP1///ACTB",
                        "IP3R///IP3R1///ITPR1",
                        "FOS///p55///AP-1///C-FOS",
                        "MYC///LMYC///MYCL1///bHLHe38///L-Myc///v-myc"
                        )

# Extract unique genes from the \code{Bai_gsc.tmod} gene set:
gene_symbols.to <- unique( unlist( tmod2gsc( Bai_gsc.tmod ) ) )

mapped_symbols <- pick_MappedGeneSymbol( .from = gene_symbols.from,
                                         .to = gene_symbols.to )

# mapped_symbols returns: "ACTB", "ITPR1", "FOS", "MYC"

\donttest{
 # This example requires a web-based download of a GEO data set
 # and takes > 20 seconds to run on some platforms.

 # This function is particularly useful with when mapping
 # the \code{`Gene symbol`} field of GEO feature data to
 # gene symbols in a GSC:

 library(GSNA)
 library(GEOquery)
 library(tmod)

 gset <- getGEO("GSE75203", GSEMatrix =TRUE, AnnotGPL=TRUE)
 GSE75203.fdata <- fData(gset$GSE75203_series_matrix.txt.gz)

 # We can match the gene gene symbols in GSE75203.fdata with
 # those in the provided Bai_gsc.tmod object, and add the
 # mapped gene symbol to a new column in GSE75203.fdata,
 # 'MappedGeneSymbol':
 GSE75203.fdata$MappedGeneSymbol <-
   pick_MappedGeneSymbol( .from = GSE75203.fdata$`Gene symbol`,
                          .to = Bai_gsc.tmod$GENES$ID )
 # NOTE, if you were using a tmodGS object, the above
 # would be this instead:
 # GSE75203.fdata$MappedGeneSymbol <-
 #   pick_MappedGeneSymbol( .from = GSE75203.fdata$`Gene symbol`,
 #                          .to = Bai_gsc.tmodGS$gv )
}


}
