% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.r
\name{calcAdjFrag}
\alias{calcAdjFrag}
\title{Calculate adjacency matrices and fragility matrix from iEEG recording}
\source{
Recreation of the method described in
Li A, Huynh C, Fitzgerald Z, Cajigas I, Brusko D, Jagid J, et al.
Neural fragility as an EEG marker of the seizure onset zone.
Nat Neurosci. 2021 Oct;24(10):1465–74
(\href{https://pubmed.ncbi.nlm.nih.gov/34354282/}{pubmed}).
We have found solutions to fill up missing details in the paper method description
}
\usage{
calcAdjFrag(
  epoch,
  window,
  step,
  lambda = NULL,
  nSearch = 100L,
  progress = FALSE,
  parallel = FALSE
)
}
\arguments{
\item{epoch}{Matrix or Epoch object. iEEG data matrix or Epoch object. If matrix, the row names are the electrode names and the column names are the time points}

\item{window}{Integer. The number of time points to use in each window}

\item{step}{Integer. The number of time points to move the window each time}

\item{lambda}{Numeric. The lambda value for regularization to use in the ridge regression.
If NULL, the lambda will be chosen automatically
ensuring that ensuring that the adjacent matrix is stable (see details)}

\item{nSearch}{Integer. Number of instable eigenvalues with norm=1 to search for the minimum norm perturbation. This parameter is used only when the lambda is NULL}

\item{progress}{Logical. If TRUE, print progress information. If \code{parallel} is TRUE, this option only support the \code{doSNOW} backend.}

\item{parallel}{Logical. If TRUE, use parallel computing.
Users must register a parallel backend with the foreach package}
}
\value{
A Fragility object
}
\description{
The function calculates the neural fragility column
from an adjacency matrix in each time window
}
\details{
1/ For each time window i, a discrete stable Linear time system
(adjacency matrix) is computed named \eqn{A_i}
such that
\eqn{A_i x(t) = x(t+1)}. The 'lambda' option is the regularization parameter
for the ridge regression.
\code{lambda=NULL}(default) will find a lambda value that ensures
the stability of the estimated \eqn{A_i}.

2/For each stable estimated \eqn{A_i}, the minimum norm perturbation \eqn{\Gamma_{ik}} (k index of the electrodes)
for column perturbation is computed.
Each column is normalized \eqn{\frac{max(\Gamma_{i})-\Gamma_{ik}}{max(\Gamma_i)}}
}
\examples{
## A dummy example with 5 electrodes and 20 time points
data <- matrix(rnorm(100), nrow = 5)
## create an Epoch object
epoch <- Epoch(data)
windowNum <- 10
step <- 5
lambda <- 0.1
calcAdjFrag(
    epoch = epoch, window = windowNum,
    step = step, lambda = lambda, progress = TRUE
)

## A more realistic example with parallel computing
\donttest{
if (requireNamespace("doSNOW")) {
    ## Register a SNOW backend with 4 workers
    library(parallel)
    library(doSNOW)
    cl <- makeCluster(4, type = "SOCK")
    registerDoSNOW(cl)

    data("pt01EcoG")
    epoch <- Epoch(pt01EcoG)
    window <- 250
    step <- 125
    title <- "PT01 seizure 1"
    calcAdjFrag(
        epoch = epoch, window = window,
        step = step, parallel = TRUE, progress = TRUE
    )

    ## stop the parallel backend
    stopCluster(cl)
}
}

}
