% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootEGA.R
\name{bootEGA}
\alias{bootEGA}
\title{Bootstrap Exploratory Graph Analysis}
\usage{
bootEGA(
  data,
  n = NULL,
  corr = c("auto", "cor_auto", "cosine", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  model = c("BGGM", "glasso", "TMFG"),
  algorithm = c("leiden", "louvain", "walktrap"),
  uni.method = c("expand", "LE", "louvain"),
  iter = 500,
  type = c("parametric", "resampling"),
  ncores,
  EGA.type = c("EGA", "EGA.fit", "hierEGA", "riEGA"),
  plot.itemStability = TRUE,
  typicalStructure = FALSE,
  plot.typicalStructure = FALSE,
  seed = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis}

\item{n}{Numeric (length = 1).
Sample size if \code{data} provided is a correlation matrix}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"cosine"} --- Uses \code{\link[EGAnet]{cosine}} to compute cosine similarity

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{model}{Character (length = 1).
Defaults to \code{"glasso"}.
Available options:

\itemize{

\item \code{"BGGM"} --- Computes the Bayesian Gaussian Graphical Model.
Set argument \code{ordinal.categories} to determine
levels allowed for a variable to be considered ordinal.
See \code{?BGGM::estimate} for more details

\item \code{"glasso"} --- Computes the GLASSO with EBIC model selection.
See \code{\link[EGAnet]{EBICglasso.qgraph}} for more details

\item \code{"TMFG"} --- Computes the TMFG method.
See \code{\link[EGAnet]{TMFG}} for more details

}}

\item{algorithm}{Character or
\code{igraph} \code{cluster_*} function (length = 1).
Defaults to \code{"walktrap"}.
Three options are listed below but all are available
(see \code{\link[EGAnet]{community.detection}} for other options):

\itemize{

\item \code{"leiden"} --- See \code{\link[igraph]{cluster_leiden}} for more details

\item \code{"louvain"} --- By default, \code{"louvain"} will implement the Louvain algorithm using
the consensus clustering method (see \code{\link[EGAnet]{community.consensus}}
for more information). This function will implement
\code{consensus.method = "most_common"} and \code{consensus.iter = 1000}
unless specified otherwise

\item \code{"walktrap"} --- See \code{\link[igraph]{cluster_walktrap}} for more details

}}

\item{uni.method}{Character (length = 1).
What unidimensionality method should be used?
Defaults to \code{"louvain"}.
Available options:

\itemize{

\item \code{"expand"} --- Expands the correlation matrix with four variables correlated 0.50.
If number of dimension returns 2 or less in check, then the data
are unidimensional; otherwise, regular EGA with no matrix
expansion is used. This method was used in the Golino et al.'s (2020)
\emph{Psychological Methods} simulation

\item \code{"LE"} --- Applies the Leading Eigenvector algorithm
(\code{\link[igraph]{cluster_leading_eigen}})
on the empirical correlation matrix. If the number of dimensions is 1,
then the Leading Eigenvector solution is used; otherwise, regular EGA
is used. This method was used in the Christensen et al.'s (2023)
\emph{Behavior Research Methods} simulation

\item \code{"louvain"} --- Applies the Louvain algorithm (\code{\link[igraph]{cluster_louvain}})
on the empirical correlation matrix. If the number of dimensions is 1,
then the Louvain solution is used; otherwise, regular EGA is used.
This method was validated Christensen's (2022) \emph{PsyArXiv} simulation.
Consensus clustering can be used by specifying either
\code{"consensus.method"} or \code{"consensus.iter"}

}}

\item{iter}{Numeric (length = 1).
Number of replica samples to generate from the bootstrap analysis.
Defaults to \code{500} (recommended)}

\item{type}{Character (length = 1).
What type of bootstrap should be performed?
Defaults to \code{"parametric"}.
Available options:

\itemize{

\item \code{"parametric"} --- Generates \code{iter} new datasets from
(multivariate normal random distributions) based on the
original dataset using \code{\link[MASS]{mvrnorm}}

\item \code{"resampling"} --- Generates \code{iter} new datasets from random subsamples
of the original data

}}

\item{ncores}{Numeric (length = 1).
Number of cores to use in computing results.
Defaults to \code{ceiling(parallel::detectCores() / 2)} or half of your
computer's processing power.
Set to \code{1} to not use parallel computing

If you're unsure how many cores your computer has,
then type: \code{parallel::detectCores()}}

\item{EGA.type}{Character (length = 1).
Type of EGA model to use.
Defaults to \code{"EGA"}
Available options:

\itemize{

\item \code{"\link[EGAnet]{EGA}"} --- Uses standard exploratory graph analysis

\item \code{"\link[EGAnet]{EGA.fit}"} --- Uses \code{\link[EGAnet]{tefi}} to determine best fit of
\code{\link[EGAnet]{EGA}}

\item \code{"\link[EGAnet]{hierEGA}"} --- Uses hierarchical exploratory graph analysis

\item \code{"\link[EGAnet]{riEGA}"} --- Uses random-intercept exploratory graph analysis

}

Arguments for \code{EGA.type} can be added (see links
for details on specific function arguments)}

\item{plot.itemStability}{Boolean (length = 1).
Should the plot be produced for \code{item.replication}?
Defaults to \code{TRUE}}

\item{typicalStructure}{Boolean (length = 1).
If \code{TRUE}, returns the median (\code{"glasso"} or \code{"BGGM"}) or
mean (\code{"TMFG"}) network structure and estimates its dimensions
(see \strong{Details} for more information).
Defaults to \code{FALSE}}

\item{plot.typicalStructure}{Boolean (length = 1).
If \code{TRUE}, returns a plot of the typical network structure.
Defaults to \code{FALSE}}

\item{seed}{Numeric (length = 1).
Defaults to \code{NULL} or random results.
Set for reproducible results.
See \href{https://r-ega.net/articles/reproducibility-prng.html}{Reproducibility and PRNG}
for more details on random number generation in \code{EGAnet}}

\item{verbose}{Boolean (length = 1).
Should progress be displayed?
Defaults to \code{TRUE}.
Set to \code{FALSE} to not display progress}

\item{...}{Additional arguments that can be passed on to
\code{\link[EGAnet]{auto.correlate}},
\code{\link[EGAnet]{network.estimation}},
\code{\link[EGAnet]{community.detection}},
\code{\link[EGAnet]{community.consensus}},
\code{\link[EGAnet]{EGA}},
\code{\link[EGAnet]{EGA.fit}},
\code{\link[EGAnet]{hierEGA}}, and
\code{\link[EGAnet]{riEGA}}}
}
\value{
Returns a list containing:

\item{iter}{Number of replica samples in bootstrap}

\item{bootGraphs}{A list containing the networks of each replica sample}

\item{boot.wc}{A matrix of membership assignments for each replica network
with variables down the columns and replicas across the rows}

\item{boot.ndim}{Number of dimensions identified in each replica sample}

\item{summary.table}{A data frame containing number of replica samples,
median, standard deviation, standard error, 95\% confidence intervals, and
quantiles (lower = 2.5\% and upper = 97.5\%)}

\item{frequency}{A data frame containing the proportion of times the number of dimensions was identified
(e.g., .85 of 1,000 = 850 times that specific number of dimensions was found)}

\item{TEFI}{\code{\link[EGAnet]{tefi}} value for each replica sample}

\item{type}{Type of bootstrap used}

\item{EGA}{Output of the empirical EGA results
(output will vary based on \code{EGA.type})}

\item{EGA.type}{Type of \code{*EGA} function used}

\item{typicalGraph}{A list containing:

\itemize{

\item \code{graph} --- Network matrix of the median network structure

\item \code{typical.dim.variables} --- An ordered matrix of item allocation

\item \code{wc} --- Membership assignments of the median network

}

}

\item{plot.typical.ega}{Plot output if \code{plot.typicalStructure = TRUE}}
}
\description{
\code{bootEGA} Estimates the number of dimensions of \code{iter} bootstraps
using the empirical zero-order correlation matrix (\code{"parametric"}) or
\code{"resampling"} from the empirical dataset (non-parametric). \code{bootEGA}
estimates a typical median network structure, which is formed by the median or
mean pairwise (partial) correlations over the \emph{iter} bootstraps (see
\strong{Details} for information about the typical median network structure).
}
\details{
The typical network structure is derived from the median (or mean) value
of each pairwise relationship. These values tend to reflect the
"typical" value taken by an edge across the bootstrap networks. Afterward,
the same community detection algorithm is applied to the typical network as the
bootstrap networks.

Because the community detection algorithm is applied to the typical network structure,
there is a possibility that the community algorithm determines
a different number of dimensions than the median number derived from the bootstraps.
The typical network structure (and number of dimensions) may \emph{not}
match the empirical \code{\link[EGAnet]{EGA}} number of dimensions or
the median number of dimensions from the bootstrap. This result is known
and \emph{not} a bug.
}
\examples{
# Load data
wmt <- wmt2[,7:24]

\dontrun{
# Standard EGA parametric example
boot.wmt <- bootEGA(
  data = wmt, iter = 500,
  type = "parametric", ncores = 2
)

# Standard resampling example
boot.wmt <- bootEGA(
  data = wmt, iter = 500,
  type = "resampling", ncores = 2
)

# Example using {igraph} `cluster_*` function
boot.wmt.spinglass <- bootEGA(
  data = wmt, iter = 500,
  algorithm = igraph::cluster_spinglass,
  # use any function from {igraph}
  type = "parametric", ncores = 2
)

# EGA fit example
boot.wmt.fit <- bootEGA(
  data = wmt, iter = 500,
  EGA.type = "EGA.fit",
  type = "parametric", ncores = 2
)

# Hierarchical EGA example
boot.wmt.hier <- bootEGA(
  data = wmt, iter = 500,
  EGA.type = "hierEGA",
  type = "parametric", ncores = 2
)

# Random-intercept EGA example
boot.wmt.ri <- bootEGA(
  data = wmt, iter = 500,
  EGA.type = "riEGA",
  type = "parametric", ncores = 2
)}

}
\references{
\strong{Original implementation of bootEGA} \cr
Christensen, A. P., & Golino, H. (2021).
Estimating the stability of the number of factors via Bootstrap Exploratory Graph Analysis: A tutorial.
\emph{Psych}, \emph{3}(3), 479-500.
}
\seealso{
\code{\link[EGAnet]{itemStability}} to estimate the stability of
the variables in the empirical dimensions and
\code{\link[EGAnet]{dimensionStability}} to estimate the stability of
the dimensions (structural consistency)
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
