% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/StatModels.R
\name{EmpiricalRiskMinimizationDP.CMS}
\alias{EmpiricalRiskMinimizationDP.CMS}
\title{Privacy-preserving Empirical Risk Minimization for Binary Classification}
\description{
This class implements differentially private empirical risk
minimization \insertCite{chaudhuri2011}{DPpack}. Either the output or the
objective perturbation method can be used. It is intended to be a framework
for building more specific models via inheritance. See
\code{\link{LogisticRegressionDP}} for an example of this type of
structure.
}
\details{
To use this class for empirical risk minimization, first use the
\code{new} method to construct an object of this class with the desired
function values and hyperparameters. After constructing the object, the
\code{fit} method can be applied with a provided dataset and data bounds to
fit the model.  In fitting, the model stores a vector of coefficients
\code{coeff} which satisfy differential privacy. These can be released
directly, or used in conjunction with the \code{predict} method to
privately predict the outcomes of new datapoints.

Note that in order to guarantee differential privacy for empirical risk
minimization, certain constraints must be satisfied for the values used to
construct the object, as well as for the data used to fit. These conditions
depend on the chosen perturbation method. Specifically, the provided loss
function must be convex and differentiable with respect to \code{y.hat},
and the absolute value of the first derivative of the loss function must be
at most 1. If objective perturbation is chosen, the loss function must also
be doubly differentiable and the absolute value of the second derivative of
the loss function must be bounded above by a constant c for all possible
values of \code{y.hat} and \code{y}, where \code{y.hat} is the predicted
label and \code{y} is the true label. The regularizer must be 1-strongly
convex and differentiable. It also must be doubly differentiable if
objective perturbation is chosen. Finally, it is assumed that if x
represents a single row of the dataset X, then the l2-norm of x is at most
1 for all x. Note that because of this, a bias term cannot be included
without appropriate scaling/preprocessing of the dataset. To ensure
privacy, the add.bias argument in the \code{fit} and \code{predict} methods
should only be utilized in subclasses within this package where appropriate
preprocessing is implemented, not in this class.
}
\examples{
# Build train dataset X and y, and test dataset Xtest and ytest
N <- 200
K <- 2
X <- data.frame()
y <- data.frame()
for (j in (1:K)){
  t <- seq(-.25, .25, length.out = N)
  if (j==1) m <- stats::rnorm(N,-.2, .1)
  if (j==2) m <- stats::rnorm(N, .2, .1)
  Xtemp <- data.frame(x1 = 3*t , x2 = m - t)
  ytemp <- data.frame(matrix(j-1, N, 1))
  X <- rbind(X, Xtemp)
  y <- rbind(y, ytemp)
}
Xtest <- X[seq(1,(N*K),10),]
ytest <- y[seq(1,(N*K),10),,drop=FALSE]
X <- X[-seq(1,(N*K),10),]
y <- y[-seq(1,(N*K),10),,drop=FALSE]

# Construct object for logistic regression
mapXy <- function(X, coeff) e1071::sigmoid(X\%*\%coeff)
# Cross entropy loss
loss <- function(y.hat,y) -(y*log(y.hat) + (1-y)*log(1-y.hat))
regularizer <- 'l2' # Alternatively, function(coeff) coeff\%*\%coeff/2
eps <- 1
gamma <- 1
perturbation.method <- 'objective'
c <- 1/4 # Required value for logistic regression
mapXy.gr <- function(X, coeff) as.numeric(e1071::dsigmoid(X\%*\%coeff))*t(X)
loss.gr <- function(y.hat, y) -y/y.hat + (1-y)/(1-y.hat)
regularizer.gr <- function(coeff) coeff
ermdp <- EmpiricalRiskMinimizationDP.CMS$new(mapXy, loss, regularizer, eps,
                                             gamma, perturbation.method, c,
                                             mapXy.gr, loss.gr,
                                             regularizer.gr)

# Fit with data
# Bounds for X based on construction
upper.bounds <- c( 1, 1)
lower.bounds <- c(-1,-1)
ermdp$fit(X, y, upper.bounds, lower.bounds) # No bias term
ermdp$coeff # Gets private coefficients

# Predict new data points
predicted.y <- ermdp$predict(Xtest)
n.errors <- sum(round(predicted.y)!=ytest)

}
\references{
\insertRef{chaudhuri2011}{DPpack}
}
\keyword{internal}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{mapXy}}{Map function of the form \code{mapXy(X, coeff)} mapping input
data matrix \code{X} and coefficient vector or matrix \code{coeff} to
output labels \code{y}.}

\item{\code{mapXy.gr}}{Function representing the gradient of the map function with
respect to the values in \code{coeff} and of the form \code{mapXy.gr(X,
  coeff)}, where \code{X} is a matrix and \code{coeff} is a matrix or
numeric vector.}

\item{\code{loss}}{Loss function of the form \code{loss(y.hat, y)}, where
\code{y.hat} and \code{y} are matrices.}

\item{\code{loss.gr}}{Function representing the gradient of the loss function with
respect to \code{y.hat} and of the form \code{loss.gr(y.hat, y)}, where
\code{y.hat} and \code{y} are matrices.}

\item{\code{regularizer}}{Regularization function of the form
\code{regularizer(coeff)}, where \code{coeff} is a vector or matrix.}

\item{\code{regularizer.gr}}{Function representing the gradient of the
regularization function with respect to \code{coeff} and of the form
\code{regularizer.gr(coeff)}.}

\item{\code{gamma}}{Nonnegative real number representing the regularization
constant.}

\item{\code{eps}}{Positive real number defining the epsilon privacy budget. If set
to Inf, runs algorithm without differential privacy.}

\item{\code{perturbation.method}}{String indicating whether to use the 'output' or
the 'objective' perturbation methods \insertCite{chaudhuri2011}{DPpack}.}

\item{\code{c}}{Positive real number denoting the upper bound on the absolute
value of the second derivative of the loss function, as required to
ensure differential privacy for the objective perturbation method.}

\item{\code{coeff}}{Numeric vector of coefficients for the model.}

\item{\code{kernel}}{Value only used in child class \code{\link{svmDP}}. String
indicating which kernel to use for SVM. Must be one of {'linear',
'Gaussian'}. If 'linear' (default), linear SVM is used. If 'Gaussian,'
uses the sampling function corresponding to the Gaussian (radial) kernel
approximation.}

\item{\code{D}}{Value only used in child class \code{\link{svmDP}}. Nonnegative
integer indicating the dimensionality of the transform space
approximating the kernel. Higher values of \code{D} provide better kernel
approximations at a cost of computational efficiency.}

\item{\code{sampling}}{Value only used in child class \code{\link{svmDP}}.
Sampling function of the form \code{sampling(d)}, where \code{d} is the
input dimension, returning a (\code{d}+1)-dimensional vector of samples
corresponding to the Fourier transform of the kernel to be approximated.}

\item{\code{phi}}{Value only used in child class \code{\link{svmDP}}. Function of
the form \code{phi(x, theta)}, where \code{x} is an individual row of the
original dataset, and theta is a (\code{d}+1)-dimensional vector sampled
from the Fourier transform of the kernel to be approximated, where
\code{d} is the dimension of \code{x}. The function returns a numeric
scalar corresponding to the pre-filtered value at the given row with the
given sampled vector.}

\item{\code{kernel.param}}{Value only used in child class \code{\link{svmDP}}.
Positive real number corresponding to the Gaussian kernel parameter.}

\item{\code{prefilter}}{Value only used in child class \code{\link{svmDP}}. Matrix
of pre-filter values used in converting data into transform space.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-EmpiricalRiskMinimizationDP.CMS-new}{\code{EmpiricalRiskMinimizationDP.CMS$new()}}
\item \href{#method-EmpiricalRiskMinimizationDP.CMS-fit}{\code{EmpiricalRiskMinimizationDP.CMS$fit()}}
\item \href{#method-EmpiricalRiskMinimizationDP.CMS-predict}{\code{EmpiricalRiskMinimizationDP.CMS$predict()}}
\item \href{#method-EmpiricalRiskMinimizationDP.CMS-clone}{\code{EmpiricalRiskMinimizationDP.CMS$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EmpiricalRiskMinimizationDP.CMS-new"></a>}}
\if{latex}{\out{\hypertarget{method-EmpiricalRiskMinimizationDP.CMS-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{EmpiricalRiskMinimizationDP.CMS} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EmpiricalRiskMinimizationDP.CMS$new(
  mapXy,
  loss,
  regularizer,
  eps,
  gamma,
  perturbation.method = "objective",
  c = NULL,
  mapXy.gr = NULL,
  loss.gr = NULL,
  regularizer.gr = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mapXy}}{Map function of the form \code{mapXy(X, coeff)} mapping input
data matrix \code{X} and coefficient vector or matrix \code{coeff} to
output labels \code{y}. Should return a column matrix of predicted labels
for each row of \code{X}. See \code{\link{mapXy.sigmoid}} for an example.}

\item{\code{loss}}{Loss function of the form \code{loss(y.hat, y)}, where
\code{y.hat} and \code{y} are matrices. Should be defined such that it
returns a matrix of loss values for each element of \code{y.hat} and
\code{y}. See \code{\link{loss.cross.entropy}} for an example. It must be
convex and differentiable, and the absolute value of the first derivative
of the loss function must be at most 1. Additionally, if the objective
perturbation method is chosen, it must be doubly differentiable and the
absolute value of the second derivative of the loss function must be
bounded above by a constant c for all possible values of \code{y.hat} and
\code{y}.}

\item{\code{regularizer}}{String or regularization function. If a string, must be
'l2', indicating to use l2 regularization. If a function, must have form
\code{regularizer(coeff)}, where \code{coeff} is a vector or matrix, and
return the value of the regularizer at \code{coeff}. See
\code{\link{regularizer.l2}} for an example. Additionally, in order to
ensure differential privacy, the function must be 1-strongly convex and
differentiable. If the objective perturbation method is chosen, it must
also be doubly differentiable.}

\item{\code{eps}}{Positive real number defining the epsilon privacy budget. If set
to Inf, runs algorithm without differential privacy.}

\item{\code{gamma}}{Nonnegative real number representing the regularization
constant.}

\item{\code{perturbation.method}}{String indicating whether to use the 'output' or
the 'objective' perturbation methods \insertCite{chaudhuri2011}{DPpack}.
Defaults to 'objective'.}

\item{\code{c}}{Positive real number denoting the upper bound on the absolute
value of the second derivative of the loss function, as required to
ensure differential privacy for the objective perturbation method. This
input is unnecessary if perturbation.method is 'output', but is required
if perturbation.method is 'objective'. Defaults to NULL.}

\item{\code{mapXy.gr}}{Optional function representing the gradient of the map
function with respect to the values in \code{coeff}. If given, must be of
the form \code{mapXy.gr(X, coeff)}, where \code{X} is a matrix and
\code{coeff} is a matrix or numeric vector. Should be defined such that
the ith row of the output represents the gradient with respect to the ith
coefficient. See \code{\link{mapXy.gr.sigmoid}} for an example. If not
given, non-gradient based optimization methods are used to compute the
coefficient values in fitting the model.}

\item{\code{loss.gr}}{Optional function representing the gradient of the loss
function with respect to \code{y.hat} and of the form
\code{loss.gr(y.hat, y)}, where \code{y.hat} and \code{y} are matrices.
Should be defined such that the ith row of the output represents the
gradient of the loss function at the ith set of input values. See
\code{\link{loss.gr.cross.entropy}} for an example. If not given,
non-gradient based optimization methods are used to compute the
coefficient values in fitting the model.}

\item{\code{regularizer.gr}}{Optional function representing the gradient of the
regularization function with respect to \code{coeff} and of the form
\code{regularizer.gr(coeff)}. Should return a vector. See
\code{\link{regularizer.gr.l2}} for an example. If \code{regularizer} is
given as a string, this value is ignored. If not given and
\code{regularizer} is a function, non-gradient based optimization methods
are used to compute the coefficient values in fitting the model.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{EmpiricalRiskMinimizationDP.CMS} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EmpiricalRiskMinimizationDP.CMS-fit"></a>}}
\if{latex}{\out{\hypertarget{method-EmpiricalRiskMinimizationDP.CMS-fit}{}}}
\subsection{Method \code{fit()}}{
Fit the differentially private empirical risk minimization
model. This method runs either the output perturbation or the objective
perturbation algorithm \insertCite{chaudhuri2011}{DPpack}, depending on
the value of perturbation.method used to construct the object, to
generate an objective function. A numerical optimization method is then
run to find optimal coefficients for fitting the model given the training
data and hyperparameters. The built-in \code{\link{optim}} function using
the "BFGS" optimization method is used. If \code{mapXy.gr},
\code{loss.gr}, and \code{regularizer.gr} are all given in the
construction of the object, the gradient of the objective function is
utilized by \code{optim} as well. Otherwise, non-gradient based
optimization methods are used. The resulting privacy-preserving
coefficients are stored in \code{coeff}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EmpiricalRiskMinimizationDP.CMS$fit(
  X,
  y,
  upper.bounds,
  lower.bounds,
  add.bias = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Dataframe of data to be fit.}

\item{\code{y}}{Vector or matrix of true labels for each row of \code{X}.}

\item{\code{upper.bounds}}{Numeric vector of length \code{ncol(X)} giving upper
bounds on the values in each column of X. The \code{ncol(X)} values are
assumed to be in the same order as the corresponding columns of \code{X}.
Any value in the columns of \code{X} larger than the corresponding upper
bound is clipped at the bound.}

\item{\code{lower.bounds}}{Numeric vector of length \code{ncol(X)} giving lower
bounds on the values in each column of \code{X}. The \code{ncol(X)}
values are assumed to be in the same order as the corresponding columns
of \code{X}. Any value in the columns of \code{X} larger than the
corresponding upper bound is clipped at the bound.}

\item{\code{add.bias}}{Boolean indicating whether to add a bias term to \code{X}.
Defaults to FALSE.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EmpiricalRiskMinimizationDP.CMS-predict"></a>}}
\if{latex}{\out{\hypertarget{method-EmpiricalRiskMinimizationDP.CMS-predict}{}}}
\subsection{Method \code{predict()}}{
Predict label(s) for given \code{X} using the fitted
coefficients.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EmpiricalRiskMinimizationDP.CMS$predict(X, add.bias = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Dataframe of data on which to make predictions. Must be of same
form as \code{X} used to fit coefficients.}

\item{\code{add.bias}}{Boolean indicating whether to add a bias term to \code{X}.
Defaults to FALSE. If add.bias was set to TRUE when fitting the
coefficients, add.bias should be set to TRUE for predictions.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Matrix of predicted labels corresponding to each row of \code{X}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-EmpiricalRiskMinimizationDP.CMS-clone"></a>}}
\if{latex}{\out{\hypertarget{method-EmpiricalRiskMinimizationDP.CMS-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{EmpiricalRiskMinimizationDP.CMS$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
