% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ordinate_cladistic_matrix.R
\name{ordinate_cladistic_matrix}
\alias{ordinate_cladistic_matrix}
\title{Principal Coordinates on a Cladistic Matrix}
\usage{
ordinate_cladistic_matrix(
  cladistic_matrix,
  distance_metric = "mord",
  ged_type = "wills",
  distance_transformation = "arcsine_sqrt",
  distance_polymorphism_behaviour = "min_difference",
  distance_uncertainty_behaviour = "min_difference",
  distance_inapplicable_behaviour = "missing",
  character_dependencies = NULL,
  alpha = 0.5,
  correction = "cailliez",
  time_tree = NULL,
  estimate_all_nodes = FALSE,
  estimate_tip_values = FALSE,
  inapplicables_as_missing = FALSE,
  ancestral_polymorphism_behaviour = "equalp",
  ancestral_uncertainty_behaviour = "equalp",
  threshold = 0.01,
  all_missing_allowed = FALSE
)
}
\arguments{
\item{cladistic_matrix}{A character-taxon matrix in the format imported by \link{read_nexus_matrix}.}

\item{distance_metric}{See \link{calculate_morphological_distances}.}

\item{ged_type}{See \link{calculate_morphological_distances}.}

\item{distance_transformation}{See \link{calculate_morphological_distances}.}

\item{distance_polymorphism_behaviour}{See \link{calculate_morphological_distances}.}

\item{distance_uncertainty_behaviour}{See \link{calculate_morphological_distances}.}

\item{distance_inapplicable_behaviour}{See \link{calculate_morphological_distances}.}

\item{character_dependencies}{See \link{calculate_morphological_distances}.}

\item{alpha}{See \link{calculate_morphological_distances}.}

\item{correction}{The negative eigenvalue correction to use (one of "lingoes", "none", or "cailliez" - the default). See \link{pcoa} for more details.}

\item{time_tree}{If a phylmorphospace is desired then a tree with root age and branch-lengths must be included.}

\item{estimate_all_nodes}{See \link{estimate_ancestral_states}.}

\item{estimate_tip_values}{See \link{estimate_ancestral_states}.}

\item{inapplicables_as_missing}{See \link{estimate_ancestral_states}.}

\item{ancestral_polymorphism_behaviour}{See \link{estimate_ancestral_states}.}

\item{ancestral_uncertainty_behaviour}{See \link{estimate_ancestral_states}.}

\item{threshold}{See \link{estimate_ancestral_states}.}

\item{all_missing_allowed}{See \link{estimate_ancestral_states}.}
}
\value{
\item{time_tree}{The tree (if supplied). Note this may be pruned from the input tree by \link{trim_matrix}.}
\item{distance_matrix}{The distance matrix. Note this may be pruned by \link{trim_matrix} and thus not include all taxa.}
\item{removed_taxa}{A vector of taxa and/or nodes removed by \link{trim_matrix}. Returns NULL if none were removed.}
\item{note}{See \link{pcoa}.}
\item{values}{See \link{pcoa}.}
\item{vectors}{See \link{pcoa}. Note: this will be the same as \code{vectors.cor} from the \link{pcoa} output if a correction was applied.}
\item{trace}{See \link{pcoa}. Note: this will be the same as \code{trace.cor} from the \link{pcoa} output if a correction was applied.}
}
\description{
Performs Principal Coordinates Analysis (PCoA) on a cladistic matrix.
}
\details{
Takes a cladistic matrix in the format imported by \link{read_nexus_matrix} and performs Principal Coordinates (Gower 1966) analysis on it.

This function is effectively a wrapper for the pipeline:

\link{estimate_ancestral_states} -> \link{calculate_morphological_distances} -> \link{pcoa}

With the first part being optional (if wanting a phylomorphospace) and the latter coming from the \link{ape} package (the user is referred there for some of the options, e.g., using the Caillez 1983 approach to avoiding negative eigenvalues). (See Lloyd 2016 for more on disparity pipelines.)

If providing a tree and inferring ancestral states then options to also infer missing or uncertain tips and whether to infer values for all characters at all internal nodes are provided by the \link{estimate_ancestral_states} part.

Other options within the function concern the distance metric to use and the transformation to be used if selecting a propotional distance (see \link{calculate_morphological_distances}).

IMPORTANT: The function can remove taxa (or if including a tree, nodes as well) if they lead to an incomplete distance matrix (see \link{trim_matrix} for more details).
}
\examples{

# Run on Michaux (1989) data set with default settings:
x <- ordinate_cladistic_matrix(cladistic_matrix = michaux_1989)

# Show entire output:
x

# Generate a (made up) tree:
time_tree <- ape::rtree(n = length(x = rownames(x = michaux_1989$matrix_1$matrix)))

# Add taxon names to it:
time_tree$tip.label <- rownames(x = michaux_1989$matrix_1$matrix)

# Set root time by making youngest taxon extant:
time_tree$root.time <- max(diag(x = ape::vcv(phy = time_tree)))

# Run with tree:
y <- ordinate_cladistic_matrix(cladistic_matrix = michaux_1989, time_tree = time_tree)

# Show new output:
y
}
\references{
Cailliez, F., 1983. The analytical solution of the additive constant problem. \emph{Psychometrika}, \bold{48}, 305-308.

Gower, J. C., 1966. Some distance properties of latent root and vector methods used in multivariate analysis. \emph{Biometrika}, \bold{53}, 325-338.
}
\seealso{
\link{assign_taxa_to_bins}, \link{plot_chronophylomorphospace}, \link{plot_morphospace_stack}, \link{plot_morphospace}, \link{plot_multi_morphospace}
}
\author{
Graeme T. Lloyd \email{graemetlloyd@gmail.com}
}
