% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CST_RainFARM.R
\name{RainFARM}
\alias{RainFARM}
\title{RainFARM stochastic precipitation downscaling (reduced version)}
\usage{
RainFARM(data, lon, lat, nf, weights = 1, nens = 1, slope = 0, kmin = 1,
  fglob = FALSE, fsmooth = TRUE, time_dim = NULL, lon_dim = "lon",
  lat_dim = "lat", drop_realization_dim = FALSE, verbose = FALSE)
}
\arguments{
\item{data}{Precipitation array to downscale.
The input array is expected to have at least two dimensions named "lon" and "lat" by default
(these default names can be changed with the \code{lon_dim} and \code{lat_dim} parameters)
and one or more dimensions over which to average these slopes,
which can be specified by parameter \code{time_dim}.
The number of longitudes and latitudes in the input data is expected to be even and the same. If not
the function will perform a subsetting to ensure this condition.}

\item{lon}{Vector or array of longitudes.}

\item{lat}{Vector or array of latitudes.}

\item{nf}{Refinement factor for downscaling (the output resolution is increased by this factor).}

\item{weights}{Matrix with climatological weights which can be obtained using
the \code{CST_RFWeights} function. If \code{weights=1.} (default) no weights are used.
The matrix should have dimensions (lon, lat) in this order.
The names of these dimensions are not checked.}

\item{nens}{Number of ensemble members to produce (default: \code{nens=1}).}

\item{slope}{Prescribed spectral slope. The default is \code{slope=0.}
meaning that the slope is determined automatically over the dimensions specified by \code{time_dim}.}

\item{kmin}{First wavenumber for spectral slope (default: \code{kmin=1}).}

\item{fglob}{Logical to conseve global precipitation over the domain (default: FALSE)}

\item{fsmooth}{Logical to conserve precipitation with a smoothing kernel (default: TRUE)}

\item{time_dim}{String or character array with name(s) of time dimension(s)
(e.g. "ftime", "sdate", "time" ...) over which to compute spectral slopes.
If a character array of dimension names is provided, the spectral slopes
will be computed over all elements belonging to those dimensions.
If omitted one of c("ftime", "sdate", "time")
is searched and the first one with more than one element is chosen.}

\item{lon_dim}{Name of lon dimension ("lon" by default).}

\item{lat_dim}{Name of lat dimension ("lat" by default).}

\item{drop_realization_dim}{Logical to remove the "realization" stochastic ensemble dimension (default: FALSE)

1) if \code{nens==1}: the dimension is dropped;

2) if \code{nens>1} and a "member" dimension exists:
   the "realization" and "member" dimensions are compacted (multiplied) and the resulting dimension is named "member";

3) if \code{nens>1} and a "member" dimension does not exist: the "realization" dimension is renamed to "member".}

\item{verbose}{logical for verbose output (default: FALSE).}
}
\value{
RainFARM() returns a list containing the fine-scale longitudes, latitudes
and the sequence of \code{nens} downscaled fields.
If \code{nens>1} an additional dimension named "realization" is added to the output array
after the "member" dimension (if it exists and unless \code{drop_realization_dim=TRUE} is specified).
The ordering of the remaining dimensions in the \code{exp} element of the input object is maintained.
}
\description{
This function implements the RainFARM stochastic precipitation downscaling method
and accepts in input an array with named dims ("lon", "lat")
and one or more dimension (such as "ftime", "sdate" or "time")
over which to average automatically determined spectral slopes.
Adapted for climate downscaling and including orographic correction.
References:
Terzago, S. et al. (2018). NHESS 18(11), 2825-2840. http://doi.org/10.5194/nhess-18-2825-2018,
D'Onofrio et al. (2014), J of Hydrometeorology 15, 830-843; Rebora et. al. (2006), JHM 7, 724.
}
\examples{
# Example for the 'reduced' RainFARM function 
nf <- 8   # Choose a downscaling by factor 8
nens <- 3 # Number of ensemble members
# create a test array with dimension 8x8 and 20 timesteps
# or provide your own read from a netcdf file
pr <- rnorm(8 * 8 * 20)
dim(pr) <- c(lon = 8, lat = 8, ftime = 20)
lon_mat <- seq(10, 13.5, 0.5) # could also be a 2d matrix
lat_mat <- seq(40, 43.5, 0.5)
# Create a test array of weights
ww <- array(1., dim = c(8 * nf, 8 * nf))
# or create proper weights using an external fine-scale climatology file
#     Specify a weightsfn filename if you wish to save the weights
\dontrun{
ww <- CST_RFWeights("./worldclim.nc", nf, lon = lon_mat, lat = lat_mat, 
                    fsmooth = TRUE)
}
# downscale using weights (ww=1. means do not use weights)
res <- RainFARM(pr, lon_mat, lat_mat, nf, 
                fsmooth = TRUE, fglob = FALSE, 
                weights = ww, nens = 2, verbose = TRUE)
str(res)
#List of 3
# $ data: num [1:3, 1:20, 1:64, 1:64] 0.186 0.212 0.138 3.748 0.679 ...
# $ lon : num [1:64] 9.78 9.84 9.91 9.97 10.03 ...
# $ lat : num [1:64] 39.8 39.8 39.9 40 40 ...
dim(res$data)
#  lon         lat       ftime realization 
#   64          64          20           2 

}
\author{
Jost von Hardenberg - ISAC-CNR, \email{j.vonhardenberg@isac.cnr.it}
}

