#' Bayesian analysis for the OSL age estimation of one sample
#'
#' This function computes the age (in ka) of a sample according to the model developed in Combes and Philippe (2017),
#' based on an output of \code{\link{Generate_DataFile}} or \code{\link{Generate_DataFile_MG}}.\cr
#' A sample, for which data is avalilable in several BIN files, can be analysed.
#'
#' @param DATA list of objects: LT, sLT, ITimes, dLab, ddot_env, regDose, J, K, Nb_measurement,
#' provided by the function \code{\link{Generate_DataFile}} or \code{\link{Generate_DataFile_MG}}.
#' \code{DATA} can contain informations for more than one sample.
#' @param SampleName character: name of the sample.
#' @param PriorAge numeric vector (with default): lower and upper bounds for the sample age parameter (in ka).
#' Note that, \code{length(PriorAge)=2}.
#' @param BinPerSample integer vector (with default): vector with the number of BIN files per sample.
#' If in \code{DATA} there is more than one sample,
#' the \code{BinPerSample} vector must be the same as that used to run the function
#' \code{\link{Generate_DataFile}} or in \code{\link{Generate_DataFile_MG}} for generating the \code{DATA} object.
#' @param SavePdf boolean (with default): if TRUE save graph in pdf file named \code{OutputFileName} in folder \code{OutputFilePath}.
#' @param OutputFileName character (with default): name of the pdf file that will be generated by the function if \code{SavePdf}=TRUE;
#' \code{length(OutputFileName)=2}, see \bold{PLOT OUTPUT} in \bold{Value} section for more informations.
#' @param OutputFilePath character (with default): path to the pdf file that will be generated by the function if \code{SavePdf}=TRUE.
#' If it is not equal to "", it must be terminated by "/".
#' @param SaveEstimates boolean (with default): if TRUE save Bayes estimates and credible interval at level 68\% and 95\%  and
#' the result of the gelman en Rudin test of convergency, in a csv table named \code{OutputFileName} in folder \code{OutputFilePath}.
#' @param OutputTableName character (with default): name of the table that will be generated by the function if \code{SaveEstimates}=TRUE.
#' @param OutputTablePath character (with default): path to the table that will be generated by the function if \code{SaveEstimates}=TRUE.
#' If it is not equal to "", it must be terminated by "/".
#' @param LIN_fit logical (with default): if TRUE (default) allows a linear component,
#' on top of the (default) saturating exponential curve, for the fitting of dose response curves.
#' See details section for more informations on the proposed dose response curves.
#' @param Origin_fit logical (with default): if TRUE, forces the dose response curves to pass through the origin.
#' See details section for more informations on the proposed growth curves.
#' @param distribution character (with default): type of distribution that defines
#' how individual equivalent dose values are distributed around the palaeodose.
#' Allowed inputs are \bold{"cauchy"}, \bold{"gaussian"}, \bold{"lognormal_A"} and \bold{"lognormal_M"}, see details section for more informations.
#' @param I integer (with default): if \code{DATA} contains data from more than one sample,
#' I indicates the ID number of the sample to be analysed.
#' @param Iter integer (with default): number of iterations for the MCMC computation (for more information see \code{\link{jags.model}}).
#' @param t integer (with default): 1 every \code{t} iterations of the MCMC is considered for sampling the posterior distribution
#' (for more information see \code{\link{jags.model}}).
#' @param Nb_chaines integer (with default): number of independent chains for the model (for more information see \code{\link{jags.model}}).
#'
#' @details
#'
#' \bold{** Option on growth curves **}\cr
#'
#' As for \code{\link{AgeS_Computation}} and \code{\link{Palaeodose_Computation}}, the user can choose from 4 dose response curves:
#' \itemize{
#'   \item \bold{Saturating exponential plus linear growth} (\code{AgeMultiBF_EXPLIN}):
#'
#'   for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+cx+d}; select
#'   \itemize{
#'     \item \code{LIN_fit=TRUE}
#'     \item \code{Origin_fit=FALSE}
#'   }
#'   \item \bold{Saturating exponential growth} (\code{AgeMultiBF_EXP}):
#'
#'   for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+d}; select
#'   \itemize{
#'     \item \code{LIN_fit=FALSE}
#'     \item \code{Origin_fit=FALSE}
#'   }
#'   \item \bold{Saturating exponential plus linear growth and fitting through the origin} (\code{AgeMultiBF_EXPLINZO}):
#'
#'   for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))+cx}; select
#'   \itemize{
#'     \item \code{LIN_fit=TRUE}
#'     \item \code{Origin_fit=TRUE}
#'   }
#'   \item \bold{Saturating exponential growth and fitting through the origin} (\code{AgeMultiBF_EXPZO}):
#'
#'   for all \code{x} in IR+, \code{f(x)=a(1-exp(-x/b))}; select
#'   \itemize{
#'     \item \code{LIN_fit=FALSE}
#'     \item \code{Origin_fit=TRUE}
#'   }
#' }
#'
#' \bold{** Option on equivalent dose distribution around the palaeodose **}\cr
#'
#' The use can choose between :
#' \itemize{
#'   \item \code{cauchy}: a Cauchy distribution with location parameter equal to the palaeodose of the sample
#'   \item \code{gaussian}: a Gaussian distribution with mean equal to the palaeodose of the sample
#'   \item \code{lognormal_A}: a log-normal distribution with mean or \bold{A}verage equal to the palaeodose of the sample
#'   \item \code{lognormal_M}: a log-normal distribution with \bold{M}edian equal to the palaeodose of the sample
#' }
#'
#' @return
#' \bold{NUMERICAL OUTPUT}
#'
#' \enumerate{
#'  \item \bold{A list containing the following objects}:
#'  \itemize{
#'   \item \bold{Sampling} that corresponds to a sample of the posterior distributions
#'  of the age (in ka), palaeodose (in Gy) and equivalent dose dispersion (in Gy) parameters.
#'   \item \bold{Model_GrowthCurve}, stating which dose response fitting option was chosen;
#'   \item \bold{Distribution}, stating which distribution was chosen to model the dispersion of
#'  individual equivalent dose values around the palaeodose of the sample;
#'   \item \bold{PriorAge}, stating the priors used for the age parameter (in ka).
#'  }
#'  \item\bold{The Gelman and Rudin test of convergency}: prints the result of the Gelman and Rudin test of convergency for the age, palaeodose and equivalent dose dispersion parameters.
#' A result close to one is expected.\cr
#' In addition, the user must visually assess the convergency of the trajectories by looking at the graph
#' generated by the function (see \bold{PLOT OUTPUT} for more informations).\cr
#' If both convergencies (Gelman and Rudin test and plot checking) are satisfactory,
#' the user can consider the printed estimates as valid. Otherwise, the user may try increasing the number of MCMC interations
#' (\code{Iter}), or being more precise on the \code{PriorAge} parameter (for example specify if it is a young sample \code{c(0.01,10)} an old sample \code{c(10,100)}),
#' or changing the parameter \code{distribution} or the growth curve, to reach convergency.to reach convergency.
#' \item\bold{Credible intervals and Bayes estimates}: prints the Bayes esitmates, the credible intervals at 95\% and 68\% for
#' the age, palaeodose and equivalent dose dispersion parameters of the sample.
#' }
#'
#' \bold{PLOT OUTPUT}
#'
#' A graph with the MCMC trajectories and posterior distributions of the age, palaeodose and equivalent dose dispersion parameters is displayed.\cr
#' The first line of the figure correponds to the age parameter, the second to the palaeodose parameter and the third to the equivalent dose dispersion parameter.
#' On each line, the plot on the left represents the MCMC trajectories, and the one on the right the posterior distribution of the parameter.
#'
#' To give the results in a publication, we recommend to give the Bayes estimate of the parameter as well as the credible interval at 95\% or 68\%.
#'
#' @author Claire Christophe, Anne Philippe, Guillaume Guerin
#'
#' @seealso \code{\link{Generate_DataFile}}, \code{\link{Generate_DataFile_MG}},
#' \code{\link{rjags}}, \code{\link{MCMC_plot}}
#' \code{\link{AgeS_Computation}}, \code{\link{Palaeodose_Computation}}
#'
#' @references
#' Combes, Benoit and Philippe, Anne, 2017.
#' Bayesian analysis of multiplicative Gaussian error for multiple ages estimation in optically stimulated luminescence dating.
#' Quaternary Geochronology (39, 24-34)
#'
#' Combes, B., Philippe, A., Lanos, P., Mercier, N., Tribolo, C., Guerin, G., Guibert, P., Lahaye, C., 2015.
#' A Bayesian central equivalent dose model for optically stimulated luminescence dating.
#' Quaternary Geochronology 28, 62-70. doi:10.1016/j.quageo.2015.04.001
#'
#' @examples
#' ## load data file generated by the function Generate_DataFile
#' data(DATA1,envir = environment())
#' priorage=c(10,60) # GDB3 is an old sample
#' Age=Age_Computation(DATA=DATA1,SampleName="GDB3",PriorAge=priorage,Iter=100)
#'
#' @export
Age_Computation<-function(DATA,SampleName,
                          PriorAge=c(0.01,100),
                          BinPerSample=c(1),
                          SavePdf=FALSE,OutputFileName=c("MCMCplot"),OutputFilePath=c(''),
                          SaveEstimates=FALSE,OutputTableName=c("DATA"),OutputTablePath=c(''),
                          LIN_fit = TRUE,Origin_fit = FALSE,
                          distribution=c("cauchy"),
                          I=1,Iter=50000,t=5,Nb_chaines=3){

  Model_Age<-0
  data(Model_Age,envir = environment())

  if(LIN_fit==TRUE){
    cLIN=c('LIN')
  }else{cLIN=c()}
  if(Origin_fit==TRUE){
    cO=c("ZO")
  }else{cO=c()}
  Model_GrowthCurve=c(paste("AgeMultiBF_EXP",cLIN,cO,sep=""))

  CSBinPerSample=cumsum(BinPerSample)

  if(BinPerSample[I]==1){
    index2=0
  }else{
    index2=rep(0,BinPerSample[I])
    index2[2:BinPerSample[I]]=cumsum(DATA$J[(CSBinPerSample[I]-BinPerSample[I]+1):(CSBinPerSample[I]-1)])
  }

  dataList = list('N'= DATA$LT[[I]],'sN'=DATA$sLT[[I]],"IT"=DATA$ITimes[[I]],
                  "sDlab"=DATA$dLab[1,(CSBinPerSample[I]-BinPerSample[I]+1):(CSBinPerSample[I])],
                  'J'=DATA$J[(CSBinPerSample[I]-BinPerSample[I]+1):(CSBinPerSample[I])],
                  'K'=DATA$K[(CSBinPerSample[I]-BinPerSample[I]+1):(CSBinPerSample[I])],
                  "ddot"=DATA$ddot_env[1,(CSBinPerSample[I]-BinPerSample[I]+1)],
                  "Sigma"=DATA$ddot_env[2,(CSBinPerSample[I]-BinPerSample[I]+1)]+DATA$ddot_env[1,(CSBinPerSample[I]-BinPerSample[I]+1)]^2*DATA$dLab[2,(CSBinPerSample[I]-BinPerSample[I]+1)],
                  "xbound"=PriorAge,"index"=index2,"BinPerSample"=BinPerSample[I])

  jags <- rjags::jags.model(textConnection(Model_Age[[Model_GrowthCurve]][[distribution]]), data = dataList, n.chains = Nb_chaines, n.adapt=Iter)
  update(jags,Iter)
  echantillon = rjags::coda.samples(jags,c("A","D","sD"),min(Iter,10000),thin=t)

  CV=gelman.diag(echantillon)


  sample=echantillon[[1]]
  for(i in 2:Nb_chaines){
    sample=rbind(sample,echantillon[[i]])
  }
  MCMC_plot(sample,
            length(echantillon[[1]][,1]),
            SampleNames=SampleName,
            Nb_sample=1,
            Nb_chaines=Nb_chaines,
            value=c(0,1,2),
            param=c("A","D","sD"))

  if(SavePdf==TRUE){
    dev.print(pdf,file=paste(OutputFilePath,OutputFileName,'.pdf',sep=""),width=8,height=10)
  }


  cat(paste("\n\n>> Sample name <<\n"))
  cat("----------------------------------------------\n")
  cat(paste(SampleName))

  cat(paste("\n\n>> Result of Gelman and Rudin critere of convergency<<\n"))
  cat("----------------------------------------------\n")
  cat(paste("\t", "Point estimate", "Uppers confidence interval\n"))
  cat(paste("A\t",round(CV$psrf[1,1],2),"\t\t",round(CV$psrf[1,2],2),"\n"))
  cat(paste("D\t",round(CV$psrf[2,1],2),"\t\t",round(CV$psrf[2,2],2),"\n"))
  cat(paste("sD\t",round(CV$psrf[3,1],2),"\t\t",round(CV$psrf[3,2],2),"\n"))

  cat("\n\n________________________________________________________________________________\n")
  cat(" *** WARNING: following informations are only valid if MCMC chains converged  ***\n")
  cat("________________________________________________________________________________\n\n")

  # Matrix of results
  R=matrix(data=NA,ncol=8,nrow=3,dimnames=list(c("A","D","sD"),
          c("lower bound at 95%","lower bound at 68%","Bayes estimate","upper bound at 68%","upper bound at 95%","","Convergencies: Bayes estimate","Convergencies: uppers credible interval")))

  cat(paste("parameter", "\t","Bayes estimate","\t"," Credible interval \n"))
  cat("----------------------------------------------\n")
  cat(paste("A","\t\t", round(mean(sample[,1]),3),'\n'))
  HPD_95=ArchaeoPhases::CredibleInterval(echantillon[[1]][,1],0.95)
  HPD_68=ArchaeoPhases::CredibleInterval(echantillon[[1]][,1],0.68)
  cat("\t\t\t\t\t\t lower bound \t upper bound\n")
  cat("\t\t\t\t at level 95%\t",round(c(HPD_95[2]),2),"\t\t",round(c(HPD_95[3]),2),"\n")
  cat("\t\t\t\t at level 68%\t",round(c(HPD_68[2]),2),"\t\t",round(c(HPD_68[3]),2),"\n")
  cat("----------------------------------------------\n")

  R[1,3]=round(mean(sample[,1]),3)
  R[1,c(1,5)]=round(HPD_95[2:3],3)
  R[1,c(2,4)]=round(HPD_68[2:3],3)

  cat(paste("D","\t\t", round(mean(sample[,2]),3),'\n'))
  HPD_95=ArchaeoPhases::CredibleInterval(echantillon[[1]][,2],0.95)
  HPD_68=ArchaeoPhases::CredibleInterval(echantillon[[1]][,2],0.68)
  cat("\t\t\t\t\t\t lower bound \t upper bound\n")
  cat("\t\t\t\t at level 95%\t",round(c(HPD_95[2]),2),"\t\t",round(c(HPD_95[3]),2),"\n")
  cat("\t\t\t\t at level 68%\t",round(c(HPD_68[2]),2),"\t\t",round(c(HPD_68[3]),2),"\n")

  R[2,3]=round(mean(sample[,2]),3)
  R[2,c(1,5)]=round(HPD_95[2:3],3)
  R[2,c(2,4)]=round(HPD_68[2:3],3)

  cat("----------------------------------------------\n")
  cat(paste("sD","\t\t", round(mean(sample[,3]),3),'\n'))
  HPD_95=ArchaeoPhases::CredibleInterval(echantillon[[1]][,3],0.95)
  HPD_68=ArchaeoPhases::CredibleInterval(echantillon[[1]][,3],0.68)
  cat("\t\t\t\t\t\t lower bound \t upper bound\n")
  cat("\t\t\t\t at level 95%\t",round(c(HPD_95[2]),2),"\t\t",round(c(HPD_95[3]),2),"\n")
  cat("\t\t\t\t at level 68%\t",round(c(HPD_68[2]),2),"\t\t",round(c(HPD_68[3]),2),"\n")

  R[3,3]=round(mean(sample[,3]),3)
  R[3,c(1,5)]=round(HPD_95[2:3],3)
  R[3,c(2,4)]=round(HPD_68[2:3],3)

  R[,6]=c('','','')
  R[,7]=round(CV$psrf[,1],2)
  R[,8]=round(CV$psrf[,2],2)

  if(SaveEstimates==TRUE){
    write.csv(R,file=c(paste(OutputTablePath,"Estimates",OutputTableName,".csv",sep="")))
  }

  Info=list("Sampling"=echantillon,"Model_GrowthCurve"=Model_GrowthCurve, "Distribution"=distribution,"PriorAge"=PriorAge)
  return(Info)

}
