% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bcov.R
\name{bcov.test}
\alias{bcov.test}
\alias{bcov.test.default}
\alias{bcov.test.formula}
\title{Ball Covariance Test}
\usage{
bcov.test(x, ...)

\method{bcov.test}{default}(
  x,
  y = NULL,
  num.permutations = 99,
  method = c("permutation", "limit"),
  distance = FALSE,
  weight = FALSE,
  seed = 1,
  num.threads = 0,
  ...
)

\method{bcov.test}{formula}(formula, data, subset, na.action, ...)
}
\arguments{
\item{x}{a numeric vector, matrix, data.frame, or a list containing at least two numeric vectors, matrices, or data.frames.}

\item{...}{further arguments to be passed to or from methods.}

\item{y}{a numeric vector, matrix, or data.frame.}

\item{num.permutations}{the number of permutation replications. When \code{num.permutations = 0}, the function just returns
the Ball Covariance statistic. Default: \code{num.permutations = 99}.}

\item{method}{if \code{method = "permutation"}, a permutation procedure is carried out to compute the \eqn{p}-value;
if \code{ method = "limit"}, an approximate null distribution is used when \code{weight = "constant"}.
Any unambiguous substring can be given. Default \code{method = "permutation"}.}

\item{distance}{if \code{distance = TRUE}, the elements of \code{x} and \code{y} are considered as distance matrices.}

\item{weight}{a logical or character string used to choose the weight form of Ball Covariance statistic.. 
If input is a character string, it must be one of \code{"constant"}, \code{"probability"}, or \code{"chisquare"}. 
Any unambiguous substring can be given. 
If input is a logical value, it is equivalent to \code{weight = "probability"} if \code{weight = TRUE} while 
equivalent to \code{weight = "constant"} if \code{weight = FALSE}.
Default: \code{weight = FALSE}.}

\item{seed}{the random seed. Default \code{seed = 1}.}

\item{num.threads}{number of threads. If \code{num.threads = 0}, then all of available cores will be used. Default \code{num.threads = 0}.}

\item{formula}{a formula of the form \code{~ u + v}, where each of \code{u} and \code{v} are numeric variables giving the data values for one sample. The samples must be of the same length.}

\item{data}{an optional matrix or data frame (or similar: see \code{model.frame}) containing the variables in the formula formula. By default the variables are taken from environment(formula).}

\item{subset}{an optional vector specifying a subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when the data contain \code{NA}s. Defaults to \code{getOption("na.action")}.}
}
\value{
If \code{num.permutations > 0}, \code{bcov.test} returns a \code{htest} class object containing the following components:
\item{\code{statistic}}{Ball Covariance statistic.}            
\item{\code{p.value}}{the p-value for the test.}  
\item{\code{replicates}}{permutation replications of the test statistic.}
\item{\code{size}}{sample size.} 
\item{\code{complete.info}}{a \code{list} mainly containing two vectors, the first vector is the Ball Covariance statistics 
with different weights, the second is the \eqn{p}-values of weighted Ball Covariance tests.}
\item{\code{alternative}}{a character string describing the alternative hypothesis.}
\item{\code{method}}{a character string indicating what type of test was performed.}
\item{\code{data.name}}{description of data.}
If \code{num.permutations = 0}, \code{bcov.test} returns a statistic value.
}
\description{
Ball Covariance test of independence. 
Ball covariance are generic dependence measures in Banach spaces.
}
\details{
\code{bcov.test} is non-parametric tests of independence in Banach spaces. 
It can detect the dependence between two random objects (variables) and 
the mutual dependence among at least three random objects (variables).

If two samples are pass to arguments \code{x} and \code{y}, the sample sizes (i.e. number of rows or length of the vector) 
of the two variables must agree. If a \code{\link{list}} object is passed to \code{x}, this \code{list} must contain at least
two numeric vectors, matrices, or data.frames, and each element of this \code{list} 
must with the same sample size. Moreover, data pass to \code{x} or \code{y} 
must not contain missing or infinite values. 
If \code{distance = TRUE}, \code{x} is considered as a distance matrix or a list containing distance matrices, 
and \code{y} is considered as a distance matrix; otherwise, these arguments are treated as data.

\code{bcov.test} utilizes the Ball Covariance statistics (see \code{\link{bcov}}) to measure dependence and 
derives a \eqn{p}-value via replicating the random permutation \code{num.permutations} times.

See Pan et al 2018 for theoretical properties of the test, including statistical consistency.
}
\note{
Actually, \code{bcov.test} simultaneously computing Ball Covariance statistics with 
\code{"constant"}, \code{"probability"}, and \code{"chisquare"} weights.
Users can get other Ball Covariance statistics with different weight and their corresponding \eqn{p}-values 
in the \code{complete.info} element of output. We give a quick example below to illustrate.
}
\examples{
set.seed(1)

################# Quick Start #################
noise <- runif(50, min = -0.3, max = 0.3)
x <- runif(50, 0, 4*pi)
y <- cos(x) + noise
# plot(x, y)
res <- bcov.test(x, y)
res
## get all Ball Covariance statistics:
res[["complete.info"]][["statistic"]]
## get all test result:
res[["complete.info"]][["p.value"]]

################# Quick Start #################
x <- matrix(runif(50 * 2, -pi, pi), nrow = 50, ncol = 2)
noise <- runif(50, min = -0.1, max = 0.1)
y <- sin(x[,1] + x[,2]) + noise
bcov.test(x = x, y = y, weight = "prob")

################# Ball Covariance Test for Non-Hilbert Data #################
# load data:
data("ArcticLake")
# Distance matrix between y:
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
# Distance matrix between x:
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with BCov:
bcov.test(x = Dx, y = Dy, distance = TRUE)

################  Weighted Ball Covariance Test  #################
data("ArcticLake")
Dy <- nhdist(ArcticLake[["x"]], method = "compositional")
Dx <- dist(ArcticLake[["depth"]])
# hypothesis test with weighted BCov:
bcov.test(x = Dx, y = Dy, distance = TRUE, weight = "prob")

################# Mutual Independence Test #################
x <- rnorm(50)
y <- (x > 0) * x + rnorm(50)
z <- (x <= 0) * x + rnorm(50)
data_list <- list(x, y, z)
bcov.test(data_list)
data_list <- lapply(data_list, function(x) {
  as.matrix(dist(x))
})
bcov.test(data_list, distance = TRUE)
bcov.test(data_list, distance = FALSE, weight = "chi")

################# Mutual Independence Test for Meteorology data #################
data("meteorology")
bcov.test(meteorology)

################  Testing via approximate limit distribution  #################
\dontrun{
set.seed(1)
n <- 2000
x <- rnorm(n)
y <- rnorm(n)
bcov.test(x, y, method = "limit")
bcov.test(x, y)
}

################  Formula interface  ################
## independence test:
bcov.test(~ CONT + INTG, data = USJudgeRatings)
## independence test with chisquare weight:
bcov.test(~ CONT + INTG, data = USJudgeRatings, weight = "chi")
## mutual independence test:
bcov.test(~ CONT + INTG + DMNR, data = USJudgeRatings)
}
\references{
Wenliang Pan, Xueqin Wang, Heping Zhang, Hongtu Zhu & Jin Zhu (2019) Ball Covariance: A Generic Measure of Dependence in Banach Space, Journal of the American Statistical Association, DOI: 10.1080/01621459.2018.1543600

Jin Zhu, Wenliang Pan, Wei Zheng, and Xueqin Wang (2021). Ball: An R Package for Detecting Distribution Difference and Association in Metric Spaces, Journal of Statistical Software, Vol.97(6), doi: 10.18637/jss.v097.i06.
}
\seealso{
\code{\link{bcov}}, \code{\link{bcor}}
}
\author{
Wenliang Pan, Xueqin Wang, Heping Zhang, Hongtu Zhu, Jin Zhu
}
