#' @return Returns a list of class \code{bv_dummy.}
#'
#' @noRd
dummy <- function(
  mode = 1,
  min = 0.0001, max = 5,
  ...) {

  if(0 >= min || min >= max) {stop("Boundaries misspecified.")}
  if(mode < 0) {stop("Parameter misspecified.")}

  out <- list("mode" = mode, "min" = min, "max" = max, ...)
  class(out) <- "bv_dummy"

  return(out)
}


#' Dummy prior settings
#'
#' Allows the creation of dummy observation priors for \code{\link{bv_priors}}.
#' See the Details section for information on common dummy priors.
#'
#' Dummy priors are often used to "reduce the importance of the deterministic
#' component implied by VARs estimated conditioning on the initial
#' observations" (Giannone et al. 2015, p. 440). One such prior is the
#' sum-of-coefficients (SOC) prior, which imposes the notion that a no-change
#' forecast is optimal at the beginning of a time series. Its key parameter
#' \eqn{\mu} controls the tightness - i.e. for low values the model is pulled
#' towards a form with as many unit roots as variables and no cointegration.
#' Another such prior is the single-unit-root (SUR) prior, that allows for
#' cointegration relationships in the data. It pushes variables either towards
#' their unconditional mean or towards the presence of at least one unit root.
#' These priors are implemented via Theil mixed estimation, i.e. by adding
#' dummy-observations on top of the data matrix. They are readily available
#' via the shorthand functions \code{\link{bv_soc}} and \code{\link{bv_sur}}.
#'
#' @param fun Function taking \emph{Y}, \emph{lags} and the prior's parameter
#' \emph{par} to generate and return a named list with elements \emph{X} and
#' \emph{Y} (numeric matrices).
#' @inheritParams bv_mn
#'
#' @return Returns a named list of class \code{bv_dummy} for
#' \code{\link{bv_priors}}.
#'
#' @references
#'     Giannone, D., Lenza, M., & Primiceri, G. E. (2015). Prior Selection for Vector Autoregressions. Review of Economics and Statistics, 97, 436-451. \url{https://doi.org/10.1162/REST_a_00483}.
#'
#' @seealso \code{\link{bv_priors}}; \code{\link{bv_minnesota}}
#'
#' @export
#'
#' @examples
#' # Create a sum-of-coefficients prior
#' add_soc <- function(Y, lags, par) {
#'   soc <- if(lags == 1) {diag(Y[1, ]) / par} else {
#'     diag(colMeans(Y[1:lags, ])) / par
#'   }
#'   Y_soc <- soc
#'   X_soc <- cbind(rep(0, ncol(Y)), matrix(rep(soc, lags), nrow = ncol(Y)))
#'
#'   return(list("Y" = Y_soc, "X" = X_soc))
#' }
#' soc <- bv_dummy(mode = 1, sd = 1, min = 0.0001, max = 50, fun = add_soc)
#'
#' # Create a single-unit-root prior
#' add_sur <- function(Y, lags, par) {
#'   sur <- if(lags == 1) {Y[1, ] / par} else {
#'     colMeans(Y[1:lags, ]) / par
#'   }
#'   Y_sur <- sur
#'   X_sur <- c(1 / par, rep(sur, lags))
#'
#'   return(list("Y" = Y_sur, "X" = X_sur))
#' }
#'
#' sur <- bv_dummy(mode = 1, sd = 1, min = 0.0001, max = 50, fun = add_sur)
#'
#' # Adding them to the prior list with `bv_priors()`
#' priors_dum <- bv_priors(hyper = "auto", soc = soc, sur = sur)
bv_dummy <- function(
  mode = 1, sd = 1,
  min = 0.0001, max = 5,
  fun) {

  if(sd <= 0) {stop("Parameter sd misspecified.")}
  fun <- match.fun(fun)

  return(dummy(mode, min, max, sd = sd, fun = fun,
               coef = gamma_coef(mode, sd)))
}
