% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paths.R
\name{paths}
\alias{paths}
\title{Plot covariate paths for BTLLasso}
\usage{
paths(model, y.axis = c("penalty", "L2"), x.axis = c("penalty", "lambda",
  "loglambda"))
}
\arguments{
\item{model}{BTLLasso or cv.BTLLasso object}

\item{y.axis}{Two possible values for the y-axis. Variables can either be plotted
with regard to their contribution to the total penalty term (\code{y.axis="penalty"}) or
with regard to the $L_2$ norm of the corresponding parameter vector (\code{y.axis="L2"}).}

\item{x.axis}{Should the paths be plotted against the scaled penalty term (between 0 and 1),
against lambda or against log(lambda+1)?}
}
\description{
Plots paths for every covariate of a BTLLasso object or a cv.BTLLasso
object. In contrast, to \code{\link{singlepaths}}, only one plot is created,
every covariate is illustrated by one path. For cv.BTLLasso object, the
optimal model according to the cross-validation is marked by a vertical
dashed line.
}
\details{
Plots for BTLLasso and cv.BTLLasso objects only differ by the additional
vertical line indicating the optimal model according to cross-validation.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151),log(1.05),length=30))-1

## Specify control argument, allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)

singlepaths(m.sim, x.axis = "loglambda")

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)


singlepaths(m.sim.cv, x.axis = "loglambda", plot.order.effect = FALSE, plot.intercepts = FALSE, 
            plot.Z2 = FALSE)
paths(m.sim.cv, y.axis="L2")

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
ci.BTLLasso(m.sim.boot)

##### Example with small version from GLES data set
data(GLESsmall)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c("","female (1); male (0)")

## vector of tuning parameters
lambda <- exp(seq(log(61),log(1),length=30))-1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles, x.axis = "loglambda", subs.X = subs.X)
paths(m.gles, y.axis="L2")

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles.cv, x.axis = "loglambda", subs.X = subs.X)
}

}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183
}
\seealso{
\code{\link{BTLLasso}}, \code{\link{cv.BTLLasso}},
\code{\link{singlepaths}}
}
\keyword{BTLLasso}
\keyword{covariate}
\keyword{paths}

