
<!-- README.md is generated from README.Rmd. Please edit that file -->
BNPMediation
============

The goal of BNPMediation is to provide two functions `bnpmediation` and `bnpconmediation`. The function `bnpmediation` provides the posterior means and credible intervals of the effects of mediation, whereas `bnpconmediation` provides the posterior means and credible intervals of the conditional effects. Another function `plot_effects` is provide that obtains the posterior plots of the effects. Bayesian Nonparametric (BNP) models are used for the observed data to identify and estimate the causal effects. The published journal can be found at <http://dx.doi.org/10.1111/biom.12575>.

Installation
------------

You can install BNPMediation from github with:

``` r
# install.packages("devtools")
devtools::install_github("mjdaniels/BNPMediationProject") 
```

Example
-------

``` r

dataTreatment = c(11.163989,  13.357644,  12.264292,  16.848934,  20.902562,  13.037647,  10.523266, 13.869733,  16.399047,  13.249197,  11.284890,  11.450062,  10.817139,  15.245700, 9.920019,  11.573960,  10.911009,  10.489587,  10.814474,  10.479123,  11.576099, 9.527664,  19.139293,  10.419879,  11.477210, 9.974351,  11.661321,  11.107294, 10.703919,  10.229351,  14.304395,  11.966633,  12.320991,  17.905459,  9.661749, 10.385607,  20.406423,  10.777457,  9.825228,  13.691681, 9.871743,  10.500229, 10.285037,  14.930915,  13.608770,  16.167101,  13.602650,  14.786788,  10.867811, 10.512889,  11.412777,  11.982880,  8.774465,  10.481118,  9.444334,  15.367506,  9.772886,  10.974271,  9.893823,  20.102887,  11.444247,  10.500275,  9.847932,  11.732694,  11.462192,  13.380679,  12.794651,  9.678496,  17.062716,  14.091250,  9.289893,  10.719766,  10.636155,  9.798370,  10.298192,  10.618451,  10.365415,  18.189078,  15.972311,  11.072824,  12.343840,  11.811951,  10.639700,  15.114964,  11.880889,  11.335969,  11.388849,  10.878676,  10.722346,  11.307921,  9.956277,  9.213929,  10.085210,  11.800195,  10.658099,  10.718170,  11.422773,  10.623135,  11.142688,  10.719141,  12.885994,  9.060620,  13.221060,  9.687241,  12.399068,  12.719150,  10.968212,  12.011608,  11.513917,  9.541409,  11.621904,  13.197714,  11.697384,  14.032590,  12.267005,  15.554890,  11.958848 )
dataControl = c(9.060491, 7.566083, 8.081978, 6.450560, 7.252086, 7.545289, 9.019265 ,7.678507, 7.353213, 10.708195, 8.253596, 8.413139, 9.588276, 10.589635 ,8.619169, 9.088016, 7.063381, 6.403902, 8.413106, 7.254758, 7.946906,7.013008, 6.828835, 7.198124, 8.133756, 14.843925, 8.043300, 9.978774,11.630415, 6.912515, 7.354103, 6.634561, 9.672022, 9.785791, 7.490891,12.384268, 9.223109, 6.719683, 8.311750, 9.388895, 7.773400, 14.815319,20.050661, 6.417857, 10.674383, 9.293519, 14.880615, 8.090496, 7.880399, 7.181893, 7.913207, 7.086220, 10.728450, 8.687074, 6.958637, 14.135874, 7.273476, 10.325233, 7.736805, 14.173735, 7.773935, 9.227020, 8.257979 ,11.986135, 7.875595, 7.341740, 6.036009, 8.976403, 7.396193, 7.217195, 6.849920, 10.399304, 12.968451, 7.879505, 7.816833, 7.270345, 7.299595,10.799175, 6.471268, 6.234942, 8.203609, 6.411400, 10.768465, 8.162683, 7.262786, 8.473937, 9.886522, 7.018580, 6.836707, 7.283106, 7.344599, 8.686345, 9.547393, 10.468727, 14.321967, 7.183868, 10.743336, 6.865275, 7.725663, 7.168826, 9.654838, 14.515487, 9.493838, 12.502887, 9.970782, 7.366840, 8.856238, 7.529891, 15.989223, 7.827295, 7.505671, 7.966335, 8.062032, 6.708697, 10.414558, 10.033180, 7.512152, 10.476679, 9.338416, 6.898800, 7.380856, 10.896545, 9.672738 )

  
 nburn <- 500
 nsave <- 1000
 nskip <- 10
 ndisplay <- 100
 mcmc <- list(nburn=nburn,nsave=nsave,nskip=nskip,ndisplay=ndisplay)

 s2 <- matrix(c(1000,0,0,1),ncol=2)
 m2 <- c(20,3)
 psiinv2 <- solve(matrix(c(1000,0,0,1),ncol=2))
 prior <- list(a0=1,b0=1/5,nu1=4,nu2=4,s2=s2,m2=m2,psiinv2=psiinv2,tau1=0.01,tau2=0.01)

library(BNPMediation)
bnpmediation(dataTreatment, dataControl, prior, mcmc, state, status = TRUE, na.action, q = 2, NN = 10, n1 = 10, n0 = 10, extra.thin = 0, seed= 12345)
#> ***** Fitting observed data models via DPpackage::DPdensity()
#> 
#> MCMC scan 100 of 1000 (CPU time: 0.368 s)
#> MCMC scan 200 of 1000 (CPU time: 0.697 s)
#> MCMC scan 300 of 1000 (CPU time: 1.012 s)
#> MCMC scan 400 of 1000 (CPU time: 1.350 s)
#> MCMC scan 500 of 1000 (CPU time: 1.660 s)
#> MCMC scan 600 of 1000 (CPU time: 1.988 s)
#> MCMC scan 700 of 1000 (CPU time: 2.296 s)
#> MCMC scan 800 of 1000 (CPU time: 2.623 s)
#> MCMC scan 900 of 1000 (CPU time: 2.950 s)
#> MCMC scan 1000 of 1000 (CPU time: 3.273 s)
#> 
#> 
#> MCMC scan 100 of 1000 (CPU time: 0.432 s)
#> MCMC scan 200 of 1000 (CPU time: 0.774 s)
#> MCMC scan 300 of 1000 (CPU time: 1.144 s)
#> MCMC scan 400 of 1000 (CPU time: 1.499 s)
#> MCMC scan 500 of 1000 (CPU time: 1.836 s)
#> MCMC scan 600 of 1000 (CPU time: 2.221 s)
#> MCMC scan 700 of 1000 (CPU time: 2.566 s)
#> MCMC scan 800 of 1000 (CPU time: 2.940 s)
#> MCMC scan 900 of 1000 (CPU time: 3.320 s)
#> MCMC scan 1000 of 1000 (CPU time: 3.708 s)
#> 
#> ***** Running bnpmediation
#> 
  |                                                                       
  |                                                                 |   0%
  |                                                                       
  |                                                                 |   1%
  |                                                                       
  |=                                                                |   1%
  |                                                                       
  |=                                                                |   2%
  |                                                                       
  |==                                                               |   2%
  |                                                                       
  |==                                                               |   3%
  |                                                                       
  |==                                                               |   4%
  |                                                                       
  |===                                                              |   4%
  |                                                                       
  |===                                                              |   5%
  |                                                                       
  |====                                                             |   5%
  |                                                                       
  |====                                                             |   6%
  |                                                                       
  |====                                                             |   7%
  |                                                                       
  |=====                                                            |   7%
  |                                                                       
  |=====                                                            |   8%
  |                                                                       
  |======                                                           |   8%
  |                                                                       
  |======                                                           |   9%
  |                                                                       
  |======                                                           |  10%
  |                                                                       
  |=======                                                          |  10%
  |                                                                       
  |=======                                                          |  11%
  |                                                                       
  |=======                                                          |  12%
  |                                                                       
  |========                                                         |  12%
  |                                                                       
  |========                                                         |  13%
  |                                                                       
  |=========                                                        |  13%
  |                                                                       
  |=========                                                        |  14%
  |                                                                       
  |=========                                                        |  15%
  |                                                                       
  |==========                                                       |  15%
  |                                                                       
  |==========                                                       |  16%
  |                                                                       
  |===========                                                      |  16%
  |                                                                       
  |===========                                                      |  17%
  |                                                                       
  |===========                                                      |  18%
  |                                                                       
  |============                                                     |  18%
  |                                                                       
  |============                                                     |  19%
  |                                                                       
  |=============                                                    |  19%
  |                                                                       
  |=============                                                    |  20%
  |                                                                       
  |=============                                                    |  21%
  |                                                                       
  |==============                                                   |  21%
  |                                                                       
  |==============                                                   |  22%
  |                                                                       
  |===============                                                  |  22%
  |                                                                       
  |===============                                                  |  23%
  |                                                                       
  |===============                                                  |  24%
  |                                                                       
  |================                                                 |  24%
  |                                                                       
  |================                                                 |  25%
  |                                                                       
  |=================                                                |  25%
  |                                                                       
  |=================                                                |  26%
  |                                                                       
  |=================                                                |  27%
  |                                                                       
  |==================                                               |  27%
  |                                                                       
  |==================                                               |  28%
  |                                                                       
  |===================                                              |  28%
  |                                                                       
  |===================                                              |  29%
  |                                                                       
  |===================                                              |  30%
  |                                                                       
  |====================                                             |  30%
  |                                                                       
  |====================                                             |  31%
  |                                                                       
  |====================                                             |  32%
  |                                                                       
  |=====================                                            |  32%
  |                                                                       
  |=====================                                            |  33%
  |                                                                       
  |======================                                           |  33%
  |                                                                       
  |======================                                           |  34%
  |                                                                       
  |======================                                           |  35%
  |                                                                       
  |=======================                                          |  35%
  |                                                                       
  |=======================                                          |  36%
  |                                                                       
  |========================                                         |  36%
  |                                                                       
  |========================                                         |  37%
  |                                                                       
  |========================                                         |  38%
  |                                                                       
  |=========================                                        |  38%
  |                                                                       
  |=========================                                        |  39%
  |                                                                       
  |==========================                                       |  39%
  |                                                                       
  |==========================                                       |  40%
  |                                                                       
  |==========================                                       |  41%
  |                                                                       
  |===========================                                      |  41%
  |                                                                       
  |===========================                                      |  42%
  |                                                                       
  |============================                                     |  42%
  |                                                                       
  |============================                                     |  43%
  |                                                                       
  |============================                                     |  44%
  |                                                                       
  |=============================                                    |  44%
  |                                                                       
  |=============================                                    |  45%
  |                                                                       
  |==============================                                   |  45%
  |                                                                       
  |==============================                                   |  46%
  |                                                                       
  |==============================                                   |  47%
  |                                                                       
  |===============================                                  |  47%
  |                                                                       
  |===============================                                  |  48%
  |                                                                       
  |================================                                 |  48%
  |                                                                       
  |================================                                 |  49%
  |                                                                       
  |================================                                 |  50%
  |                                                                       
  |=================================                                |  50%
  |                                                                       
  |=================================                                |  51%
  |                                                                       
  |=================================                                |  52%
  |                                                                       
  |==================================                               |  52%
  |                                                                       
  |==================================                               |  53%
  |                                                                       
  |===================================                              |  53%
  |                                                                       
  |===================================                              |  54%
  |                                                                       
  |===================================                              |  55%
  |                                                                       
  |====================================                             |  55%
  |                                                                       
  |====================================                             |  56%
  |                                                                       
  |=====================================                            |  56%
  |                                                                       
  |=====================================                            |  57%
  |                                                                       
  |=====================================                            |  58%
  |                                                                       
  |======================================                           |  58%
  |                                                                       
  |======================================                           |  59%
  |                                                                       
  |=======================================                          |  59%
  |                                                                       
  |=======================================                          |  60%
  |                                                                       
  |=======================================                          |  61%
  |                                                                       
  |========================================                         |  61%
  |                                                                       
  |========================================                         |  62%
  |                                                                       
  |=========================================                        |  62%
  |                                                                       
  |=========================================                        |  63%
  |                                                                       
  |=========================================                        |  64%
  |                                                                       
  |==========================================                       |  64%
  |                                                                       
  |==========================================                       |  65%
  |                                                                       
  |===========================================                      |  65%
  |                                                                       
  |===========================================                      |  66%
  |                                                                       
  |===========================================                      |  67%
  |                                                                       
  |============================================                     |  67%
  |                                                                       
  |============================================                     |  68%
  |                                                                       
  |=============================================                    |  68%
  |                                                                       
  |=============================================                    |  69%
  |                                                                       
  |=============================================                    |  70%
  |                                                                       
  |==============================================                   |  70%
  |                                                                       
  |==============================================                   |  71%
  |                                                                       
  |==============================================                   |  72%
  |                                                                       
  |===============================================                  |  72%
  |                                                                       
  |===============================================                  |  73%
  |                                                                       
  |================================================                 |  73%
  |                                                                       
  |================================================                 |  74%
  |                                                                       
  |================================================                 |  75%
  |                                                                       
  |=================================================                |  75%
  |                                                                       
  |=================================================                |  76%
  |                                                                       
  |==================================================               |  76%
  |                                                                       
  |==================================================               |  77%
  |                                                                       
  |==================================================               |  78%
  |                                                                       
  |===================================================              |  78%
  |                                                                       
  |===================================================              |  79%
  |                                                                       
  |====================================================             |  79%
  |                                                                       
  |====================================================             |  80%
  |                                                                       
  |====================================================             |  81%
  |                                                                       
  |=====================================================            |  81%
  |                                                                       
  |=====================================================            |  82%
  |                                                                       
  |======================================================           |  82%
  |                                                                       
  |======================================================           |  83%
  |                                                                       
  |======================================================           |  84%
  |                                                                       
  |=======================================================          |  84%
  |                                                                       
  |=======================================================          |  85%
  |                                                                       
  |========================================================         |  85%
  |                                                                       
  |========================================================         |  86%
  |                                                                       
  |========================================================         |  87%
  |                                                                       
  |=========================================================        |  87%
  |                                                                       
  |=========================================================        |  88%
  |                                                                       
  |==========================================================       |  88%
  |                                                                       
  |==========================================================       |  89%
  |                                                                       
  |==========================================================       |  90%
  |                                                                       
  |===========================================================      |  90%
  |                                                                       
  |===========================================================      |  91%
  |                                                                       
  |===========================================================      |  92%
  |                                                                       
  |============================================================     |  92%
  |                                                                       
  |============================================================     |  93%
  |                                                                       
  |=============================================================    |  93%
  |                                                                       
  |=============================================================    |  94%
  |                                                                       
  |=============================================================    |  95%
  |                                                                       
  |==============================================================   |  95%
  |                                                                       
  |==============================================================   |  96%
  |                                                                       
  |===============================================================  |  96%
  |                                                                       
  |===============================================================  |  97%
  |                                                                       
  |===============================================================  |  98%
  |                                                                       
  |================================================================ |  98%
  |                                                                       
  |================================================================ |  99%
  |                                                                       
  |=================================================================|  99%
  |                                                                       
  |=================================================================| 100%
#> $Y11
#>    [1] 7.773675 6.993048 8.624499 7.481020 6.866863 6.530911 8.205857
#>    [8] 7.061465 8.044678 6.599416 6.523774 7.052494 7.763620 7.092487
#>   [15] 6.827072 6.932883 7.294010 7.402536 7.067454 7.334946 6.734727
#>   [22] 6.888365 7.440086 6.896419 7.221363 8.328016 6.997745 6.841911
#>   [29] 7.183612 6.327545 6.819923 6.469993 7.052804 6.782449 7.832731
#>   [36] 7.073042 6.988804 6.966057 7.151287 7.207013 6.928021 6.876416
#>   [43] 6.845423 6.671691 6.870281 6.868089 7.871114 7.532270 7.023381
#>   [50] 7.054410 7.354639 7.723036 7.550786 7.641947 7.104509 7.074449
#>   [57] 7.362249 7.649607 7.122009 7.295189 6.836744 6.785060 7.721237
#>   [64] 7.065443 6.960071 6.856241 6.888840 6.551076 6.989307 6.475585
#>   [71] 6.637302 6.715977 7.777858 7.564438 7.321250 7.170706 7.215503
#>   [78] 6.966251 6.970067 6.982928 7.052112 6.685796 7.284711 6.948182
#>   [85] 6.664429 6.881041 7.540977 6.952241 7.263533 7.103741 7.786308
#>   [92] 7.062915 8.287590 7.193470 7.386878 7.811139 6.745865 6.827538
#>   [99] 6.433352 7.077547 7.470062 7.443137 6.500155 6.933023 7.051367
#>  [106] 7.171999 6.961830 6.766189 6.894111 6.646019 6.963567 7.332654
#>  [113] 6.659979 7.291739 6.989772 7.592550 6.916920 6.653401 7.028746
#>  [120] 7.134488 8.151686 7.958638 7.212722 6.295014 6.511080 6.739869
#>  [127] 6.537393 6.898641 6.852085 6.818231 7.054588 7.663107 7.563959
#>  [134] 6.935537 7.019953 6.736165 7.137269 6.753123 7.410780 7.631499
#>  [141] 6.694399 8.050111 8.366753 7.232401 6.798149 6.955127 6.694074
#>  [148] 6.935203 6.825447 7.713388 7.096430 7.488563 7.659321 7.950820
#>  [155] 6.811398 6.615780 7.665870 6.826357 7.109326 7.684675 7.345140
#>  [162] 7.712625 7.112546 7.400998 7.588683 7.165350 7.867233 7.383884
#>  [169] 7.366373 6.696997 7.199291 6.938729 7.820240 7.645170 7.187298
#>  [176] 7.089403 7.744254 7.136230 6.980979 6.629376 6.853326 6.883924
#>  [183] 7.403668 6.621646 7.089875 7.030888 8.059002 7.879050 7.035875
#>  [190] 7.468648 8.031107 6.736316 7.360287 6.797137 6.367527 6.906319
#>  [197] 7.301243 6.765147 7.705439 6.945101 7.172712 6.498698 6.571090
#>  [204] 6.972871 7.681781 7.479021 7.408286 7.918718 6.882366 7.835942
#>  [211] 7.240087 6.868007 7.304146 7.887267 7.857803 7.943837 7.247616
#>  [218] 9.105090 7.391071 7.232010 9.086480 7.626725 6.696195 6.892108
#>  [225] 6.955847 7.233168 8.017392 7.068157 6.662228 6.977606 6.940878
#>  [232] 6.955426 6.884641 6.781602 6.548637 6.602060 6.843557 7.191160
#>  [239] 7.154595 7.094400 7.079782 7.312355 7.129986 7.088516 7.094298
#>  [246] 6.808450 6.339874 6.956202 7.366273 7.456405 6.902237 6.800497
#>  [253] 7.065377 6.630614 6.698606 7.726020 7.540357 7.776586 7.309934
#>  [260] 7.119641 7.592463 7.749829 7.122562 6.598967 6.935121 6.990734
#>  [267] 6.882635 7.158433 6.981848 8.367417 7.270226 7.389863 7.291926
#>  [274] 7.474401 7.362498 7.627564 7.377948 6.892284 6.509773 6.524307
#>  [281] 6.938061 7.213330 6.673066 6.830218 6.816391 6.818801 7.418034
#>  [288] 6.844225 7.072243 7.085264 6.635230 7.178549 7.793486 7.624941
#>  [295] 7.243705 7.775897 6.757770 7.211024 7.629646 6.422488 6.793587
#>  [302] 6.616362 6.905840 6.672316 6.787069 6.772656 6.518412 6.910979
#>  [309] 6.487190 7.301575 7.672304 6.740238 6.973936 6.537248 6.968981
#>  [316] 7.259951 7.811832 6.973119 7.343367 7.932576 7.626121 6.822110
#>  [323] 6.940033 6.866130 7.083628 7.311659 7.072382 7.505795 6.966517
#>  [330] 6.857284 7.081949 7.088575 6.854921 6.827440 7.086346 6.580764
#>  [337] 6.782769 6.840817 7.763892 7.637699 7.792326 9.103766 7.019944
#>  [344] 7.176734 6.743917 6.676663 6.939893 7.150784 6.436434 6.761987
#>  [351] 6.830529 6.691985 7.002390 7.845166 7.079568 6.687823 6.877492
#>  [358] 6.911390 7.119199 6.894470 7.060122 7.298292 6.986777 6.770445
#>  [365] 6.726139 7.230547 6.952726 7.463969 7.630539 6.953819 7.313608
#>  [372] 6.721147 7.094905 7.333882 7.099648 6.774037 6.974669 6.347847
#>  [379] 6.501327 6.637127 7.303418 7.541230 7.702958 6.993160 7.996939
#>  [386] 7.426971 7.817686 7.306467 7.094809 7.720268 6.881102 7.941977
#>  [393] 8.484485 6.939190 6.631196 6.903156 7.299885 6.664019 7.151875
#>  [400] 7.240689 7.152146 7.615455 7.480706 7.190567 7.002781 6.587207
#>  [407] 7.707821 7.568188 7.423006 6.626725 7.043238 7.857212 7.654918
#>  [414] 7.164688 7.373175 7.407217 7.445985 7.223233 7.552603 7.359087
#>  [421] 6.914750 6.669558 7.344145 7.084986 7.134536 6.797750 7.110225
#>  [428] 7.035000 7.411634 6.841539 7.066611 6.923465 6.566245 6.698794
#>  [435] 7.087307 6.867642 7.071023 6.686716 6.852866 7.215900 6.500505
#>  [442] 7.002610 7.380047 7.352709 7.648800 7.620404 7.360999 7.457208
#>  [449] 8.520533 7.364726 7.112849 7.035661 6.843302 7.296735 6.841409
#>  [456] 6.917166 6.874335 6.698309 7.567837 7.895895 7.027005 6.910646
#>  [463] 7.670304 7.772271 6.847500 6.990862 7.220974 7.734904 7.019930
#>  [470] 6.694371 6.949850 6.718034 7.219640 7.142378 6.927717 7.000957
#>  [477] 7.448232 6.807280 7.873821 7.122530 6.962374 7.898967 7.791275
#>  [484] 6.649076 6.952423 8.035597 7.395658 6.729295 6.561352 7.803202
#>  [491] 7.646185 7.622771 6.835874 6.927067 6.354892 6.316263 6.414939
#>  [498] 7.125370 6.984005 6.664807 6.608841 6.959035 6.344361 6.790003
#>  [505] 6.729269 7.153985 6.689187 6.728574 6.462129 7.174593 6.632621
#>  [512] 7.351308 7.849415 6.946050 6.716368 6.666855 6.895639 7.181399
#>  [519] 6.851630 6.946843 7.447002 6.895909 6.534252 7.120803 6.754108
#>  [526] 6.718094 6.694441 7.062498 7.297792 6.913705 6.927362 7.232492
#>  [533] 7.803843 8.452261 7.668234 7.059250 7.788484 6.733652 7.075230
#>  [540] 6.975059 6.574955 6.737802 7.031578 6.560848 6.525034 7.437329
#>  [547] 7.107544 7.109493 6.884279 7.150628 6.936238 6.434178 6.638905
#>  [554] 9.310885 8.048864 7.117916 6.833691 6.825911 7.136443 7.847917
#>  [561] 7.102652 6.823204 6.814392 7.009387 6.577049 6.945147 7.236196
#>  [568] 7.355542 6.761728 7.091379 7.830576 7.471873 7.751685 7.590113
#>  [575] 7.505020 7.083655 7.630955 7.217464 7.249208 7.124530 7.396304
#>  [582] 6.703606 6.780802 6.709021 7.046769 7.348807 7.372185 8.057786
#>  [589] 8.142077 6.956201 6.977935 6.647104 6.659226 7.225919 7.845615
#>  [596] 6.938136 6.680927 7.167303 6.816104 6.731677 6.971721 6.556555
#>  [603] 6.790351 6.763130 6.599151 7.462858 7.894746 7.262919 6.848325
#>  [610] 6.737212 6.727477 6.720964 6.559949 7.012214 6.707818 7.149090
#>  [617] 7.168250 7.033664 6.742855 6.958710 6.758736 6.871032 7.078749
#>  [624] 6.842615 7.449347 7.593280 7.085837 7.899775 7.273110 7.457483
#>  [631] 7.378911 7.801647 8.314701 7.604834 6.964851 6.512745 7.053832
#>  [638] 7.751636 7.690451 7.514632 7.698423 7.558380 7.930319 6.568632
#>  [645] 7.448089 7.645119 7.300637 7.281999 7.595367 7.205575 6.703362
#>  [652] 6.696777 6.811870 7.225066 6.899242 7.056525 6.934511 6.974060
#>  [659] 6.589444 6.653121 7.178537 7.072632 7.311815 7.055959 7.203259
#>  [666] 6.800452 6.894136 7.169515 7.522479 8.039468 7.329948 7.130078
#>  [673] 7.353075 8.829174 8.843065 7.352216 6.567817 7.103341 7.522707
#>  [680] 6.912625 7.934886 6.748421 6.781574 6.696497 7.421211 7.140330
#>  [687] 6.846535 7.065720 6.697908 7.210211 6.874946 7.328602 6.672074
#>  [694] 7.987365 7.146565 7.876483 7.153137 6.975016 6.496152 6.827531
#>  [701] 7.052489 7.765700 7.265492 7.169324 7.275093 7.086445 7.449105
#>  [708] 7.432243 6.993264 7.286355 6.961813 7.239357 7.198898 7.231424
#>  [715] 7.139748 7.034521 6.977987 7.143793 7.723539 6.746894 6.697212
#>  [722] 6.730855 6.546526 8.371927 6.857512 7.117103 7.509464 7.175118
#>  [729] 6.578131 6.817016 7.285623 7.487155 7.251765 6.848283 7.085146
#>  [736] 7.613751 7.025014 7.776691 7.224281 7.354887 7.051443 6.482014
#>  [743] 6.631644 6.959831 7.106100 6.852790 6.836386 6.717843 8.404492
#>  [750] 7.366016 7.403038 6.756423 7.421600 7.059103 6.593291 7.259171
#>  [757] 7.761386 7.570292 6.929315 6.828610 6.617779 6.944788 8.224893
#>  [764] 7.243087 7.140541 6.949968 7.779705 7.196549 8.197759 7.332103
#>  [771] 7.431086 7.252171 7.382179 6.803621 7.368051 7.178135 6.548322
#>  [778] 7.059404 6.849926 6.907300 6.867931 6.958434 6.566403 7.039871
#>  [785] 6.702990 6.959075 7.266296 7.469984 7.702411 7.925776 7.877035
#>  [792] 7.378743 7.171173 6.850787 6.453226 7.020134 6.855256 7.064054
#>  [799] 6.586436 6.806764 6.440416 7.048945 7.410327 7.219404 7.262425
#>  [806] 7.638446 7.139285 7.134531 7.609118 8.282219 7.114747 7.176487
#>  [813] 7.063397 7.916830 7.235873 7.347789 6.705613 6.851099 6.519579
#>  [820] 6.977587 6.714543 6.696795 7.321292 6.552497 7.079582 6.688195
#>  [827] 7.070752 6.993564 7.365078 6.967388 7.125666 6.895108 7.651166
#>  [834] 7.466643 7.064771 7.091904 7.868025 6.760389 6.780382 7.176462
#>  [841] 7.313387 7.176776 7.122262 7.396825 7.160457 7.002358 7.417490
#>  [848] 7.694113 7.230568 7.205979 7.323080 6.811915 6.681276 6.934278
#>  [855] 7.084038 7.267067 7.284940 7.778567 8.129461 7.116314 7.253182
#>  [862] 7.094905 6.879678 6.934654 6.690898 6.684993 7.595025 7.438817
#>  [869] 6.914738 7.070475 6.970860 6.786781 6.801368 7.416155 7.386941
#>  [876] 7.093992 6.737276 6.441030 6.818197 6.291027 6.893370 7.601153
#>  [883] 7.054569 6.737089 6.688633 7.297737 6.520070 6.718160 6.857455
#>  [890] 8.046756 6.749411 7.191356 7.011583 6.914087 6.504979 7.246912
#>  [897] 6.855135 6.709106 6.712481 7.779586 7.028198 7.160210 7.073518
#>  [904] 6.418391 6.826642 6.921275 8.318988 6.688432 6.722906 6.536334
#>  [911] 6.733586 7.005795 7.691462 6.662637 6.672198 7.073587 6.481127
#>  [918] 6.687015 6.583382 6.892979 6.918733 8.254890 7.485340 7.262523
#>  [925] 7.409641 7.255016 7.244960 6.997436 7.158149 6.971744 6.835649
#>  [932] 6.947363 6.856750 6.894780 6.611409 6.736569 7.109899 6.876805
#>  [939] 6.630869 7.145523 6.868581 6.928340 7.505460 7.608022 6.575040
#>  [946] 6.596073 7.096323 6.364231 7.587116 6.965383 6.612505 6.796084
#>  [953] 6.518908 6.796253 6.716029 6.608965 7.540961 6.634679 7.371143
#>  [960] 6.820785 8.012535 7.138201 6.610551 7.595268 7.011500 6.870174
#>  [967] 7.259014 6.908480 7.422939 6.820730 6.870641 7.013223 6.718734
#>  [974] 6.514376 6.814241 7.457872 7.195205 6.865625 6.750600 6.700279
#>  [981] 7.337165 6.671139 6.763794 6.575994 7.623664 7.500963 6.968622
#>  [988] 7.566725 6.731278 6.958346 7.280047 6.904240 7.000238 7.019967
#>  [995] 6.875924 7.844162 6.931375 7.242636 7.577927 7.870819
#> 
#> $Y00
#>    [1] 5.228763 4.978343 4.795500 4.711087 4.988305 4.695599 4.704602
#>    [8] 4.886986 5.208229 5.073764 4.731635 4.695943 4.868534 4.952728
#>   [15] 4.708619 4.989160 4.928166 5.132929 5.283567 5.223050 4.787690
#>   [22] 5.815131 4.977847 4.984932 5.063374 4.813045 4.902737 4.689526
#>   [29] 5.069739 5.070093 4.667069 5.206000 5.433653 5.610305 4.675753
#>   [36] 4.716392 5.995226 5.645892 5.872214 5.350029 4.792785 5.997640
#>   [43] 5.062407 4.920562 4.717853 5.142808 4.712093 5.016675 4.603550
#>   [50] 6.219019 4.733169 4.693046 4.889631 5.000393 4.894343 4.539436
#>   [57] 4.939910 4.945511 4.768618 4.970778 5.032438 5.278024 4.869921
#>   [64] 5.471998 4.764693 4.903612 5.412140 5.117263 5.067665 4.624221
#>   [71] 4.842438 5.081279 4.697374 4.547133 4.747898 4.882697 4.574871
#>   [78] 4.642264 4.780677 5.261282 4.940558 5.771699 4.873745 4.785553
#>   [85] 4.810997 5.040413 5.017274 5.313902 4.699598 5.262194 4.590946
#>   [92] 4.644121 4.602081 4.729970 4.712841 5.085022 5.073537 4.867733
#>   [99] 5.695504 5.984171 5.266083 4.725469 5.192948 4.868463 5.083285
#>  [106] 5.031887 4.924222 5.370063 4.878767 4.974337 4.739423 5.048475
#>  [113] 5.151718 5.540495 6.155799 4.758407 4.945121 4.910412 4.806277
#>  [120] 4.976519 5.334275 5.283002 4.906510 5.105955 5.031047 5.118751
#>  [127] 4.821777 4.832039 4.801659 4.739601 4.932952 4.722290 4.703196
#>  [134] 5.033448 5.071734 6.018100 5.167992 4.801575 4.873949 4.788408
#>  [141] 5.239348 5.248528 4.790149 4.673805 4.918661 4.718793 4.702497
#>  [148] 4.793360 5.042223 4.955576 4.764187 4.719893 4.734648 4.705789
#>  [155] 4.448710 4.574920 4.522700 4.866013 4.874424 5.017354 4.986910
#>  [162] 4.947374 4.972174 4.790603 4.953325 4.714893 4.859126 5.304368
#>  [169] 5.496255 5.615564 5.496460 5.043450 5.234402 4.721389 4.745623
#>  [176] 4.691316 4.850460 4.786614 4.684268 4.697341 4.803578 5.097281
#>  [183] 5.123509 4.591117 4.699328 4.523131 5.070577 4.638785 4.629121
#>  [190] 4.882680 5.015014 4.821675 4.669108 4.860609 4.527376 4.756038
#>  [197] 4.971295 5.001473 4.890345 4.956904 5.037768 5.154675 5.190009
#>  [204] 5.267029 5.163176 5.405445 5.016635 5.735791 5.563644 5.414415
#>  [211] 5.858907 4.752085 4.651214 4.898959 4.830082 4.799818 5.083100
#>  [218] 4.877661 4.957342 4.750219 4.733604 4.873178 4.971318 4.996313
#>  [225] 4.787850 4.885812 4.800785 4.837152 5.120840 4.832838 4.590855
#>  [232] 4.711489 4.839506 4.613555 5.009610 4.779558 4.972739 4.914391
#>  [239] 4.749411 4.719333 4.814599 4.654396 4.777725 4.842675 4.633005
#>  [246] 4.860926 4.765294 5.186814 4.786059 4.941680 5.022806 4.640625
#>  [253] 4.820532 4.776450 4.658563 4.841625 4.879563 4.768387 4.740598
#>  [260] 5.259896 5.055506 4.716257 4.792582 5.029962 5.064012 5.662825
#>  [267] 4.957248 5.236642 4.982162 4.896479 5.043073 5.091620 5.599968
#>  [274] 5.184900 4.946562 5.454700 4.870005 5.083741 4.915541 5.306217
#>  [281] 4.800317 4.738968 4.677865 4.634116 4.798282 4.751193 4.770225
#>  [288] 5.193505 5.115334 5.349785 4.795118 5.044086 4.730664 5.036537
#>  [295] 4.700285 4.706189 4.702134 4.537790 4.481516 4.629211 4.606468
#>  [302] 4.716881 4.821111 4.956918 4.807977 5.186362 4.830857 4.762370
#>  [309] 4.564702 4.775908 4.778375 5.139871 5.351923 5.115882 5.134148
#>  [316] 5.333992 4.876046 4.805299 5.875488 5.067545 5.424956 5.398577
#>  [323] 4.880991 4.961300 4.787113 4.953368 4.633546 4.849423 4.878149
#>  [330] 4.814262 4.801882 4.810402 5.032991 4.789375 5.133013 4.853800
#>  [337] 4.913226 4.957548 4.652761 4.588964 4.691683 4.752766 4.539047
#>  [344] 4.993729 5.240900 4.687135 4.850291 4.616165 4.618147 4.774626
#>  [351] 4.853328 4.757834 5.051969 5.038759 5.552345 5.055694 5.282895
#>  [358] 5.570683 5.112783 4.978329 4.974516 4.965480 4.755459 5.118469
#>  [365] 4.710918 5.137338 4.833107 4.928184 4.988581 4.736726 4.895772
#>  [372] 5.011253 4.945663 4.702966 4.568828 4.826328 4.696510 4.769584
#>  [379] 4.842161 5.459611 4.861328 5.339604 4.966900 4.884761 5.477168
#>  [386] 4.868025 5.175533 5.425587 5.769984 5.552272 4.990233 4.675525
#>  [393] 5.025961 4.963864 4.843521 4.934454 5.332031 5.004127 4.802619
#>  [400] 4.777344 4.630878 4.901784 4.957797 4.982082 5.199854 4.688683
#>  [407] 4.580847 4.855462 5.267948 4.716912 5.340627 4.759332 4.773163
#>  [414] 4.786715 4.731037 4.723643 4.736538 5.202386 4.761502 4.914568
#>  [421] 4.716210 5.023970 5.126767 5.272699 4.965433 4.686357 4.811462
#>  [428] 4.884423 4.696273 4.645883 4.855850 5.024743 4.956548 4.694979
#>  [435] 4.637332 4.952657 4.999030 5.303092 4.896077 4.829267 4.827749
#>  [442] 4.679662 4.763218 5.101035 4.720887 4.953242 4.674922 4.703021
#>  [449] 5.225706 5.522172 4.744499 4.779440 5.479824 5.148351 4.886465
#>  [456] 4.642266 4.622532 4.872811 4.745564 4.699042 4.793326 4.830295
#>  [463] 4.737164 4.569625 4.848345 4.553461 5.187376 5.403635 5.009103
#>  [470] 5.320901 5.268076 4.798819 5.542282 5.718356 5.226199 4.762522
#>  [477] 4.903041 4.859782 4.600965 4.645875 4.719972 4.805870 5.092316
#>  [484] 4.752715 4.930552 5.319474 4.795445 5.090804 4.645901 5.002374
#>  [491] 5.337900 5.170676 5.557663 5.163566 4.830845 5.198031 4.852431
#>  [498] 5.193961 4.907244 5.498198 4.797392 5.591491 5.004709 5.316462
#>  [505] 5.294418 5.405935 4.833015 5.101307 5.091800 4.841746 5.300382
#>  [512] 4.806909 5.991806 4.815492 5.195570 5.265888 4.959754 4.754702
#>  [519] 4.688692 4.984762 5.138279 5.283606 5.079702 5.763983 5.657048
#>  [526] 5.211637 4.745398 4.990523 5.048839 5.735975 5.560643 5.346823
#>  [533] 5.248508 5.063574 4.923953 4.948827 5.254874 5.070889 4.682984
#>  [540] 4.766332 4.730647 4.859912 4.792917 5.189160 4.789586 4.608625
#>  [547] 5.255053 4.682590 4.746587 4.820643 4.829273 4.656487 4.545821
#>  [554] 5.224816 4.931224 4.586746 4.793645 4.746977 5.018805 4.745095
#>  [561] 4.980095 4.983336 4.813887 4.780358 5.166667 4.905687 4.907415
#>  [568] 4.872616 4.830507 5.280446 4.880379 5.087728 4.982066 4.714570
#>  [575] 4.842921 4.937308 4.826836 5.025969 5.271644 5.195389 4.700655
#>  [582] 4.869237 4.967177 5.143798 5.159215 5.592567 4.796729 4.806130
#>  [589] 4.949450 5.317975 5.048924 4.724582 4.661483 4.822561 4.897943
#>  [596] 4.739901 5.097155 4.792634 4.979123 4.982072 4.821560 5.104040
#>  [603] 4.890146 4.716108 4.588605 4.542125 4.779534 4.586928 4.824719
#>  [610] 4.879289 4.958780 4.687936 4.760020 4.717280 5.176647 5.217501
#>  [617] 4.795291 4.672888 4.709307 4.709346 4.804434 4.819149 4.709157
#>  [624] 5.178495 4.844859 5.018393 5.511274 5.206767 4.648112 4.729819
#>  [631] 5.085300 5.106607 5.214671 5.279525 4.863901 4.620257 4.621350
#>  [638] 4.760851 4.785734 4.889072 5.359398 4.861395 5.029709 4.652041
#>  [645] 4.760501 4.604677 4.939318 5.455372 4.783328 4.669595 5.097085
#>  [652] 4.873543 5.048541 5.188844 4.593088 4.694136 4.913540 5.132404
#>  [659] 5.088786 5.149201 5.686508 4.988192 5.777838 4.994654 5.197493
#>  [666] 5.342350 5.115528 4.793126 5.017826 4.920508 4.949253 4.800155
#>  [673] 4.871199 4.856307 4.916939 5.133349 5.020743 5.690765 5.021588
#>  [680] 4.743248 4.637926 5.079745 4.902824 4.926348 4.941388 5.066538
#>  [687] 6.061166 5.544989 6.132555 5.552639 5.087329 5.091180 5.254058
#>  [694] 4.770663 4.497678 4.521148 4.595386 4.642898 4.616698 4.825961
#>  [701] 4.734588 5.037115 4.978509 4.980794 4.841910 8.035920 4.965777
#>  [708] 5.945386 4.898271 4.929872 4.816526 4.895585 4.865528 4.556079
#>  [715] 4.707772 5.091471 5.192264 4.816680 4.742181 4.947286 4.817636
#>  [722] 4.973482 4.710873 4.867415 4.975422 4.659297 4.736178 4.884995
#>  [729] 4.743216 4.936425 4.811440 4.813031 4.738220 4.798793 4.662817
#>  [736] 4.549470 4.916183 4.858720 5.283700 4.625472 4.943701 4.795005
#>  [743] 4.638021 4.541629 5.208172 4.537210 4.865575 5.076079 4.735425
#>  [750] 4.633112 4.689204 4.732006 4.714189 5.122197 4.778845 5.701975
#>  [757] 4.686210 4.874890 4.694133 4.960521 4.751901 4.732551 4.714777
#>  [764] 4.959277 5.421952 4.681750 4.839475 4.916513 5.008378 5.654726
#>  [771] 5.140690 5.230206 4.914545 4.941742 4.669350 4.545210 4.752491
#>  [778] 5.206324 4.889337 4.779193 4.724378 4.913250 4.887144 4.632401
#>  [785] 5.993313 5.268928 5.404651 5.644494 4.979346 5.176312 5.249589
#>  [792] 4.844722 4.765196 4.919343 5.158777 4.945692 4.883167 4.710155
#>  [799] 5.027397 5.066536 4.769843 4.671607 4.812860 4.533153 4.496338
#>  [806] 4.815450 4.636432 4.673706 4.786130 4.772965 4.886042 4.639498
#>  [813] 4.706123 4.762228 4.609312 4.818615 5.341116 4.902594 4.905838
#>  [820] 4.841029 5.330641 4.915683 4.674252 4.546291 4.553654 4.748080
#>  [827] 4.724705 4.577132 4.578409 4.921739 5.151469 5.023109 5.281964
#>  [834] 4.980044 4.977240 5.084393 4.880006 4.869186 4.677458 4.701002
#>  [841] 4.695450 5.017959 5.015533 5.020478 4.997203 5.248994 5.071998
#>  [848] 5.506294 4.851917 5.452370 5.194445 5.136535 5.074107 4.802104
#>  [855] 4.835274 5.048402 5.010951 5.128694 5.427576 4.816115 4.806195
#>  [862] 5.156572 5.022147 5.129236 4.821058 4.791568 5.034344 4.891750
#>  [869] 4.844616 4.604878 4.972339 4.774369 4.901014 4.687501 4.689462
#>  [876] 4.856326 4.771279 4.775940 4.904365 4.595666 4.801538 4.779864
#>  [883] 5.053905 5.373475 4.845289 4.775161 4.871406 4.929036 4.716728
#>  [890] 5.169673 4.910219 5.231293 5.137657 5.203749 4.923512 5.342335
#>  [897] 5.074064 4.851720 5.085887 5.280697 5.101608 4.737978 4.782492
#>  [904] 4.759765 4.702328 4.459059 4.479113 4.520861 4.522885 4.553945
#>  [911] 4.562790 4.614142 4.457084 4.620940 4.640579 4.643019 4.693535
#>  [918] 4.641197 4.749947 4.685121 4.810003 4.892378 4.866342 4.892809
#>  [925] 5.008690 5.183794 5.148454 6.598572 5.226396 4.658101 4.886322
#>  [932] 4.967458 4.785457 4.952332 4.832508 4.688243 4.866875 4.834215
#>  [939] 5.070860 4.826424 4.920745 5.043888 4.786705 4.723389 4.648971
#>  [946] 4.892320 4.680344 4.903799 4.620268 5.121947 4.657650 5.061350
#>  [953] 5.464482 5.841249 4.972082 5.003123 5.126240 5.106305 4.608487
#>  [960] 4.619377 4.815398 4.723334 4.966068 5.126282 4.881898 4.823340
#>  [967] 4.769456 4.660539 4.754073 4.936202 5.042556 4.900983 5.066827
#>  [974] 5.050705 4.710723 4.996137 5.573895 4.583561 4.754173 4.875171
#>  [981] 4.760085 4.860151 4.649824 4.671671 4.909889 4.718907 5.215605
#>  [988] 4.841976 4.679359 5.001945 4.843078 5.049004 5.172001 4.851154
#>  [995] 4.757447 4.927041 4.844887 4.806506 4.657697 5.784349
#> 
#> $Y10
#>    [1]  9.582372  8.286290  9.547317 11.190942 11.748658 10.778729
#>    [7] 12.227235  8.973460 11.087169  9.082510 10.915588  8.774650
#>   [13]  7.020994  8.401154  9.916690 10.660697  7.366386  9.294274
#>   [19]  9.000707  7.978389  7.722928 10.709353 10.531579 10.592698
#>   [25] 10.399745  9.020707  7.595816 10.575430  9.293662  3.629424
#>   [31]  9.960092  9.754651 10.553037  8.106724 10.685647  8.517302
#>   [37]  6.773453 11.069569  6.229006  8.764365  8.359807  9.635109
#>   [43]  8.305141  9.337377  9.445274  8.724299 10.461328  9.522269
#>   [49] 10.643082  9.965671 10.696860 11.590481  9.082492 10.588725
#>   [55] 10.938043  9.055285 12.111870  9.426079 11.272755 11.805392
#>   [61]  8.138697 11.402237 12.412215  9.566931 11.222404  9.562958
#>   [67]  8.916800 12.379583  9.301297 10.461059  6.617481  9.552772
#>   [73] 10.264576 11.456695 10.405926  9.929612  8.558682 10.024393
#>   [79]  7.995365 10.972733 11.143730  7.818277 11.072307  8.307418
#>   [85] 10.132246  9.260636  8.149857 10.355323 10.443782 10.399372
#>   [91] 10.214604 10.088316  6.944972 11.622931 10.117797  8.791238
#>   [97]  8.875623 10.814692  6.577234 12.216453  7.560324 10.370006
#>  [103]  7.585143 10.584420  6.567620  8.295046  9.436773  8.145721
#>  [109]  8.641151  7.790173 10.115991 10.485215  7.940852  9.677836
#>  [115] 11.120864 11.384471  8.770543 10.531239  9.311559 11.311988
#>  [121] 11.254332  8.363629 11.042978  9.179573 10.695704  6.521764
#>  [127]  7.865698  5.420796  8.425371 11.589800  7.824765  9.494724
#>  [133]  9.294597  8.577675  8.260006  7.541137  6.653711 10.504514
#>  [139]  9.649055 10.275775 12.144514 11.648650 10.737740 11.454838
#>  [145]  6.786957 10.570939 10.841263  9.062783 11.053177  8.501576
#>  [151]  9.605578  7.333423  8.625032  7.164481 10.872096  9.477885
#>  [157]  9.981516  9.967387 11.084818 10.193455 11.314897 10.034729
#>  [163]  9.666407 10.292139 10.913602 11.150576  9.805234 10.586547
#>  [169]  9.519855  7.164203  9.615080 10.900794 11.800434 11.518086
#>  [175] 11.168387  9.055596 11.170210 10.100200  8.882482 10.394316
#>  [181] 10.906444  8.975278 12.964388 11.089046 10.893325 10.168532
#>  [187] 10.977718  7.624408 10.782742 10.870000 10.214041  9.507359
#>  [193] 11.483030 10.688698 10.197375  8.408369  9.603503 10.059135
#>  [199]  9.309589  8.714336 10.029903  8.851802  9.034916  7.602402
#>  [205]  8.837038  6.707891  8.921026 10.845232 11.008175  8.571035
#>  [211] 10.563664  9.582110 13.102078 10.280545 10.824263  8.867512
#>  [217]  7.352418  6.189613 10.230285  7.470126  9.766444  9.462863
#>  [223]  9.681642 11.815666  7.555394 10.005245 10.194760 15.303544
#>  [229]  8.527175 11.136634  9.689032  8.893326 10.051824 10.657505
#>  [235]  5.408484 10.798680 10.406409 10.056774  8.181411  7.714264
#>  [241]  9.079617 10.388285  8.876086 10.907395 10.470624 10.700959
#>  [247]  9.038477  8.781198  9.962129 10.186688 11.030059  8.117704
#>  [253] 10.253378 12.045678  9.505074  8.554474 11.162925  8.914834
#>  [259]  7.669477  9.173339  9.935810 10.471819 10.243432  9.426597
#>  [265]  8.331944  9.848430 10.721048  8.657681  9.336199 10.846157
#>  [271] 10.358214 10.990856  9.724527 11.894517  9.429313  7.271822
#>  [277] 10.524780  8.509679 10.060365 10.564033 11.411862  9.127453
#>  [283]  8.440427 10.695475 11.163771 10.540046  7.750688  7.713395
#>  [289]  8.339439 10.800680  8.846456  9.043539  7.837468  9.385964
#>  [295] 10.707187 10.248697  9.403196 11.865466  7.768959  9.540433
#>  [301]  8.066960  9.580187 10.848519 10.350464 11.703424 12.591846
#>  [307]  9.457546  9.012733  9.149874 10.583257  8.958535  7.790350
#>  [313]  9.580330  9.546636  9.896671 12.482522 10.418034 11.400816
#>  [319]  9.240365  8.370530  9.305134  8.774338  8.896532 10.954466
#>  [325] 10.317170  9.691758 10.432773 11.429942  8.299824 11.026284
#>  [331]  9.770888  8.180709  9.870354 10.876971  6.393680  8.934209
#>  [337]  7.518748 10.966831  8.113246 10.131153  9.037511  6.793585
#>  [343] 11.125907  8.154456  8.357230  9.843490  9.681644  9.047039
#>  [349]  9.032779 10.247075  9.962503  9.268571  7.799094 10.887234
#>  [355]  6.739053  6.112093  6.582805  6.989508 10.782627  8.002105
#>  [361]  8.802151  8.853082 11.564834  9.040358  9.667341 10.085897
#>  [367] 10.334362 10.696227  9.777783  8.340859 12.929431 11.344447
#>  [373] 11.658443 11.516429  9.803837 10.186097 10.192716  9.756327
#>  [379]  7.239149  6.405822  7.820445  8.546779  8.785556  8.375088
#>  [385]  8.568859  9.022946  9.010954  9.574121  6.999321 10.336127
#>  [391] 10.078939  9.886955  8.638047 10.912848 10.896979  9.632776
#>  [397] 10.172920  9.562134  9.522656 10.588303  9.722882  9.716238
#>  [403] 11.939144  9.410769  9.192738  9.939576  8.676590  9.534435
#>  [409]  8.839670  9.811224 11.294276 10.573066  9.950840 11.496780
#>  [415] 10.401356 10.780372  9.848957 10.706655 10.076252  9.477387
#>  [421]  9.425822  9.886235  9.503002 11.160018 10.952532 10.405124
#>  [427]  8.686444 11.871353 11.569956 11.815767  8.343251 10.063588
#>  [433]  7.613972  9.404316  9.266110  9.510547 11.268717  9.802148
#>  [439]  6.385772  8.651518 10.587486 11.234224 10.266094  9.457338
#>  [445] 11.329676  8.244442 10.342349  7.285047  7.385403 10.832893
#>  [451]  8.718049  7.440722  7.927029  8.597894 10.483066 10.627138
#>  [457] 10.724898  8.787222  9.781421 10.614836  9.671304 10.790688
#>  [463]  9.515575  9.706967  8.673373 11.543026 10.048596 11.452184
#>  [469]  9.585644 10.266567  8.726318  9.712607 10.931177  8.560162
#>  [475] 10.464191 10.425541 10.415494 10.642113  9.456079 11.018899
#>  [481]  8.688760 11.579652 10.123593 10.557773  9.869883 10.661418
#>  [487]  8.275461  9.608211 10.029295  8.269619  9.962669  9.156719
#>  [493] 11.099421  8.871184  7.889361  3.893565 10.573862 10.296199
#>  [499] 10.839472  6.992510  9.207721  6.334472  7.383074 10.581140
#>  [505]  8.323460  8.356589  8.579468  8.933690 10.594139  8.541014
#>  [511]  7.322568 10.979325  8.868419 10.582604  7.770583 10.453589
#>  [517] 10.173494  9.957935 11.424079  9.468986 10.237143  9.248863
#>  [523] 11.822529  9.883460  9.611059  9.541665  7.180021  9.099278
#>  [529]  8.606619 10.012372  7.227394  9.467664  7.960527  8.118650
#>  [535]  9.427206  9.287545  8.640385 10.163703  6.554070  7.797298
#>  [541]  6.744845  8.223011 10.243911  9.013752  6.123270 11.367282
#>  [547]  9.986551 11.101546 12.697445 10.917715  8.179364  9.470735
#>  [553]  9.514366  8.122712 10.970969 10.447082  9.483897 10.800541
#>  [559] 10.974213  9.989949  9.806156  8.614706 11.015560 10.462441
#>  [565]  8.218402  9.777896 11.469187 12.643978 10.628014  9.925406
#>  [571] 11.729759 10.298827  9.058536 10.646162 11.000715  9.987201
#>  [577]  8.487232 10.346519  9.463414 11.035217 10.197552  9.529258
#>  [583] 10.633720 10.756015 10.396271 11.799367 11.392673  8.398155
#>  [589]  9.255729  7.162383  8.564856 10.656210  7.832573 11.302059
#>  [595]  9.952116  8.301494  6.918291 11.804012 11.425416  9.146381
#>  [601] 11.419850  9.402058 12.727209 10.875944 10.821421 10.476206
#>  [607] 11.585363 11.323725 10.695583  8.258162  8.279495  9.450324
#>  [613]  9.331741 10.139530  7.675778  9.913848 10.785649  8.794736
#>  [619] 10.119353  9.844361 12.125773  9.306974  9.473320 10.191739
#>  [625]  7.053519  7.846051  8.893620  9.037062  8.449898 12.066126
#>  [631] 10.724109 10.224691  7.288141 10.489789 10.012738 11.104942
#>  [637] 11.019873 12.200977 12.958395 10.173121 12.354731 12.808078
#>  [643]  9.454838 12.070499  9.154633 10.245075  8.757394 10.323241
#>  [649] 10.025285  9.665167 10.026746  9.677780  7.708918 10.625854
#>  [655] 10.932548  9.347437  9.537789  9.934260  8.754576 11.024092
#>  [661] 11.341267  9.151015 10.039570  9.835229 10.651903  9.964715
#>  [667]  9.845513 10.755526  9.960750 10.040664  9.017961 12.199566
#>  [673] 11.445572  8.217396  9.013253  9.674078  8.363110  8.636267
#>  [679]  8.083588  9.049505  8.648381  8.752221 10.154460  8.899370
#>  [685] 12.424942  8.163367  8.386173 11.054278  8.804724 10.089049
#>  [691]  9.259822  8.939944  6.176380  9.603678  8.999121 10.429693
#>  [697] 11.089872  9.577552 10.376345  8.032568  7.432838 11.907707
#>  [703]  9.709681  6.708350 10.929303  9.165371  9.401676  9.349923
#>  [709] 10.777345 11.840427  9.270257 11.277747  9.952448 11.294559
#>  [715] 11.412411  8.921875 10.454634 10.138525 11.684658 10.412604
#>  [721]  9.318036  8.913441  6.193079  9.594224  8.056908  9.030482
#>  [727]  8.848375 10.875166  9.966881  9.602231 11.148074 10.410291
#>  [733] 12.224741  9.879649  9.510502 11.429014 11.413868 11.909807
#>  [739]  8.924805 11.554712 10.500328  7.784206  9.543233 10.466921
#>  [745] 10.576705 10.773581 11.112288  8.312534 11.706475 10.995869
#>  [751] 10.611369  8.793532 10.685432  8.569555  8.983106  9.806364
#>  [757] 10.595757 10.780851 10.425684  9.563714 10.322788 10.511517
#>  [763]  9.694556 11.968079  9.258876 11.655012 12.021388 10.332193
#>  [769]  9.466751 12.233156 10.501295  9.848594  8.132747  9.458487
#>  [775]  8.309760  9.416586  8.950407 10.369366  9.722050  9.829364
#>  [781]  9.317953  8.070696  8.895186 10.154974  5.896469  5.274603
#>  [787] 10.557813 10.662636 10.755070  9.891990 10.209733 11.143043
#>  [793] 10.580617 10.849453  9.483360 10.636180 10.509973  9.559786
#>  [799]  9.272337 10.907532  8.771167  9.195591  9.367227  8.089169
#>  [805] 11.269514 10.484806 10.843433 10.565195 12.198548  9.442001
#>  [811] 10.677962  9.247010 11.702200 10.550737 11.385645  9.680785
#>  [817]  9.944766 11.344780  7.985588  9.913736  8.375597  9.144176
#>  [823] 10.413793 10.962041 12.558761  9.540976 11.586332 11.337177
#>  [829] 10.649039  9.440905  8.433637 12.456596 10.898577 12.001869
#>  [835] 10.285121 11.021671  9.836067  9.804145  9.137375  9.239404
#>  [841]  8.635301 10.611035  7.716600  7.966339 10.857916  9.370972
#>  [847] 11.119622  8.429634 11.086868  9.848494 10.420813  8.906719
#>  [853]  9.306560 10.605568 12.001903 12.480325 10.682390 11.618913
#>  [859] 11.432581  8.813549 12.378291  9.318949 10.175568  6.461989
#>  [865]  9.247761  5.319413  6.584992 10.657908 10.995878  8.415953
#>  [871] 11.994551 11.194957  8.137202  9.128034  9.241652  9.902063
#>  [877] 11.329928  8.787100  9.670600  7.208586  8.773652  9.240583
#>  [883] 10.289809  8.656719  8.904266 10.245809 10.358748  9.544136
#>  [889] 10.283122 10.233617  7.529640 10.021925  9.863273 11.161187
#>  [895] 10.584682  9.172508  8.730139 11.031454  6.766914  8.810735
#>  [901] 11.198958  9.212397 11.000014  9.936482 11.223078 11.207464
#>  [907]  9.899798  9.940672 10.217330 10.278334 11.547793 11.287880
#>  [913] 10.503207  9.177424  9.364388 10.010535  9.402282  7.091392
#>  [919] 10.464028  7.781170 10.353617 10.879603 12.389784  9.908369
#>  [925]  8.908058 11.281745 10.735878  9.867883  9.995849  8.695572
#>  [931]  9.553404  9.172282  9.476889  9.061542  8.256502  9.987123
#>  [937]  9.248848 10.667025 10.655567 10.669765  7.154433  7.352040
#>  [943] 10.496828 10.635179 10.357846 11.088428 10.307637  8.676489
#>  [949] 12.177664 10.002981  9.247853  8.827718  8.440090  8.756152
#>  [955]  7.044217 10.058607 10.468815  8.731020 11.170453  9.334457
#>  [961] 11.535732  9.667567  9.063004  8.014067  6.558392 11.172422
#>  [967] 12.028850 10.423596  9.306574  8.786613  9.748520 11.354558
#>  [973]  8.787617  8.327121  8.373705  9.333729  7.711995  9.174361
#>  [979]  8.080680 10.656454  7.807798  7.655317 12.270739 10.300398
#>  [985] 10.944739 11.072560 10.930406 10.175985 10.282994 10.014402
#>  [991] 11.311171  5.962929  7.617423 11.258817 11.326268  7.695156
#>  [997]  9.102228  7.377585 10.251228 10.189969
#> 
#> $ENIE
#> [1] -2.595363
#> 
#> $ENDE
#> [1] 4.776552
#> 
#> $ETE
#> [1] 2.181189
#> 
#> $TE.c.i
#> [1] 1.218090 3.251656
#> 
#> $IE.c.i
#> [1] -5.023691  0.483747
#> 
#> $DE.c.i
#> [1] 1.403014 7.336307
#> 
#> $call
#> bnpmediation(dataTreatment = dataTreatment, dataControl = dataControl, 
#>     prior = prior, mcmc = mcmc, state = state, status = TRUE, 
#>     na.action = na.action, q = 2, NN = 10, n1 = 10, n0 = 10, 
#>     extra.thin = 0, seed = 12345)
#> 
#> attr(,"class")
#> [1] "bnpmediation"
```
