
#' Summary Methods For Objects in the Class \code{mixedAn} (Mixed Analysis)
#' 
#' Prints a summary table for the results of the mixed analysis for the
#' economic evaluation of a given model.
#' 
#' @param object An object of the class \code{mixedAn}, which is the results of
#' the function \code{\link{mixedAn}}, generating the economic evaluation of a
#' set of interventions, considering given market shares for each option.
#' @param wtp The value of the willingness to pay choosen to present the
#' analysis.
#' @param ...  Additional arguments affecting the summary produced.
#' @return Produces a table with summary information on the loss in expected
#' value of information generated by the inclusion of non cost-effective
#' interventions in the market.
#' @author Gianluca Baio
#' @seealso \code{\link{bcea}},
#'          \code{\link{mixedAn}}
#' @references
#' Baio, G. and Russo, P. (2009). A decision-theoretic framework for
#' the application of cost-effectiveness analysis in regulatory processes.
#' Pharmacoeconomics 27(8), 645-655 doi:10.2165/11310250
#' 
#' Baio, G., Dawid, A. P. (2011). Probabilistic Sensitivity Analysis in Health
#' Economics. Statistical Methods in Medical Research
#' doi:10.1177/0962280211419832.
#' 
#' Baio G. (2012). Bayesian Methods in Health Economics. CRC/Chapman Hall, London.
#' @keywords Health economic evaluation Mixed analysis
#' 
#' @examples
#' 
#' # See Baio G., Dawid A.P. (2011) for a detailed description of the 
#' # Bayesian model and economic problem
#'
#' # Load the processed results of the MCMC simulation model
#' data(Vaccine)
#' 
#' # Runs the health economic evaluation using BCEA
#' m <- bcea(e=e,c=c,          # defines the variables of 
#'                             #  effectiveness and cost
#'       ref=2,                # selects the 2nd row of (e,c) 
#'                             #  as containing the reference intervention
#'       interventions=treats, # defines the labels to be associated 
#'                             #  with each intervention
#'       Kmax=50000            # maximum value possible for the willingness 
#'                             #  to pay threshold; implies that k is chosen 
#'                             #  in a grid from the interval (0,Kmax)
#' )
#'
#' mixedAn(m) <- NULL      # uses the results of the mixed strategy 
#'                         #  analysis (a "mixedAn" object)
#'                         # the vector of market shares can be defined 
#'                         #  externally. If NULL, then each of the T 
#'                         #  interventions will have 1/T market share
#'
#' # Prints a summary of the results
#' summary(m,         # uses the results of the mixed strategy analysis 
#'         wtp=25000) #  (a "mixedAn" object)
#'                    # selects the relevant willingness to pay 
#'                    #  (default: 25,000)
#' 
#' @export
#' 
summary.mixedAn <- function(object,
                            wtp = 25000,...) {
  if (max(object$k) < wtp) {
    wtp <- max(object$k)
  }
  if (length(which(object$k == wtp)) == 0) {
    stop(
      paste(
        "The willingness to pay parameter is defined in the interval [0-",
        object$Kmax,
        "],
                     with increments of ",
        object$step,
        "\n",
        sep = "")
    )
  }
  
  n.digits <- 2
  n.small <- 2
  cat("\n")
  cat(paste(
    "Analysis of mixed strategy for willingness to pay parameter k = ",
    wtp,
    "\n",
    sep = ""))
  cat("\n")
  cat(
    paste(
      "Reference intervention: ",
      object$interventions[object$ref],
      " (",
      format(100 * object$mkt.shares[object$ref],
             digits = n.digits,
             nsmall = n.small),
      "% market share)\n",
      sep = "")
  )
  if (object$n_comparisons == 1) {
    text.temp <-
      paste(
        "Comparator intervention: ",
        object$interventions[object$comp],
        " (",
        format(
          100 * object$mkt.shares[object$comp],
          digits = n.digits,
          nsmall = n.small),
        "% market share)\n",
        sep = "")
    cat(text.temp)
  }
  
  if (object$n_comparisons > 1) {
    text.temp <-
      paste(
        "Comparator intervention(s): ",
        object$interventions[object$comp[1]],
        " (",
        format(
          100 * object$mkt.shares[object$comp[1]],
          digits = n.digits,
          nsmall = n.small),
        "% market share)\n",
        sep = "")
    cat(text.temp)
    for (i in 2:object$n_comparisons) {
      cat(paste(
        "                          : ",
        object$interventions[object$comp[i]],
        " (",
        format(
          100 * object$mkt.shares[object$comp[i]],
          digits = n.digits,
          nsmall = n.small),
        "% market share)\n",
        sep = ""))
    }
  }
  cat("\n")
  cat(paste(
    "Loss in the expected value of information = ",
    format(
      object$evi.star[object$k == wtp] - object$evi[object$k == wtp],
      digits = n.digits,
      nsmall = n.small),
    "\n",
    sep = ""))
  cat("\n")
}

