#' Minimizer of the parallel vorob criterion
#'
#' Minimization of the Vorob'ev criterion for a batch of candidate sampling points.
#'
#' @param lower vector containing the lower bounds of the design space.
#' @param upper vector containing the upper bounds of the design space.
#' @param optimcontrol optional list of control parameters for the optimization of the sampling criterion. The field "\code{method}" defines which optimization method is used: it can be either \code{"multistart"} (default) for an optimization with \code{L-BFGS-B} multistart, or \code{"discrete"} for an optimization over a specified discrete set, or \code{"genoud"} for an optimization using the genoud algorithm. (See details)
#' @param batchsize number of points to sample simultaneously. The sampling criterion will return batchsize points at a time for sampling.
#' @param integration.param optional list of control parameter for the computation of integrals, containing the fields \code{integration.points}: a p*d matrix corresponding to p integration points. \code{integration.weights}: a vector of size p corresponding to the weights of these integration points, and \code{alpha}: the Vorob'ev threshold.
#' @param object an object of class \code{\link[GPCsign]{gpcm}} .
#' @param new.noise.var optional scalar value of the noise variance of the new observations. Default is 0.
#' @param seed to fix the seed.
#'
#' @return a list with components:
#'  \item{par}{the best set of parameters found.}
#'  \item{value}{the value of the Vorob'ev criterion at \code{par}.}
#'  \item{allvalues}{if an optimization on a discrete set of points is chosen, the value of the criterion at all these points.}
#'  \item{current.vorob}{current vorob’ev deviation.}
#'  \item{alpha}{the Vorob'ev thresold.}
#'
#' @import future.apply
#' @import rgenoud
#' @import GPCsign
#' @importFrom KrigInv vorob_threshold
#'
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Chevalier, C. Fast uncertainty reduction strategies relying on Gaussian process models PhD Thesis. University of Bern (2013).
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
#' @details
#' If the field \code{method} is set to \code{"genoud"}, one can set some parameters of this algorithm:
#'   \code{pop.size}  (default: 50*d),  \code{max.generations} (10*d), \code{wait.generations} (2),  \code{BFGSburnin} (2) and the mutations \code{P1}, \code{P2}, up to \code{P9} (see \code{\link[rgenoud]{genoud}}). Numbers into brackets are the default values.
#' If the field \code{method} is set to \code{"discrete"}, one can set the field \code{optim.points}: p * d matrix corresponding to the p points where the criterion will be evaluated. If nothing is specified, 100*d points are chosen randomly.
#' Finally, one can control the field \code{optim.option} in order to decide how to optimize the sampling criterion.
#' If \code{optim.option} is set to 2 (default), \code{batchsize} sequential optimizations in dimension d are performed to find the optimum.
#' If \code{optim.option} is set to 1, only one optimization in dimension \code{batchsize*d} is performed. This option is only available with \code{"genoud"}. This option might provide more global and accurate solutions, but is a lot more expensive.
#'
#'
#' @examples
#' \donttest{
#' #-------------------------------------------------------------------
#' #------------------- max_vorob_optim_parallel_gpc-------------------
#' #-------------------------------------------------------------------
#'
#' ## 20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#' require(DiceKriging)
#' fx <- apply(x, 1, branin)
#' f <- ifelse(fx < 14, -1, 1)
#' Xf <- as.matrix(x)
#'
#' require(future) # load future package for parallelization
#' plan(multisession)  # activate parallel calculations (with available cores automatic detection)
#' ## gpcm object
#' require(GPCsign)
#' model <- gpcm(f, Xf, coef.m = -1.25, coef.cov = c(1.17,0.89))
#'
#' # parameters for max_vorob_parallel_gpc function
#' lower <- rep(0,d)
#' upper <- rep(1,d)
#' batchsize = 1
#' integration.param <- list()
#' require(randtoolbox)
#' nb.integration <- d*100
#' integration.points <- sobol(n = nb.integration, dim = d, scrambling = 0)
#' integration.param$integration.points <- rep(upper-lower,each=nb.integration) *
#'    matrix(integration.points, nrow=nb.integration) +
#'    matrix(rep(lower,each=nb.integration), nrow=nb.integration)
#' integration.param$integration.weights <- NULL
#' integration.param$alpha <- NULL
#' optimcontrol <- list()
#' optimcontrol$method <- "multistart"
#' crit <- max_vorob_parallel_gpc(lower = lower, upper = upper,
#'                                batchsize = batchsize,
#'                                integration.param = integration.param,
#'                                object = model, optimcontrol = optimcontrol, seed=1)
#' plan(sequential) # deactivate parallel calculations: back to sequential mode 
#' }
max_vorob_parallel_gpc <- function(lower, upper, optimcontrol=NULL,	batchsize,
						integration.param,  object, new.noise.var=0, seed=NULL){


  future.globals <- c("vorob_optim_parallel_gpc", "computeQuickgpccov","comp_term", "predict_update_gpc_parallel")
  future.packages <- c("DiceKriging","GPCsign")



  penalisation <- 1
  optim.option <- optimcontrol$optim.option
  if(is.null(optim.option)) optim.option <- 1
	integration.points <- as.matrix(integration.param$integration.points) ; d <- object@d
	integration.weights <- integration.param$integration.weights
  alpha <- integration.param$alpha

  if(is.null(optimcontrol$method)) optimcontrol$method <- "multistart"


    # gpc_int <- predict(object=object, newdata=integration.points)
# 	intpoints.c <- gpc_int$c
# 	intpoints.oldmean <- gpc_int$Zsimu_mean
# 	intpoints.oldsd <- sqrt(gpc_int$Zsimu_var)
# 	pn <- gpc_int$prob

#   precalc.data <- list(3)
#   precalc.data$c.K <- crossprod(intpoints.c, object@invK)
#   precalc.data$lambda.intpoints <- gpc_int$lambda
#   precalc.data$pn.intpoints <- pn

  precalc.data <- precomputeUpdateData(model=object, integration.points=integration.points)
  pn <- precalc.data$pn.intpoints
  intpoints.oldmean <- precalc.data$intpoints.oldmean
  intpoints.oldsd <- precalc.data$intpoints.oldsd

  if(is.null(alpha)) alpha <- KrigInv::vorob_threshold(pn)
  pn_bigger_than_alpha <- (pn>alpha)+0
  pn_lower_than_alpha <- 1-pn_bigger_than_alpha


  ### current vorob dev and exp comp
	if(is.null(integration.weights)) current.vorob <- mean(pn*pn_lower_than_alpha + penalisation*(1-pn)*pn_bigger_than_alpha)
	if(!is.null(integration.weights)) current.vorob <- sum(integration.weights*(pn*pn_lower_than_alpha + penalisation*(1-pn)*pn_bigger_than_alpha))





	fun.optim <- vorob_optim_parallel2_gpc
	########################################################################################
	#discrete Optimization
	#batchsize optimizations in dimension d
	if(optimcontrol$method=="discrete"){

		if (is.null(optimcontrol$optim.points)){
			n.discrete.points <- d*100
			optimcontrol$optim.points <- t(lower + t(matrix(runif(d*n.discrete.points),ncol=d)) * (upper - lower))
		}

		optim.points <- optimcontrol$optim.points
		optim.points <- data.frame(optim.points)

		if(ncol(optim.points)==d){
		  #this is the standard case:
		  fun.optim <- vorob_optim_parallel2_gpc
		  all.crit <- seq(1,nrow(optim.points))


		  other.points <- NULL
		  for (j in 1:batchsize){
			  for (i in 1:nrow(optim.points)){

				  all.crit[i] <- fun.optim(x=t(optim.points[i,]), integration.points=integration.points,integration.weights=integration.weights,
					intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
					precalc.data=precalc.data,object=object, new.noise.var=new.noise.var,
					other.points=other.points,batchsize=j,alpha=alpha,current.vorob=current.vorob)
			  }
			  ibest <- which.min(all.crit)
			  other.points <- c(other.points,as.numeric(optim.points[ibest,]))
		  }

		  o <- list(3)
		  o$par <- other.points;o$par <- t(matrix(o$par,nrow=d));
		  o$value <- min(all.crit); o$value <- as.matrix(o$value);
		  o$allvalues <- all.crit
		  return(list(par=o$par, value=o$value,allvalues=o$allvalues))
		}else{

		  fun.optim <- vorob_optim_parallel2_gpc
		  all.crit <- seq(1,nrow(optim.points))

		  for (i in 1:nrow(optim.points)){
		    all.crit[i] <- fun.optim(x=t(optim.points[i,]), integration.points=integration.points,integration.weights=integration.weights,
		                             intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
		                             precalc.data=precalc.data, object=object, new.noise.var=new.noise.var,
		                             batchsize=batchsize,alpha=alpha,current.vorob=current.vorob)
		  }
		  ibest <- which.min(all.crit)
		  o <- list(3)
		  o$par <- t(matrix(optim.points[ibest,],nrow=d,ncol=batchsize)); colnames(o$par) <- colnames(object@X)
		  o$value <- all.crit[ibest]; o$value <- as.matrix(o$value); colnames(o$value) <- colnames(object@y)
		  o$allvalues <- all.crit
		  return(list(par=o$par, value=o$value,allvalues=o$allvalues, alpha=alpha))
		}
	}

	########################################################################################
	#Optimization with Genoud
	if(optimcontrol$method=="genoud"){
		if (is.null(optimcontrol$pop.size))  optimcontrol$pop.size <- 50*d#floor(4 + 3 * log(d))
		if (is.null(optimcontrol$max.generations))  optimcontrol$max.generations <- 10*d#100*d
		if (is.null(optimcontrol$wait.generations))  optimcontrol$wait.generations <- 2#2
		if (is.null(optimcontrol$BFGSburnin)) optimcontrol$BFGSburnin <- 2#10#0
		if (is.null(optimcontrol$parinit))  optimcontrol$parinit <- NULL
		if (is.null(optimcontrol$unif.seed))  optimcontrol$unif.seed <- 1
		if (is.null(optimcontrol$int.seed))  optimcontrol$int.seed <- 1
		if (is.null(optimcontrol$print.level))  optimcontrol$print.level <- 1

		#mutations
		if (is.null(optimcontrol$P1)) optimcontrol$P1<-10*d#50
		if (is.null(optimcontrol$P2)) optimcontrol$P2<-10*d#50
		if (is.null(optimcontrol$P3)) optimcontrol$P3<-5*d#50
		if (is.null(optimcontrol$P4)) optimcontrol$P4<-0#50
		if (is.null(optimcontrol$P5)) optimcontrol$P5<-10*d
		if (is.null(optimcontrol$P6)) optimcontrol$P6<-10*d#50
		if (is.null(optimcontrol$P7)) optimcontrol$P7<-10*d
		if (is.null(optimcontrol$P8)) optimcontrol$P8<-10*d
		if (is.null(optimcontrol$P9)) optimcontrol$P9<-0

		if(optim.option==1){
			#one unique optimization in dimension batchsize * d
			domaine <- cbind(rep(lower,times=batchsize), rep(upper,times=batchsize))
			other.points <- NULL
			set.seed(seed)
			o <- genoud(fn=fun.optim, nvars=d*batchsize, max=FALSE, pop.size=optimcontrol$pop.size,
				max.generations=optimcontrol$max.generations,wait.generations=optimcontrol$wait.generations,
				hard.generation.limit=TRUE, starting.values=optimcontrol$parinit, MemoryMatrix=TRUE,
				Domains=domaine, default.domains=10, solution.tolerance=0.000000001,
				boundary.enforcement=2, lexical=FALSE, gradient.check=FALSE, BFGS=TRUE,
				data.type.int=FALSE, hessian=FALSE, unif.seed=optimcontrol$unif.seed,
				int.seed=optimcontrol$int.seed,print.level=optimcontrol$print.level, share.type=0, instance.number=0,
				output.path="stdout", output.append=FALSE, project.path=NULL,
				P1=optimcontrol$P1, P2=optimcontrol$P2, P3=optimcontrol$P3,
				P4=optimcontrol$P4, P5=optimcontrol$P5, P6=optimcontrol$P6,
				P7=optimcontrol$P7, P8=optimcontrol$P8, P9=optimcontrol$P9,
				P9mix=NULL, BFGSburnin=optimcontrol$BFGSburnin,BFGSfn=NULL, BFGShelp=NULL,
				cluster=FALSE, balance=FALSE, debug=FALSE, other.points=NULL,
				object=object, integration.points=integration.points,
				intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
				precalc.data=precalc.data,integration.weights=integration.weights,
				new.noise.var=new.noise.var,batchsize=batchsize,alpha=alpha,
        current.vorob=current.vorob)

			o$par <- t(matrix(o$par,nrow=d)); colnames(o$par) <- colnames(object@X)
			o$value <- as.matrix(o$value); colnames(o$value) <- colnames(object@y)
		}else{
			#batchsize optimizations in dimension d
			fun.optim <- vorob_optim_parallel2_gpc
			domaine <- cbind(lower,upper)
			other.points <- NULL
			for (i in 1:batchsize){
				set.seed(seed)
				o <- genoud(fn=fun.optim, nvars=d, max=FALSE, pop.size=optimcontrol$pop.size,
						max.generations=optimcontrol$max.generations,wait.generations=optimcontrol$wait.generations,
						hard.generation.limit=TRUE, starting.values=optimcontrol$parinit, MemoryMatrix=TRUE,
						Domains=domaine, default.domains=10, solution.tolerance=0.000000001,
						boundary.enforcement=2, lexical=FALSE, gradient.check=FALSE, BFGS=TRUE,
						data.type.int=FALSE, hessian=FALSE, unif.seed=optimcontrol$unif.seed,
						int.seed=optimcontrol$int.seed,print.level=optimcontrol$print.level, share.type=0, instance.number=0,
						output.path="stdout", output.append=FALSE, project.path=NULL,
						P1=optimcontrol$P1, P2=optimcontrol$P2, P3=optimcontrol$P3,
						P4=optimcontrol$P4, P5=optimcontrol$P5, P6=optimcontrol$P6,
						P7=optimcontrol$P7, P8=optimcontrol$P8, P9=optimcontrol$P9,
						P9mix=NULL, BFGSburnin=optimcontrol$BFGSburnin,BFGSfn=NULL, BFGShelp=NULL,
						cluster=FALSE, balance=FALSE, debug=FALSE, other.points=other.points,
						object=object, integration.points=integration.points,
						intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
						precalc.data=precalc.data,integration.weights=integration.weights,
						new.noise.var=new.noise.var,batchsize=i,current.vorob=current.vorob,
            alpha=alpha)

        other.points <- c(other.points,as.numeric(o$par))
			}
			o$par <- t(matrix(other.points,nrow=d));
			o$value <- as.matrix(o$value);
		}
		# stopCluster(cl = cl)

		return(list(par=o$par, value=o$value, pn=pn, current.vorob=current.vorob, alpha=alpha))
	}

	########################################################################################
	#Optimization with "L-BFGS-B" multistart
	if(optimcontrol$method=="multistart"){
	  if(optim.option==1){

	    multistart <- 2*batchsize*d+1
	    other.points <- NULL
	    fun.optim <- vorob_optim_parallel2_gpc
	    control.param <- list()
	    control.param$maxit <- 10*d*batchsize
	    o <- list(2)
		set.seed(seed)
	    # par <- matrix(rep(lower,each=multistart), nrow=multistart) + runif((d)*multistart)*matrix(rep(upper-lower,each=multistart), nrow=multistart)
	    par <-  sweep(rep(upper-lower,each=multistart) * matrix(runif((d*batchsize)*multistart), nrow=multistart),1,matrix(rep(lower,each=multistart), nrow=multistart),FUN="+")
	    par <- lapply(apply(par,1,list), unlist)

	    lres <-  future_lapply(X=par, FUN=optim, fn=fun.optim, lower=lower, upper=upper, control=control.param,
	                      method="L-BFGS-B", other.points=other.points,
	                      object=object, integration.points=integration.points,
	                      intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
	                      precalc.data=precalc.data,integration.weights=integration.weights,
	                      new.noise.var=new.noise.var,batchsize=batchsize,current.vorob=current.vorob,
	                      alpha=alpha, seed=seed, future.globals=future.globals, future.packages=future.packages, future.seed = seed)



	    bestlres <- Reduce(cbind, lapply(lres, function(alist) alist$value))
	    res <- lres[[which.min(bestlres)]]

	    o$par <- res$par
	    o$value <- res$value
	    other.points <- c(other.points,as.numeric(o$par))

	    o$par <- matrix(res$par,ncol=d);
	    o$value <- as.matrix(o$value);
	    return(list(par=o$par, value=o$value, pn=pn, current.vorob=current.vorob, alpha=alpha))

	  }else{
	  multistart <- 2*d+1
	  other.points <- NULL
	  fun.optim <- vorob_optim_parallel2_gpc
	  control.param <- list()
	  control.param$maxit <- 10*d
	  o <- list(2)
	   set.seed(seed)
	  for (i in 1:batchsize){

	    par <-  sweep(rep(upper-lower,each=multistart) * matrix(runif((d)*multistart), nrow=multistart),1,matrix(rep(lower,each=multistart), nrow=multistart),FUN="+")
	    par <- lapply(apply(par,1,list), unlist)

	    lres <-  future_lapply(X=par, FUN=optim, fn=fun.optim, lower=lower, upper=upper, control=control.param,
	                      method="L-BFGS-B", other.points=other.points,
	                      object=object,integration.points=integration.points,
	                      intpoints.oldmean=intpoints.oldmean,intpoints.oldsd=intpoints.oldsd,
	                      precalc.data=precalc.data,integration.weights=integration.weights,
	                      new.noise.var=new.noise.var,batchsize=1,current.vorob=current.vorob,
	                      alpha=alpha,seed=seed, future.globals=future.globals, future.packages=future.packages, future.seed = seed)


	    bestlres <- Reduce(cbind, lapply(lres, function(alist) alist$value))
	    res <- lres[[which.min(bestlres)]]
	    o$par <- res$par # matrix(res$par,ncol=d);
	    o$value <- as.matrix(res$value);
	    other.points <- c(other.points,as.numeric(o$par))
	  }

	  o$par <- t(matrix(other.points,nrow=d));
	  o$value <- as.matrix(o$value);
	  return(list(par=o$par, value=o$value, pn=pn, current.vorob=current.vorob,  alpha=alpha))


	}

	}
 }

