#' @title Twitter App API authentication
#'
#' @description If an individual has applied for and been granted Twitter API access they will also have a bearer token
#'   associated with their app. This token allows read-only access to the API but higher rate-limits so it is the most
#'   suited method for this package and data collection.
#'
#' @note \pkg{vosonSML} uses the \pkg{rtweet} package for twitter data collection and also gratefully acknowledges the
#'   techniques and code written by its authors reproduced in this package for creating twitter API access tokens.
#'
#' @param bearer Character string. Twitter app bearer token.
#' @param verbose Logical. Output additional information. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#'
#' @return A \code{credential} object containing an access token \code{$auth} and social media type descriptor
#'   \code{$socialmedia} set to \code{"twitter"}. Object has the class names \code{"credential"} and \code{"twitter"}.
#'
#' @examples
#' \dontrun{
#' # twitter API access using an app bearer token
#' auth <- auth_twitter_app("xxxxxxxxxxxx")
#' }
#'
#' @export
auth_twitter_app <- name <- function(bearer, verbose = FALSE, ...) {
  if (missing(bearer)) stop("Missing twitter bearer token.", call. = FALSE)
  Authenticate.twitter("twitter", bearerToken = bearer, verbose = verbose, ...)
}

#' @title Twitter Dev API authentication
#'
#' @description Twitter authentication uses OAuth and typically requires four developer API keys generated when you
#'   create a twitter app via the twitter developer web site. These keys and the authentication object produced are
#'   referenced as developer keys and developer access in this package. Software using developer access to the API can
#'   operate with full access to the API including within the user-context but with the understanding that it is an app.
#'   The \pkg{rtweet} package refers to this as 'bot' authentication.
#'
#' @note \pkg{vosonSML} uses the \pkg{rtweet} package for twitter data collection and also gratefully acknowledges the
#'   techniques and code written by its authors reproduced in this package for creating twitter API access tokens.
#'
#' @param app_name Character string. Registered twitter app name associated with the API keys.
#' @param api_key Character string. API consumer key to authenticate (also called API key).
#' @param api_secret Character string. API consumer secret to authenticate (also called API secret).
#' @param access_token Character string. API access token to authenticate.
#' @param access_token_secret Character string. API access token secret to authenticate.
#' @param verbose Logical. Output additional information. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#'
#' @return A \code{credential} object containing an access token \code{$auth} and social media type descriptor
#'   \code{$socialmedia} set to \code{"twitter"}. Object has the class names \code{"credential"} and \code{"twitter"}.
#'
#' @examples
#' \dontrun{
#' # twitter authentication using developer app API keys
#' auth <- auth_twitter_dev(
#'   app_name = "My App",
#'   api_key = "xxxxxxxxxxxx",
#'   api_secret = "xxxxxxxxxxxx",
#'   access_token = "xxxxxxxxxxxx",
#'   access_token_secret = "xxxxxxxxxxxx"
#' )
#' }
#'
#' @export
auth_twitter_dev <-
  function(app_name = "r twitter app",
           api_key,
           api_secret,
           access_token,
           access_token_secret,
           verbose = FALSE,
           ...) {

    if (missing(api_key) || missing(api_secret)) {
      stop("Missing twitter api keys.", call. = FALSE)
    }

    if (missing(access_token) || missing(access_token_secret)) {
      stop("Missing twitter access token.", call. = FALSE)
    }

    Authenticate.twitter(
      "twitter",
      appName = app_name,
      apiKey = api_key,
      apiSecret = api_secret,
      accessToken = access_token,
      accessTokenSecret = access_token_secret,
      verbose = verbose,
      ...)
  }

#' @title Twitter User API authentication
#'
#' @description A method commonly used by third-party apps in which an app can be authorized by a user to use the
#'   twitter API on their behalf. The implementation of this method in \pkg{vosonSML} does not require a developer
#'   account but does still require the user to have access to an apps two consumer API keys (generated by the app
#'   developer). The authentication object with token produced from this method allows the user to access the API within
#'   their own user-context and rate-limits.
#'
#' @note \pkg{vosonSML} uses the \pkg{rtweet} package for twitter data collection and also gratefully acknowledges the
#'   techniques and code written by its authors reproduced in this package for creating twitter API access tokens.
#'
#' @param api_key Character string. API consumer key to authenticate (also called API key).
#' @param api_secret Character string. API consumer secret to authenticate (also called API secret).
#' @param verbose Logical. Output additional information. Default is \code{FALSE}.
#' @param ... Additional parameters passed to function. Not used in this method.
#'
#' @return A \code{credential} object containing an access token \code{$auth} and social media type descriptor
#'   \code{$socialmedia} set to \code{"twitter"}. Object has the class names \code{"credential"} and \code{"twitter"}.
#'
#' @examples
#' \dontrun{
#' # twitter user authentication via authorization of an app requires the
#' # apps consumer API keys (api_key and api_secret)
#' # this method will open a web browser to twitter prompting the user to
#' # log in and authorize the app
#' auth <- auth_twitter_user(
#'   api_key = "xxxxxxxxxxxx", api_secret = "xxxxxxxxxxxx"
#' )
#' }
#'
#' @export
auth_twitter_user <- function(api_key, api_secret, verbose = FALSE, ...) {
  if (missing(api_key) || missing(api_secret)) {
    stop("Missing twitter api keys.", call. = FALSE)
  }

  Authenticate.twitter(
    "twitter",
    apiKey = api_key,
    apiSecret = api_secret,
    verbose = verbose,
    ...)
}

#' @rdname Collect.reddit
#' @export
collect_reddit_threads <-
  function(threadUrls,
           waitTime = c(3, 5),
           ua = getOption("HTTPUserAgent"),
           writeToFile = FALSE,
           verbose = FALSE,
           ...) {
    Collect(
      credential = Authenticate("reddit"),
      threadUrls = threadUrls,
      waitTime = waitTime,
      ua = ua,
      writeToFile = writeToFile,
      verbose = verbose,
      ...
    )
  }

#' @rdname Collect.web
#' @export
collect_web_hyperlinks <- function(pages = NULL, writeToFile = FALSE, verbose = FALSE, ...) {
    Collect(
      credential = Authenticate("web"),
      pages = pages,
      writeToFile = writeToFile,
      verbose = verbose,
      ...
    )
  }

#' @rdname Merge
#' @export
merge_data <- Merge

#' @rdname MergeFiles
#' @export
merge_files <- MergeFiles

#' @rdname AddText
#' @export
add_text <- AddText

#' @rdname AddUserData
#' @export
add_users <- AddUserData

#' @rdname AddVideoData
#' @export
add_videos <- AddVideoData
