#' @importFrom stats runif
#' @name johnsonsb
#' @aliases johnsonsb djohnsonsb pjohnsonsb qjohnsonsb rjohnsonsb
#'
#' @title The Johnson SB distribution
#'
#' @description Density function, distribution function, quantile function and random number generation function
#' for the Johnson SB distribution reparametrized in terms of the \eqn{\tau}-th quantile, \eqn{\tau \in (0, 1)}.
#'
#' @author
#' Josmar Mazucheli
#'
#' André F. B. Menezes
#'
#' @references
#'
#' Lemonte, A. J. and Bazán, J. L., (2015). New class of Johnson SB distributions and its associated regression model for rates and proportions. \emph{Biometrical Journal}, \bold{58}(4), 727--746.
#'
#' Johnson, N. L., (1949). Systems of frequency curves generated by methods of translation. \emph{Biometrika}, \bold{36}(1), 149--176.
#'
#' @param x,q vector of positive quantiles.
#' @param p vector of probabilities.
#' @param n number of observations. If \code{length(n) > 1}, the length is taken to be the number required.
#' @param mu location parameter indicating the \eqn{\tau}-th quantile, \eqn{\tau \in (0, 1)}.
#' @param theta nonnegative shape parameter.
#' @param tau the parameter to specify which quantile is to used.
#' @param log,log.p logical; If TRUE, probabilities p are given as log(p).
#' @param lower.tail logical; If TRUE, (default), \eqn{P(X \leq{x})} are returned, otherwise \eqn{P(X > x)}.
#'
#' @return \code{djohnsonsb} gives the density, \code{pjohnsonsb} gives the distribution function,
#' \code{qjohnsonsb} gives the quantile function and \code{rjohnsonsb} generates random deviates.
#'
#' @return Invalid arguments will return an error message.
#'
#' @details
#' Probability density function
#' \deqn{f(y\mid \alpha ,\theta )=\frac{\theta }{\sqrt{2\pi }}\frac{1}{y(1-y)}\exp\left\{ -\frac{1}{2}\left[\alpha +\theta \log\left(\frac{y}{1-y}\right)\right] ^{2}\right\}}
#'
#' Cumulative distribution function
#' \deqn{F(y\mid \alpha ,\theta )=\Phi \left[ \alpha +\theta \log \left( \frac{y}{1-y}\right) \right]}
#'
#' Quantile function
#' \deqn{Q(\tau \mid \alpha ,\theta )=\frac{\exp \left[ \frac{\Phi ^{-1}(\tau)-\alpha }{\theta }\right] }{1+\exp \left[ \frac{\Phi ^{-1}(\tau )-\alpha }{\theta }\right] }}
#'
#' Reparameterization
#' \deqn{\alpha =g^{-1}(\mu )=\Phi ^{-1}(\tau )-\theta \log \left( \frac{\mu }{1-\mu }\right)}
#'
#' @examples
#'
#' set.seed(123)
#' x <- rjohnsonsb(n = 1000, mu = 0.5, theta = 1.5, tau = 0.5)
#' R <- range(x)
#' S <- seq(from = R[1], to = R[2], by =  0.01)
#' hist(x, prob = TRUE, main = 'Johnson SB')
#' lines(S, djohnsonsb(x = S, mu = 0.5, theta = 1.5, tau = 0.5), col = 2)
#' plot(ecdf(x))
#' lines(S, pjohnsonsb(q = S, mu = 0.5, theta = 1.5, tau = 0.5), col = 2)
#' plot(quantile(x, probs = S), type = "l")
#' lines(qjohnsonsb(p = S, mu = 0.5, theta = 1.5, tau = 0.5), col = 2)
#'
##################################################
#' @rdname johnsonsb
#' @export
#
djohnsonsb <- function (x, mu, theta, tau = 0.5, log = FALSE)
{
  stopifnot(x > 0, x < 1, mu > 0, mu < 1, tau > 0, tau < 1, theta > 0);
  cpp_djohnsonsb(x, mu, theta, tau, log[1L]);
}
##################################################
#' @rdname johnsonsb
#' @export
#'
pjohnsonsb <- function (q, mu, theta, tau = 0.5, lower.tail = TRUE, log.p = FALSE)
{
  stopifnot(q > 0, q < 1, mu > 0, mu < 1, tau > 0, tau < 1, theta > 0)
  cpp_pjohnsonsb(q, mu, theta, tau, lower.tail[1L], log.p[1L])
}
##################################################
#' @rdname johnsonsb
#' @export
#'
qjohnsonsb <- function(p, mu, theta, tau = 0.5, lower.tail = TRUE, log.p = FALSE)
{
  stopifnot(p > 0, p < 1, mu > 0, mu < 1, tau > 0, tau < 1, theta > 0)
  cpp_qjohnsonsb(p, mu, theta, tau, lower.tail[1L], log.p[1L])
}
##################################################
#' @rdname johnsonsb
#' @export
#'
rjohnsonsb <- function(n, mu, theta, tau = 0.5)
{
  stopifnot(n > 0, mu > 0, mu < 1, tau > 0, tau < 1, theta > 0)
  cpp_qjohnsonsb(runif(n), mu, theta, tau, TRUE, FALSE)
}
