% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc_and_utility.R
\name{umx_score_scale}
\alias{umx_score_scale}
\title{Score a psychometric scale by summing normal and reversed items.}
\usage{
umx_score_scale(
  base = NULL,
  pos = NULL,
  rev = NULL,
  min = 1,
  max = NULL,
  data = NULL,
  score = c("total", "mean", "max", "factor"),
  name = NULL,
  na.rm = FALSE,
  minManifests = NA
)
}
\arguments{
\item{base}{String common to all item names.}

\item{pos}{The positive-scored item numbers.}

\item{rev}{The reverse-scored item numbers.}

\item{min}{Min possible score (default = 1). Not implemented for values other than 1 so far...}

\item{max}{Max possible score for an item (to compute how to reverse items).}

\item{data}{The data frame}

\item{score}{Whether to compute the score total, mean, max, or factor (default = "total")}

\item{name}{= name of the scale to be returned. Defaults to "base_score"}

\item{na.rm}{Whether to delete NAs when computing scores (Default = TRUE) Note: Choice affects mean!}

\item{minManifests}{If score = factor, how many missing items to tolerate for an individual?}
}
\value{
\itemize{
\item scores
}
}
\description{
Use this function to generate scores as the appropriate sum of responses to the normal and reversed items in a scale.

Items must be named on the pattern \code{baseN}, where \code{base} is the string common to all item (column) names and N is the item number in the scale.

\code{pos} and \code{rev} are vectors of the item numbers for the normal and reverse-scored item numbers.

To reverse items, the function uses \code{max} and \code{min} as the lowest and highest possible response scores to compute how to reverse items.

\emph{note}: \code{min} defaults to 1.
}
\details{
In the presence of NAs, \code{score= "mean"} and \code{score = "totals"} both return NA unless na.rm = TRUE.
\code{score = "max"}, ignores NAs no matter what.
}
\examples{
library(psych)
data(bfi)

# ==============================
# = Score Agreeableness totals =
# ==============================

# Handscore subject 1
# A1(Reversed) + A2 + A3 + A4 + A5 
#      (6+1)-2 + 4  + 3  + 4  + 4  = 20

tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= bfi, name = "A")
tmp[1, namez(tmp, "A",ignore.case=FALSE)]
#  A1 A2 A3 A4 A5  A
#  2  4  3  4  4  20

# =================================================================================
# = Note: (as of a fix in 2020-05-08) items not reversed in the returned data set =
# =================================================================================
tmp = umx_score_scale("A", pos = 1, rev = 2:5, max = 6, data= bfi, name = "A")
tmp[1, namez(tmp, "A",ignore.case=FALSE)]
#   A1 A2 A3 A4 A5   A
#   2   4  3  4  4 = 15

tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= bfi, name = "A", score="mean")
tmp$A[1] # subject 1 mean = 4

# ===========================================
# = How does mean react to a missing value? =
# ===========================================
tmpDF = bfi
tmpDF[1, "A1"] = NA
tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= tmpDF, name = "A", score="mean")
tmp$A[1] # NA: (na.rm defaults to FALSE)

tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= tmpDF, 
     name = "A", score="mean", na.rm=TRUE)
tmp$A[1] # 3.75

# ===============
# = Score = max =
# ===============
tmp = umx_score_scale("A", pos = 2:5, rev = 1, max = 6, data= bfi, name = "A", score="max")
tmp$A[1] # subject 1 max = 5 (the reversed item 1)

tmp = umx_score_scale("E", pos = c(3,4,5), rev = c(1,2), max = 6, data= tmp)
tmp$E_score[1] # default scale name

# Using @BillRevelle's psych package: More diagnostics, including alpha
scores= psych::scoreItems(items = bfi, min = 1, max = 6, keys = list(
	E = c("-E1","-E2", "E3", "E4", "E5"),
	A = c("-A1", "A2", "A3", "A4", "A5")
))
summary(scores)
scores$scores[1,]
#  E   A 
# 3.8 4.0 

# Compare output
# (note, by default psych::scoreItems replaces NAs with the sample median...)
RevelleE = as.numeric(scores$scores[,"E"]) * 5
all(RevelleE == tmp[,"E_score"], na.rm = TRUE)

}
\seealso{
Other Miscellaneous Utility Functions: 
\code{\link{install.OpenMx}()},
\code{\link{qm}()},
\code{\link{umxBrownie}()},
\code{\link{umxLav2RAM}()},
\code{\link{umxRAM2Lav}()},
\code{\link{umxVersion}()},
\code{\link{umx_array_shift}()},
\code{\link{umx_find_object}()},
\code{\link{umx_lower.tri}()},
\code{\link{umx_msg}()},
\code{\link{umx_open_CRAN_page}()},
\code{\link{umx_pad}()},
\code{\link{umx_print}()},
\code{\link{umx}}
}
\concept{Miscellaneous Utility Functions}
