# In tests/testthat/test-clean_data.R

test_that("clean_data works correctly with daily aggregation", {
  # Create a mock data.table
  env_data <- data.table::data.table(
    date = as.POSIXct(c(
      "2021-01-01 00:00:00", "2021-01-01 01:00:00",
      "2021-01-01 02:00:00"
    )),
    Station = "TEST001",
    part = "test",
    Komponente = "TMP",
    Komponente_txt = "Temperature",
    Wert = c(10, 20, 30)
  )

  # Run clean_data function with daily aggregation
  cleaned_data <- clean_data(env_data, "TEST001",
    aggregate_daily = TRUE
  )

  # Check if the cleaned data has the expected structure and content
  expect_s3_class(cleaned_data, "data.table")
  expect_true(all(c(
    "Station", "Komponente", "Komponente_txt", "day",
    "year", "Wert"
  )
  %in% colnames(cleaned_data)))
  expect_equal(nrow(cleaned_data), 1) # Aggregated to one row
  expect_equal(cleaned_data$Wert, 20) # Mean of 10, 20, 30
  expect_equal(cleaned_data$day[1], as.POSIXct("2021-01-01"))
  expect_equal(cleaned_data$year[1], 2021)
})

test_that("clean_data works correctly without daily aggregation", {
  # Create a mock data.table
  env_data <- data.table::data.table(
    date = as.POSIXct(c(
      "2021-01-01 00:00:00", "2021-01-01 01:00:00",
      "2021-01-01 02:00:00"
    )),
    Station = "TEST001",
    part = "test",
    Komponente = "TMP",
    Komponente_txt = "Temperature",
    Wert = c(10, 20, 30)
  )

  # Run clean_data function without daily aggregation
  cleaned_data <- clean_data(env_data, "TEST001",
    aggregate_daily = FALSE
  )

  # Check if the cleaned data has the expected structure and content
  expect_s3_class(cleaned_data, "data.table")
  expect_true(all(c(
    "date", "Station", "Komponente", "Komponente_txt",
    "Wert", "year"
  ) %in% colnames(cleaned_data)))
  expect_equal(nrow(cleaned_data), 3) # No aggregation
})

test_that("clean_data filters by station correctly", {
  env_data <- data.table::data.table(
    date = as.POSIXct(c("2021-01-01 00:00:00", "2021-01-01 01:00:00")),
    Station = c("TEST001", "TEST002"),
    part = "test",
    Komponente = "TMP",
    Komponente_txt = "Temperature",
    Wert = c(10, 20)
  )

  cleaned_data <- clean_data(env_data, "TEST001",
    aggregate_daily = FALSE
  )

  expect_equal(unique(cleaned_data$Station), "TEST001")
})

test_that("clean_data handles empty data.table gracefully", {
  env_data <- data.table::data.table(
    date = as.POSIXct(character()),
    Station = character(),
    part = character(),
    Komponente = character(),
    Komponente_txt = character(),
    Wert = numeric()
  )

  cleaned_data <- clean_data(env_data, "TEST001",
    aggregate_daily = TRUE
  )
  expect_equal(nrow(cleaned_data), 0)
})
