% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sequence.R
\name{sequence2}
\alias{sequence2}
\alias{seq_id}
\alias{seq_v}
\alias{seq_size}
\alias{window_sequence}
\alias{lag_sequence}
\alias{lead_sequence}
\title{Utilities for creating useful sequences}
\usage{
sequence2(size, from = 1L, by = 1L)

seq_id(size)

seq_v(from = 1L, to = 1L, by = 1L)

seq_size(from, to, by = 1L)

window_sequence(size, k, partial = TRUE, ascending = TRUE)

lag_sequence(size, k, partial = TRUE)

lead_sequence(size, k, partial = TRUE)
}
\arguments{
\item{size}{Vector of sequence lengths.}

\item{from}{Start of sequence(s).}

\item{by}{Unit increment of sequence(s).}

\item{to}{End of sequence(s).}

\item{k}{Window/lag size.}

\item{partial}{Should partial windows/lags be returned? Default is \code{TRUE}.}

\item{ascending}{Should window sequence be ascending? Default is \code{TRUE}.}
}
\value{
A vector of length \code{sum(size)} except for \code{seq_v} which
returns a vector of size \code{sum((to - from) / (by + 1))}
}
\description{
\code{sequence2} is an extension to \link{sequence} which
accepts decimal number increments. \cr
\code{seq_id} can be paired with \code{sequence2} to group individual sequences. \cr
\code{seq_v} is a vectorised version of \link{seq}. \cr
\code{window_sequence} creates a vector of window sizes for rolling calculations. \cr
\code{lag_sequence} creates a vector of lags for rolling calculations. \cr
\code{lead_sequence} creates a vector of leads for rolling calculations. \cr
}
\details{
\code{sequence2()} works in the same way as \code{sequence()} but can accept
non-integer \code{by} values.
It also recycles \code{from} and \code{to}, in the same way as \code{sequence()}. \cr
If any of the sequences contain values > \code{.Machine$integer.max},
then the result will always be a double vector.

\code{from} can be also be a date, date-time, or any object that supports
addition and multiplication.

\code{seq_v()} is a vectorised version of \code{seq()} that strictly accepts
only the arguments \code{from}, \code{to} and \code{by}. \cr
}
\examples{
library(timeplyr)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
sequence(1:3)
sequence2(1:3)

sequence(1:3, by = 0.1)
sequence2(1:3, by = 0.1)

sequence(c(3, 2), by = c(-0.1, 0.1))
sequence2(c(3, 2), by = c(-0.1, 0.1))

# We can group sequences using seq_id
size <- c(7, 0, 3)
from <- 1
by <- c(-0.1, 0.1, 1/3)
x <- sequence2(size, from, by)
names(x) <- seq_id(size)
x

# Vectorised version of seq()
seq_v(1, 10, by = c(1, 0.5))
# Same as below
c(seq(1, 10, 1), seq(1, 10, 0.5))

# Programmers may use seq_size() to determine final sequence lengths

sizes <- seq_size(1, 10, by = c(1, 0.5))
print(paste(c("sequence sizes: (", sizes, ") total size:", sum(sizes)),
            collapse = " "))

# We can group sequences using seq_id

from <- Sys.Date()
to <- from + 10
by <- c(1, 2, 3)
x <- seq_v(from, to, by)
names(x) <- seq_id(seq_size(from, to, by))
x

# Utilities for rolling calculations

window_sequence(c(3, 5), 3)
window_sequence(c(3, 5), 3, partial = FALSE)
window_sequence(c(3, 5), 3, partial = TRUE, ascending = FALSE)
# One can for example use these in data.table::frollsum
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
