% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tmap.R
\name{tmap}
\alias{tmap}
\alias{tmap_chr}
\alias{tmap_int}
\alias{tmap_lgl}
\alias{tmap_dbl}
\alias{tmap_dfc}
\alias{tmap_dfr}
\alias{tmap_df}
\title{Apply a function to each element of a \code{tbl_time} object.}
\usage{
tmap(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_chr(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_int(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_lgl(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_dbl(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_dfc(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_dfr(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)

tmap_df(.x, .f, period = "yearly", start_date = NULL, name = "data", ...)
}
\arguments{
\item{.x}{A \code{tbl_time} object.}

\item{.f}{A function, formula, or atomic vector.

If a \strong{function}, it is used as is.

If a \strong{formula}, e.g. \code{~ .x + 2}, it is converted to a function. There
are three ways to refer to the arguments:
\itemize{
\item For a single argument function, use \code{.}
\item For a two argument function, use \code{.x} and \code{.y}
\item For more arguments, use \code{..1}, \code{..2}, \code{..3} etc
}

This syntax allows you to create very compact anonymous functions.

If \strong{character vector}, \strong{numeric vector}, or \strong{list}, it
is converted to an extractor function. Character vectors index by name
and numeric vectors index by position; use a list to index by position
and name at different levels. Within a list, wrap strings in \code{get_attr()}
to extract named attributes. If a component is not present, the value of
\code{.default} will be returned.}

\item{period}{A formula or character specification used for time-based grouping.

If a formula, e.g. \code{1~year}, the formula is split and parsed to form the
grouping period. The \code{period} argument accepts a formula of the form
\code{multiple ~ period} allowing for flexible period grouping.
The following are examples:
\itemize{
\item 1 Year: \code{1~y}
\item 3 Months: \code{3~m}
\item 90 Days: \code{90~d}
}

Note that while shorthand is used above, an attempt is made to recognize
more explicit period names such as:
\itemize{
\item 2 Year: \code{2~year} / \code{2~years} / \code{2~yearly}
}

The \code{period} argument also accepts characters that are converted to their
corresponding periods. The following are accepted:
\itemize{
\item \code{"yearly"} or \code{"y"}
\item \code{"quarterly"} or \code{"q"}
\item \code{"monthly"} or \code{"m"}
\item \code{"weekly"} or \code{"w"}
\item \code{"daily"} or \code{"d"}
\item \code{"hour"} or \code{"h"}
\item \code{"minute"} or \code{"M"}
\item \code{"second"} or \code{"s"}
}}

\item{start_date}{Optional argument used to specify the start date for the
first group. The default is to start at the closest period boundary
below the minimum date in the supplied index.}

\item{name}{The character name of the list-column generated.}

\item{...}{Not currently used.}
}
\value{
A \code{tbl_time} object grouped by the time interval specified. The last
available date in that interval is returned as the new date.
}
\description{
The tmap functions transform a \code{tbl_time} input by applying a function to
each column at a specified time interval.
}
\details{
These functions are similar to \code{\link[purrr:map]{purrr::map()}},
but allow the user to also perform grouped mapping over
intervals such as \code{"yearly"}, \code{"monthly"}, \code{"hourly"}, etc. For example,
if \code{"monthly"} is chosen, one could picture the \code{tbl_time} object being
split into smaller \code{tbl_time}s, one for each month, and having the function
mapped over all of the columns in each of those smaller tibbles. The results
are then recombined into one tibble, with a list-column holding the results
of the mapping over each time period.

Groupings applied using \code{\link[dplyr:group_by]{dplyr::group_by()}} are respected.
}
\examples{

# First example -------------------------------------------------------------

data(FB)
FB <- as_tbl_time(FB, date)
# No need for the symbol column here
FB <- dplyr::select(FB, -symbol)

# Get the yearly average of every column in FB
mapped_mean <- FB \%>\%
  tmap(.f = ~mean(.x), period = "yearly")

# It is returned as a list-column because the time period adds
# an extra dimension to the mapping
mapped_mean

# Access individual elements. Here, the 2013 results
mapped_mean$data[[1]]

# More useful example -------------------------------------------------------

data(FB)
FB <- as_tbl_time(FB, date)
# No need for the symbol column here
FB <- dplyr::select(FB, -symbol)

# An easier approach might be to use `tmap_dfc` to coerce each list-column
# entry to a tibble, then unnest the result
# Here we calculate the monthly average for each column
FB \%>\%
  tmap_dfc(~mean(.x), period = "monthly") \%>\%
  tidyr::unnest()

# Functions with multiple return values -------------------------------------

data(FB)
FB <- as_tbl_time(FB, date)
# No need for the symbol column here
FB <- dplyr::select(FB, -symbol)

# Functions that return more than 1 number per map are possible, but more
# difficult to work with.
# Mapping the quantile function to each column of FB at yearly time increments
mapped_quantile <- FB \%>\%
  tmap(~quantile(.x), "yearly")

mapped_quantile$data[[1]]

# It is possible to get a cleaner result, but currently not intuitive
library(tibble)
FB \%>\%
  tidyr::gather(key = col_type, value = value, -date) \%>\%
  dplyr::group_by(col_type) \%>\%
  tmap_dfr(~quantile(.x) \%>\% as.list \%>\% as.tibble, period = "yearly") \%>\%
  tidyr::unnest()

}
