% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cox_regression_inter.R
\name{cox_regression_inter}
\alias{cox_regression_inter}
\alias{h_coxreg_inter_effect}
\alias{h_coxreg_inter_effect.numeric}
\alias{h_coxreg_inter_effect.factor}
\alias{h_coxreg_inter_effect.character}
\alias{h_coxreg_extract_interaction}
\alias{h_coxreg_inter_estimations}
\title{Cox Regression Helper: Interactions}
\usage{
h_coxreg_inter_effect(x, effect, covar, mod, label, control, ...)

\method{h_coxreg_inter_effect}{numeric}(x, effect, covar, mod, label, control, at, ...)

\method{h_coxreg_inter_effect}{factor}(x, effect, covar, mod, label, control, data, ...)

\method{h_coxreg_inter_effect}{character}(x, effect, covar, mod, label, control, data, ...)

h_coxreg_extract_interaction(effect, covar, mod, data, at, control)

h_coxreg_inter_estimations(
  variable,
  given,
  lvl_var,
  lvl_given,
  mod,
  conf_level = 0.95
)
}
\arguments{
\item{x}{(\code{numeric} or \code{factor})\cr the values of the covariate to be tested.}

\item{effect}{(\code{string})\cr the name of the effect to be tested and estimated.}

\item{covar}{(\code{string})\cr the name of the covariate in the model.}

\item{mod}{(\code{coxph})\cr a fitted Cox regression model (see \code{\link[survival:coxph]{survival::coxph()}}).}

\item{label}{(\code{string})\cr the label to be returned as \code{term_label}.}

\item{control}{(\code{list})\cr a list of controls as returned by \code{\link[=control_coxreg]{control_coxreg()}}.}

\item{...}{see methods.}

\item{at}{(\code{list})\cr a list with items named after the covariate, every
item is a vector of levels at which the interaction should be estimated.}

\item{data}{(\code{data.frame})\cr the data frame on which the model was fit.}

\item{variable, given}{(\code{string})\cr the name of variables in interaction. We seek the estimation
of the levels of \code{variable} given the levels of \code{given}.}

\item{lvl_var, lvl_given}{(\code{character})\cr corresponding levels has given by \code{\link[=levels]{levels()}}.}

\item{conf_level}{(\code{proportion})\cr confidence level of the interval.}
}
\value{
\itemize{
\item \code{h_coxreg_inter_effect()} returns a \code{data.frame} of covariate interaction effects consisting of the following
variables: \code{effect}, \code{term}, \code{term_label}, \code{level}, \code{n}, \code{hr}, \code{lcl}, \code{ucl}, \code{pval}, and \code{pval_inter}.
}

\itemize{
\item \code{h_coxreg_extract_interaction()} returns the result of an interaction test and the estimated values. If
no interaction, \code{\link[=h_coxreg_univar_extract]{h_coxreg_univar_extract()}} is applied instead.
}

\itemize{
\item \code{h_coxreg_inter_estimations()} returns a list of matrices (one per level of variable) with rows corresponding
to the combinations of \code{variable} and \code{given}, with columns:
\itemize{
\item \code{coef_hat}: Estimation of the coefficient.
\item \code{coef_se}: Standard error of the estimation.
\item \code{hr}: Hazard ratio.
\item \verb{lcl, ucl}: Lower/upper confidence limit of the hazard ratio.
}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Test and estimate the effect of a treatment in interaction with a covariate.
The effect is estimated as the HR of the tested treatment for a given level
of the covariate, in comparison to the treatment control.
}
\details{
Given the cox regression investigating the effect of Arm (A, B, C; reference A)
and Sex (F, M; reference Female) and the model being abbreviated: y ~ Arm + Sex + Arm:Sex.
The cox regression estimates the coefficients along with a variance-covariance matrix for:
\itemize{
\item b1 (arm b), b2 (arm c)
\item b3 (sex m)
\item b4 (arm b: sex m), b5 (arm c: sex m)
}

The estimation of the Hazard Ratio for arm C/sex M is given in reference
to arm A/Sex M by exp(b2 + b3 + b5)/ exp(b3) = exp(b2 + b5).
The interaction coefficient is deduced by b2 + b5 while the standard error
is obtained as $sqrt(Var b2 + Var b5 + 2 * covariance (b2,b5))$.
}
\section{Functions}{
\itemize{
\item \code{h_coxreg_inter_effect()}: S3 generic helper function to determine interaction effect.

\item \code{h_coxreg_inter_effect(numeric)}: Method for \code{numeric} class. Estimates the interaction with a \code{numeric} covariate.

\item \code{h_coxreg_inter_effect(factor)}: Method for \code{factor} class. Estimate the interaction with a \code{factor} covariate.

\item \code{h_coxreg_inter_effect(character)}: Method for \code{character} class. Estimate the interaction with a \code{character} covariate.
This makes an automatic conversion to \code{factor} and then forwards to the method for factors.

\item \code{h_coxreg_extract_interaction()}: A higher level function to get
the results of the interaction test and the estimated values.

\item \code{h_coxreg_inter_estimations()}: Hazard ratio estimation in interactions.

}}
\note{
\itemize{
\item Automatic conversion of character to factor does not guarantee results can be generated correctly. It is
therefore better to always pre-process the dataset such that factors are manually created from character
variables before passing the dataset to \code{\link[rtables:build_table]{rtables::build_table()}}.
}
}
\examples{
library(survival)

set.seed(1, kind = "Mersenne-Twister")

# Testing dataset [survival::bladder].
dta_bladder <- with(
  data = bladder[bladder$enum < 5, ],
  data.frame(
    time = stop,
    status = event,
    armcd = as.factor(rx),
    covar1 = as.factor(enum),
    covar2 = factor(
      sample(as.factor(enum)),
      levels = 1:4,
      labels = c("F", "F", "M", "M")
    )
  )
)
labels <- c("armcd" = "ARM", "covar1" = "A Covariate Label", "covar2" = "Sex (F/M)")
formatters::var_labels(dta_bladder)[names(labels)] <- labels
dta_bladder$age <- sample(20:60, size = nrow(dta_bladder), replace = TRUE)

plot(
  survfit(Surv(time, status) ~ armcd + covar1, data = dta_bladder),
  lty = 2:4,
  xlab = "Months",
  col = c("blue1", "blue2", "blue3", "blue4", "red1", "red2", "red3", "red4")
)

mod <- coxph(Surv(time, status) ~ armcd * covar1, data = dta_bladder)
h_coxreg_extract_interaction(
  mod = mod, effect = "armcd", covar = "covar1", data = dta_bladder,
  control = control_coxreg()
)

mod <- coxph(Surv(time, status) ~ armcd * covar1, data = dta_bladder)
result <- h_coxreg_inter_estimations(
  variable = "armcd", given = "covar1",
  lvl_var = levels(dta_bladder$armcd),
  lvl_given = levels(dta_bladder$covar1),
  mod = mod, conf_level = .95
)
result

}
