#'Drug interaction landscape
#'
#'A function to visualize the synergy scores for drug combinations as 2D or 3D
#'interaction landscape over the dose-response matrix.
#'
#' @param data a list object generated by function \code{\link{CalculateSynergy}}
#' @param type a parameter to specify the type of the interaction landscape, 2D, 3D
#' or both. By default, 2D interaction landscape is returned.
#' @param save.file a logical parameter to specify if the interaction landscape is saved as a pdf file
#' in the current working directory or returned as an R object. By default, it is FALSE.
#' @return a pdf file or the interaction landscapes are only displayed depending on
#' the save.file parameter.
#' @author Liye He \email{liye.he@helsinki.fi}
#' @examples
#' data("mathews_screening_data")
#' data <- ReshapeData(mathews_screening_data)
#' scores <- CalculateSynergy(data)
#' PlotSynergy(scores, "all")

PlotSynergy <- function(data, type = "2D", save.file = FALSE){
  if(!is.list(data)) {
    stop("Input data is not a list format!")
  }
  scores <- data$scores
  drug.pairs <- data$drug.pairs
  num.pairs <- length(scores)
  plots <- list()
  for (i in 1:num.pairs) {
    scores.dose <- t(scores[[i]])
    drug.col <- drug.pairs$drug.col[i]
    drug.row <- drug.pairs$drug.row[i]
    summary.score <- round(mean(scores.dose[-1, -1]), 3)
    # drug.col: the col-wise drug
    # drug.row: the row-wise drug
    x.conc <- as.numeric(rownames(scores.dose))  ## col-wise drug concentration
    y.conc <- as.numeric(colnames(scores.dose))  ## row-wise drug concentration

    # drug.col.conc in index
    x <- c(0:(length(x.conc) - 1))
    # drug.row.conc in index
    y <- c(0:(length(y.conc) - 1))

    # smoothing by kriging
    tmp <- cbind(expand.grid(x, y),c(as.matrix(scores.dose)))
    x.vec <- tmp[, 1]
    y.vec <- tmp[, 2]
    scores.dose.vec = tmp[, 3]
    #kriged = kriging(x.vec,y.vec,delta.dose.vec,lags=3,pixels=50)
    pixels.num <- 5 * (length(x.conc) - 1) + 2
    if (dim(scores.dose)[1] < 8) {
      kriged <- kriging(x.vec, y.vec, scores.dose.vec,lags = 2, pixels = pixels.num, model = "spherical")
    } else {
      kriged <- kriging(x.vec, y.vec, scores.dose.vec,lags = 3, pixels = pixels.num, model = "spherical")
    }
    xseq <- round(kriged[["map"]]$x / kriged$pixel)
    yseq <- round(kriged[["map"]]$y / kriged$pixel)
    a <- min(xseq):max(xseq)
    b <- min(yseq):max(yseq)
    z.len <- length(kriged[["map"]]$pred) ## n
    na <- length(a)
    nb <- length(b)
    res1 <- as.double(rep(0, na * nb)) ## c
    res2 <- as.integer(rep(0,na * nb)) ## d
    for(idx1 in 1:na) {
      for(idx2 in 1:nb) {
        for(idx3 in 1:z.len) {
          if(xseq[idx3] == a[idx1] && yseq[idx3] == b[idx2]) {
            res1[idx2+(idx1-1)*nb] <- kriged[["map"]]$pred[idx3]
            res2[idx2+(idx1-1)*nb] <- 1
            break
          }
        }
      }
    }
    res1[res2 == 0] <- NA
    c <- matrix(res1, na, nb, byrow = TRUE)
    plot.title <- paste(data$method,"synergy score:", summary.score, sep = " ")

    conc.unit <- drug.pairs$concUnit[i] ## concentration unit
    unit.text <- paste("(", conc.unit, ")", sep = "")
    file.name <- paste(drug.row, drug.col, "synergy", drug.pairs$blockIDs[i], data$method, "pdf", sep = ".")
    drug.row <- paste(drug.row, unit.text, sep = " ")
    drug.col <- paste(drug.col, unit.text, sep = " ")
    max.dose <- max(abs(max(scores.dose)), abs(min(scores.dose)))
    color.range <- round(max.dose + 5, -1)
    start.point <- -color.range
    end.point <- color.range


    if (type == "3D") {
      dev.new()
      fig <- wireframe(c, scales = list(arrows = FALSE,distance = c(0.8,0.8,0.8),col=1,cex=0.8,z = list(tick.number=7),x=list(at=seq(0, pixels.num, 5),labels=round(x.conc, 3)),y=list(at=seq(0,pixels.num,5),labels=round(y.conc,3))),
                       drape = TRUE, colorkey = list(space="top",width=0.5),
                       screen = list(z = 30, x = -55),
                       zlab = list(expression("Inhibition (%)"),rot=90,cex=1,axis.key.padding = 0),
                       xlab = list(as.character(drug.row),cex=1, rot=20),
                       ylab = list(as.character(drug.col),cex=1,rot=-50),
                       zlim = c(start.point, end.point),
                       col.regions=colorRampPalette(c("green","white","red"))(100),
                       main = plot.title,
                       at=do.breaks(c(start.point, end.point),100),
                       par.settings = list(axis.line=list(col="transparent")),
                       zoom = 1
                       #par.settings=list(layout.widths=list(left.padding=0,right.padding=0), layout.heights=list(top.padding=0, bottom.padding=0)) # margin
      )
      print(fig)
    } else if (type == "2D") {
      dev.new()
      layout(matrix(c(1, 2), nrow = 2L, ncol = 1L), heights = c(0.1, 1))
      par(mar = c(0, 6.1, 2.1, 4.1))
      suppressWarnings(par(mgp = c(3, -0.4, 0)))
      levels <- seq(start.point, end.point, by = 2)
      col <- colorpanel(end.point, low = "green", mid = "white", high = "red")
      plot.new()
      plot.window(ylim = c(0, 1), xlim = range(levels), xaxs = "i", yaxs = "i")
      rect(levels[-length(levels)],0, levels[-1L],0.3, col = col, border = NA)
      axis(3,tick = F, at = do.breaks(c(start.point, end.point), end.point/10))
      title(plot.title)
      par(mar = c(5.1,4.1,2.1,2.1))
      suppressWarnings(par(mgp = c(2,1,0)))
      plot.new()
      x.2D <- (1:dim(c)[1] - 1) / (dim(c)[1] - 1)
      y.2D <- (1:dim(c)[2] - 1) / (dim(c)[2] - 1)
      plot.window(asp = NA, xlim = range(x.2D), ylim = range(y.2D), "", xaxs = "i", yaxs = "i")
      .filled.contour(x.2D, y.2D, z = c, levels, col = col)
      #grid(dim(c)[1] - 1, dim(c)[2] - 1, col = "gray", lty = "solid", lwd = 0.3)
      box()
      mtext(drug.col, 1, cex = 1, padj = 3)
      mtext(drug.row, 2, cex = 1, las = 3, padj = -3)
      axis(side = 1, at = seq(0, 1, by = 0.2), labels = round(x.conc, 1))
      axis(side = 2, at = seq(0, 1, by = 0.2)[-1], labels = round(y.conc[-1], 1))
      fig <- recordPlot()

    } else {
      syn.3d.plot <- wireframe(c,scales = list(arrows = FALSE,distance=c(0.8,0.8,0.8),col=1,cex=0.8,z = list(tick.number=7),x=list(at=seq(0, pixels.num, 5),labels=round(x.conc, 1)),y=list(at=seq(0,pixels.num,5),labels=round(y.conc,1))),
                               drape = TRUE, colorkey = list(space="top",width=0.5),
                               screen = list(z = 30, x = -55),
                               zlab=list(expression("Inhibition (%)"),rot=90,cex=1,axis.key.padding = 0),xlab=list(as.character(drug.col),cex=1, rot=20),ylab=list(as.character(drug.row),cex=1,rot=-50),
                               zlim=c(start.point, end.point),
                               col.regions=colorRampPalette(c("green","white","red"))(100),
                               main = plot.title,
                               at=do.breaks(c(start.point, end.point),100),
                               par.settings = list(axis.line=list(col="transparent")),
                               zoom = 1
                               #par.settings=list(layout.widths=list(left.padding=0,right.padding=0), layout.heights=list(top.padding=0, bottom.padding=0)) # margin
      )

      dev.new()
      layout(matrix(c(1, 2, 3, 3), nrow = 2L, ncol = 2L), heights = c(0.1, 1))
      par(mar = c(0, 6.1, 2.1, 4.1))
      suppressWarnings(par(mgp = c(3, -1, 0)))
      levels <- seq(start.point, end.point, by = 2)
      col <- colorpanel(end.point, low = "green", mid = "white", high = "red")
      plot.new()
      plot.window(ylim = c(0, 1), xlim = range(levels), xaxs = "i", yaxs = "i")
      rect(levels[-length(levels)],0, levels[-1L],0.3, col = col, border = NA)
      axis(3,tick = F, at = do.breaks(c(start.point, end.point), end.point/10))
      title(plot.title)
      par(mar = c(5.1,4.1,2.1,2.1))
      suppressWarnings(par(mgp = c(2,1,0)))
      plot.new()
      x.2D <- (1:dim(c)[1] - 1) / (dim(c)[1] - 1)
      y.2D <- (1:dim(c)[2] - 1) / (dim(c)[2] - 1)
      plot.window(asp = NA, xlim = range(x.2D), ylim = range(y.2D), "", xaxs = "i", yaxs = "i")
      .filled.contour(x.2D, y.2D, z = c, levels, col = col)
      #grid(dim(c)[1] - 1, dim(c)[2] - 1, col = "gray", lty = "solid", lwd = 0.3)
      box()
      mtext(drug.col, 1, cex = 1, padj = 3)
      mtext(drug.row, 2, cex = 1, las = 3, padj = -3)
      axis(side = 1, at = seq(0, 1, by = 0.2), labels = round(x.conc, 1))
      axis(side = 2, at = seq(0, 1, by = 0.2)[-1], labels = round(y.conc[-1], 1))
      print(syn.3d.plot, position = c(0.5,0, 1, 1), newpage = FALSE)
      fig <- recordPlot()
    }

    if(save.file) {

      pdf(file.name, width = 12, height = 6)
      print(fig)
      dev.off()
    }

  }






}
