% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kde.algo.R
\name{kdeAlgo}
\alias{kdeAlgo}
\title{Estimation of Statistical Indicators from Interval Censored Data}
\usage{
kdeAlgo(xclass, classes, threshold = 0.6, burnin = 80, samples = 400,
  bootstrap.se = FALSE, b = 100, boundary = FALSE, bw = "nrd0",
  evalpoints = 4000, adjust = 1, custom_indicator = NULL, upper = 3,
  weights = NULL, oecd = NULL)
}
\arguments{
\item{xclass}{interval censored values; factor with ordered factor values,
as in \code{\link[Kernelheaping]{dclass}}}

\item{classes}{numeric vector of classes; Inf as last value is allowed,
as in \code{\link[Kernelheaping]{dclass}}}

\item{threshold}{used for the Head-Count Ratio and Poverty Gap, default is 60\%
of the median e.g. \code{threshold=0.6}}

\item{burnin}{burn-in sample size, as in \code{\link[Kernelheaping]{dclass}}}

\item{samples}{sampling iteration size, as in \code{\link[Kernelheaping]{dclass}}}

\item{bootstrap.se}{if \code{TRUE} standard errors for the statistical
indicators are estimated}

\item{b}{number of bootstrap iterations for the estimation of the standard errors}

\item{boundary}{TRUE for positive only data (no positive density for negative values),
as in \code{\link[Kernelheaping]{dclass}}}

\item{bw}{bandwidth selector method, defaults to "nrd0", as in
\code{\link[stats]{density}}}

\item{evalpoints}{number of evaluation grid points, as in
\code{\link[Kernelheaping]{dclass}}}

\item{adjust}{the user can multiply the bandwidth by a certain factor such
that bw=adjust*bw as in \code{\link[stats]{density}}}

\item{custom_indicator}{a list of functions containing the indicators to be
calculated additionally.
Such functions must only depend on the target variable \code{y} and the
\code{threshold}. Defaults to \code{NULL}.}

\item{upper}{if the upper bound of the upper interval is \code{Inf} e.g.
\code{(15000,Inf)}, then \code{Inf} is replaced by \code{15000*upper}}

\item{weights}{any kind of survey or design weights that will be used for the
weighted estimation of the statistical indicators}

\item{oecd}{OECD weights for equivalized household size}
}
\value{
An object of class "kdeAlgo" that provides estimates for statistical indicators
and optionally, corresponding standard error estimates. Generic
functions such as, \code{\link{print}},
\code{\link{plot}}, and \code{\link{summary}} have methods that can be used
to obtain further information. See \code{\link{kdeAlgoObject}} for a description
of components of objects of class "kdeAlgo".
}
\description{
The function applies an iterative kernel density
algorithm for the estimation of a variety of statistical indicators
(e.g. mean, median, quantiles, gini) from interval censored data. The estimation
of the standard errors is facilitated by a non-parametric bootstrap.
}
\details{
The statistical indicators are estimated using pseudo samples as
proxy for the interval censored variable. The object \code{resultX} returns the
pseudo samples of each iteration run of the KDE-algorithm.
}
\examples{
\dontrun{
# Generate data
x=rlnorm(500, meanlog = 8, sdlog = 1)
classes <- c(0,500,1000,1500,2000,2500,3000,4000,5000, 6000,8000,10000, 15000,Inf)
xclass <- cut(x,breaks=classes)
weights <- abs(rnorm(500,0,1))
oecd <- rep(seq(1,6.9,0.3),25)

# Estimate statistical indicators with default settings
Indicator <- kdeAlgo(xclass = xclass, classes = classes)

# Include custom indicators
Indicator_custom <- kdeAlgo(xclass = xclass, classes = classes,
custom_indicator = list(quant5 = function(y, threshold)
{quantile(y, probs = 0.05)}))

# Indclude survey and oecd weights
Indicator_weights <- kdeAlgo(xclass = xclass, classes = classes,
weights = weights, oecd = oecd)
} \dontshow{
# Generate data
x=rlnorm(500, meanlog = 8, sdlog = 1)
classes <- c(0,500,1000,1500,2000,2500,3000,4000,5000, 6000,8000,10000, 15000,Inf)
xclass <- cut(x,breaks=classes)

# Estimate statistical indicators
Indicator <- kdeAlgo(xclass = xclass, classes = classes, burnin = 10, samples = 40)
}
}
\references{
Walter, P., Weimer, K. (2018). Estimating Poverty and Inequality Indicators
using Interval Censored Income Data from the German Microcensus.
FU-Berlin School of Business & Economics, Discussion
Paper. \cr \cr
Groß, M., U. Rendtel, T. Schmid, S. Schmon, and N. Tzavidis (2017).
Estimating the density of ethnic minorities and aged people in Berlin:
Multivariate
Kernel Density Estimation applied to sensitive georeferenced administrative
data
protected via measurement error. Journal of the Royal Statistical Society:
Series A
(Statistics in Society), 180.
}
\seealso{
\code{\link[Kernelheaping]{dclass}}, \code{\link{print.kdeAlgo}},
\code{\link{plot.kdeAlgo}}
}
