% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reigen.R
\name{reigen}
\alias{reigen}
\title{Randomized Spectral Decomposition of a matrix (reigen).}
\usage{
reigen(A, k = NULL, p = 10, q = 1, sdist = "unif")
}
\arguments{
\item{A}{array_like \cr
a real/complex input matrix (or data frame), with dimensions \eqn{(m, n)}.}

\item{k}{int, optional \cr
determines the target rank of the low-rank decomposition and should satisfy \eqn{k << min(m,n)}.}

\item{p}{int, optional \cr
oversampling parameter for (default \eqn{p=10}).}

\item{q}{int, optional \cr
number of power iterations (default \eqn{q=1}).}

\item{sdist}{str c('normal', 'unif', 'spixel'), optional \cr
Specifies the sampling distribution. \cr
'unif' : (default) Uniform `[-1,1]`. \cr
'normal' : Normal `~N(0,1)`. \cr
'col' : Random column sampling. \cr}

\item{.............}{.}
}
\value{
\code{reigen} returns a list containing the following two components:
\item{values}{  array_like \cr
          Eigenvalues; 1-d array of length \eqn{(k)}.
}

\item{vectors}{  array_like \cr
          Eigenvectors; array with dimensions \eqn{(k, k)}. \cr
}
\item{.............}{.}
}
\description{
Computes the approximate low-rank eigendecomposition of a symmetric matrix.
}
\details{
The eigenvalue decomposition plays a central role in data analysis and scientific computing.
Randomized eigen (reigen) is a fast algorithm to compute the the approximate
low-rank eigenvalue decomposition of \eqn{A'A} given the rectangular
\eqn{(m,n)} matrix \eqn{A} using a probablistic algorithm.
Given a target rank \eqn{k << n}, \code{reigen} factors the input matrix \eqn{A} as
\eqn{A'A = V * diag(d) * V'}. The eigenvectors are the columns of the
real or complex unitary matrix \eqn{V}. The eigenvalues \eqn{d} are
non-negative and real numbers.

The parameter \eqn{p} is a oversampling parameter to improve the approximation.
A value between 2 and 10 is recommended and \eqn{p=10} is set as default.

The parameter \eqn{q} specifies the number of normalized power iterations
(subspace iterations) to reduce the approximation error. This is recommended
if the the singular values decay slowly. In practice 1 or 2 iterations
archive good results, however, computing power iterations increases the
computational time. The number of power iterations is set to \eqn{q=1} by default.

If \eqn{k > (min(n,m)/1.5)}, a deterministic partial or truncated \code{\link{eigen}}
algorithm might be faster.
}
\note{
The eigenvectors are not unique and only defined up to sign
(a constant of modulus one in the complex case).
}
\examples{
library(rsvd)
set.seed(123)

#Create real random test matrix with dimension (m, n) and rank k
m = 10
n = 5
k = 3
A <- matrix(runif(m*k), m, k)
A <- A \%*\% t(A)
A <- A[,1:n]

AtA = t(A) \%*\% A

# Randomized low-rank eigenvalue decomposition k=3
reigen.out <- reigen(A, k=3)
AtA.re = reigen.out$vectors \%*\% diag(reigen.out$values) \%*\% t(reigen.out$vectors)
100 * norm( AtA - AtA.re, 'F') / norm( AtA,'F') #Percentage reconstruction error << 1e-8
print(reigen.out$values) # print eigenvalues

# Compare with the deterministic eigenvalue decomposition
eigen.out <- eigen(AtA)
AtA.re2 = eigen.out$vectors \%*\% diag(eigen.out$values) \%*\% t(eigen.out$vectors)
100 * norm( AtA - AtA.re2, 'F') / norm( AtA,'F') #Percentage reconstruction error << 1e-8
print(eigen.out$values) # print eigenvalues
all.equal(eigen.out$values[1:k], reigen.out$values)
}
\author{
N. Benjamin Erichson, \email{nbe@st-andrews.ac.uk}
}
\references{
\itemize{
  \item  [1] N. Halko, P. Martinsson, and J. Tropp.
         "Finding structure with randomness: probabilistic
         algorithms for constructing approximate matrix
         decompositions" (2009).
         (available at arXiv \url{http://arxiv.org/abs/0909.4061}).
  \item  [2] S. Voronin and P.Martinsson.
         "RSVDPACK: Subroutines for computing partial singular value
         decompositions via randomized sampling on single core, multi core,
         and GPU architectures" (2015).
         (available at `arXiv \url{http://arxiv.org/abs/1502.05366}).
}
}
\seealso{
\code{\link{rsvd}}, \code{\link{rpca}}, \code{\link{eigen}}
}

