% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{coefPlot}
\alias{coefPlot}
\alias{coefPlot.rlars}
\alias{coefPlot.grplars}
\alias{coefPlot.tslarsP}
\alias{coefPlot.seqModel}
\alias{coefPlot.tslars}
\alias{coefPlot.sparseLTS}
\alias{coefPlot.setupCoefPlot}
\title{Coefficient plot of a sequence of regression models}
\usage{
coefPlot(object, ...)

\method{coefPlot}{seqModel}(object, zeros = FALSE, labels = NULL, ...)

\method{coefPlot}{tslars}(object, p, zeros = FALSE, labels = NULL, ...)

\method{coefPlot}{sparseLTS}(
  object,
  fit = c("reweighted", "raw", "both"),
  zeros = FALSE,
  labels = NULL,
  ...
)

\method{coefPlot}{setupCoefPlot}(
  object,
  abscissa = NULL,
  size = c(0.5, 2, 4),
  offset = 1,
  facets = object$facets,
  ...
)
}
\arguments{
\item{object}{the model fit to be plotted.}

\item{\dots}{additional arguments to be passed down, eventually to
\code{\link[ggplot2]{geom_line}} and \code{\link[ggplot2]{geom_point}}.}

\item{zeros}{a logical indicating whether predictors that never enter the
model and thus have zero coefficients should be included in the plot
(\code{TRUE}) or omitted (\code{FALSE}, the default).  This is useful if the
number of predictors is much larger than the number of observations, in
which case many coefficients are never nonzero.}

\item{labels}{an optional character vector containing labels for the
predictors.  Plotting labels can be suppressed by setting this to
\code{NA}.}

\item{p}{an integer giving the lag length for which to produce the plot
(the default is to use the optimal lag length).}

\item{fit}{a character string specifying for which estimator to produce the
plot.  Possible values are \code{"reweighted"} (the default) for the
reweighted fits, \code{"raw"} for the raw fits, or \code{"both"} for both
estimators.}

\item{abscissa}{a character string specifying what to plot on the
\eqn{x}-axis.  For objects inheriting from class \code{"seqModel"}, possible
values are \code{"step"} for the step number (the default), or \code{"df"}
for the degrees of freedom.  For code{"sparseLTS"} objects, possible values
are code{"lambda"} for the value of the penalty parameter (the default), or
\code{"step"} for the step number.}

\item{size}{a numeric vector of length three giving the line width, the
point size and the label size, respectively.}

\item{offset}{an integer giving the offset of the labels from the
corresponding coefficient values from the last step (i.e., the number of
blank characters to be prepended to the label).}

\item{facets}{a faceting formula to override the default behavior.  If
supplied, \code{\link[ggplot2]{facet_wrap}} or
\code{\link[ggplot2]{facet_grid}} is called depending on whether the formula
is one-sided or two-sided.}
}
\value{
An object of class \code{"ggplot"} (see \code{\link[ggplot2]{ggplot}}).
}
\description{
Produce a plot of the coefficients from a sequence of regression models,
such as submodels along a robust or groupwise least angle regression
sequence, or sparse least trimmed squares regression models for a grid of
values for the penalty parameter.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points


## robust LARS
# fit model
fitRlars <- rlars(x, y, sMax = 10)
# create plot
coefPlot(fitRlars)


## sparse LTS over a grid of values for lambda
# fit model
frac <- seq(0.2, 0.05, by = -0.05)
fitSparseLTS <- sparseLTS(x, y, lambda = frac, mode = "fraction")
# create plot
coefPlot(fitSparseLTS)
coefPlot(fitSparseLTS, fit = "both")
}
\seealso{
\code{\link[ggplot2]{ggplot}}, \code{\link{rlars}},
\code{\link{grplars}}, \code{\link{rgrplars}}, \code{\link{tslarsP}},
\code{\link{rtslarsP}}, \code{\link{tslars}}, \code{\link{rtslars}},
\code{\link{sparseLTS}}
}
\author{
Andreas Alfons
}
\keyword{hplot}
