#'Plot the variants of a haplohh object
#'@description Plot the variants of a haplohh object. This method is intended
#'for visualization of very small data sets such as the examples
#'provided by the package.
#'@param x an object of class \code{haplo-hh} generated by \code{\link{data2haplohh}}.
#'@param mrk the focal marker. Used only, if alleles are grouped or (de-)selected.
#'@param allele if \code{NA} (default), haplotypes of all alleles are plotted,
#'otherwise for the specified alleles. Alleles must be specified by their
#'internal coding, i.e. '0' for ancestral resp. major allele, etc.
#'Haplotypes with missing values at the focal marker can only be selected in
#'combination with genotyped alleles, e.g. as \code{c(1,NA)}.
#'@param group_by_allele logical. If \code{TRUE}, group chromosomes
#'by their allele at the focal marker; alleles are ordered by their internal coding
#'unless parameter \code{alleles} is specified. If \code{FALSE} (default), haplotypes are drawn by
#'their order in the input file.
#'@param ignore.distances logical. If \code{TRUE}, markers are drawn equally-spaced.
#'@param col color for each allele (as coded internally).
#'@param linecol the color of the background lines. If more than one color is specified and sequences
#'sorted by the marker allele, the specified colors are used to distinguish the alleles; otherwise
#'consecutive sequences are set into the specified colors.
#'@param mrk.col color of the vertical line at the focal marker position.
#'@param pch symbol used for markers. See \code{\link[graphics]{points}}.
#'@param cex relative size of marker symbol. See \code{\link[graphics]{points}}.
#'@param lwd line width.
#'@param hap.names a vector containing the names of chromosomes.
#'@param mrk.names a vector containing the names of markers.
#'@param cex.lab.hap relative letter size of haplotype labels. See \code{\link[graphics]{par}}.
#'@param cex.lab.mrk relative letter size of marker labels. See \code{\link[graphics]{par}}.
#'@param family.lab font family for labels. See \code{\link[graphics]{par}}.
#'@param offset.lab.hap offset of haplotype labels. See \code{\link[graphics]{par}}.
#'@param offset.lab.mrk offset of marker labels. See \code{\link[graphics]{par}}.
#'@param pos.lab.hap position of haplotype labels. Either \code{"left"} (default),
#'\code{"right"}, \code{"none"} or \code{"both"}.
#'@param pos.lab.mrk position of marker labels. Either \code{"top"} (default) or \code{"none"}.
#'@param srt.hap rotation of haplotype labels (see \code{\link[graphics]{par}}).
#'@param srt.mrk rotation of marker labels (see \code{\link[graphics]{par}}).
#'@param ... other parameters to be passed to \code{\link[graphics]{plot.default}}.
#'@details Specifying a haplohh-object with more than 4096 haplotypes or
#'markers produces an error.
#'@seealso \code{\link{calc_haplen}}, \code{\link{plot.furcation}}.
#'@examples #example haplohh object
#'make.example.files()
#'hh <- data2haplohh("example1.vcf")
#'plot(hh)
#'hh <- data2haplohh("example2.vcf", min_perc_geno.mrk = 50)
#'plot(hh)
#'remove.example.files()
#'@export
#'@importFrom graphics lines segments text
plot.haplohh <-
  function(x,
           mrk = NA,
           allele = NA,
           group_by_allele = FALSE,
           ignore.distances = FALSE,
           col = c("blue", "red", "violet", "orange"),
           linecol = "gray",
           mrk.col = "gray",
           pch = 19,
           cex = 1,
           lwd = 1,
           hap.names = NULL,
           mrk.names = NULL,
           cex.lab.hap = 0.8,
           cex.lab.mrk = 0.8,
           family.lab = "",
           offset.lab.hap = 0.5,
           offset.lab.mrk = 0.25,
           pos.lab.hap = "left",
           pos.lab.mrk = "top",
           srt.hap = 0,
           srt.mrk = 0,
           ...) {
    # arbitrary limit on haplotypes and markers
    MAX <- 4096L
    
    ##check parameters
    if (!is.haplohh(x)) {
      stop("The data is not a valid object of a haplohh", call. = FALSE)
    }
    
    if (nrow(x@haplo) > MAX) {
      stop(paste("The object contains more than", MAX, "haplotpyes."),
           call. = FALSE)
    }
    
    if (ncol(x@haplo) > MAX) {
      stop(paste("The object contains more than", MAX, "markers."),
           call. = FALSE)
    }
    
    if (!is.na(mrk)) {
      if (is.numeric(mrk)) {
        mrk <- as.integer(mrk)
        if (mrk < 1) {
          stop(paste0("No marker numbers smaller than 1 allowed."),
               call. = FALSE)
        }
        if (mrk > nmrk(x)) {
          stop(
            paste0(
              "The marker number ",
              mrk,
              " is bigger than the number of markers in the data set (",
              nmrk(x),
              ")"
            ),
            call. = FALSE
          )
        }
      } else{
        mrk <- as.character(mrk)
        if (!(mrk %in% mrk.names(x))) {
          stop(paste0("Marker '", mrk, "' not found."), call. = FALSE)
        }
        mrk <- which(mrk.names(x) == mrk)
      }
    }
    
    dot_args <- list(...)
    
    if (!is.null(dot_args$xlim)) {
      if (!is.numeric(dot_args$xlim) |
          length(dot_args$xlim) != 2 |
          dot_args$xlim[2] < dot_args$xlim[1]) {
        stop("Incorrect specification of xlim.", call. = FALSE)
      }
    }
    
    if (!is.null(hap.names)) {
      if (length(hap.names) != nrow(x@haplo)) {
        stop(
          "Number of specified haplotype names has to be equal to number of haplotypes.",
          call. = FALSE
        )
      }
      rownames(x@haplo) <- hap.names
    }
    
    if (!is.null(mrk.names)) {
      if (length(mrk.names) != ncol(x@haplo)) {
        stop("Number of specified marker names has to be equal to number of markers.",
             call. = FALSE)
      }
      colnames(x@haplo) <- mrk.names
    }
    
    # at least one allele specified
    if (sum(!is.na(allele)) > 0) {
      if (is.na(mrk)) {
        stop("Marker has to be specified.", call. = FALSE)
      }
      if (!is.numeric(allele)) {
        stop("Alleles have to be specified by integers 0,1,2,... or NA",
             call. = FALSE)
      }
    }
    
    if (group_by_allele & is.na(mrk)) {
      stop("Marker has to be specified.", call. = FALSE)
    }
    
    if (sum(!is.na(allele)) > 0) {
      if (anyDuplicated(allele)) {
        stop("Repeated specification of the same allele.", call. = FALSE)
      }
      focal_alleles <- unique(x@haplo[, mrk])
      for (i in allele) {
        if (!(i %in% focal_alleles)) {
          stop(paste0("No allele ", i, " found at marker ", mrk, " ."),
               call. = FALSE)
        }
      }
      
      #remove non-specified alleles
      hh_subset <-
        subset(
          x,
          select.hap = x@haplo[, mrk] %in% allele,
          # at least one genotyped haplotype must be present
          min_perc_geno.mrk = 1 / nhap(x) * 100
        )
    } else{
      allele <- sort(unique(x@haplo[, mrk]))
      
      if (anyNA(x@haplo[, mrk])) {
        allele <- c(allele, NA)
      }
      
      hh_subset <- x
    }
    
    # perform plot
    if (group_by_allele) {
      #sort in order of specified alleles ("radix" -> stable sort)
      ord <- order(vapply(hh_subset@haplo[, mrk], function(x) {
        ifelse(is.na(x), which(is.na(allele)), which(x == allele))
      }, FUN.VALUE = 0L),
      method = "radix")
      hh_subset@haplo <- hh_subset@haplo[ord, ]
    }
    
    p <- floor(log(max(hh_subset@positions), 1000))
    ## only shrink big scales, but never magnify small ones (p<0)
    scale <- 1000 ** max(0, p)
    ## no unit if p < 0
    unit <- c("", "(bp)", "(kb)", "(Mb)", "(Gb)")[max(-1, p)  + 2]
    
    if (is.null(dot_args$xlim)) {
      dot_args$xlim <- range(hh_subset@positions) / scale
    } else{
      dot_args$xlim <- dot_args$xlim / scale
    }
    
    dot_args$ylim <- c(0, 1)
    
    if (is.null(dot_args$xlab)) {
      dot_args$xlab <- paste("Position", unit)
    }
    
    if (is.null(dot_args$ylab)) {
      dot_args$ylab <- ""
    }
    
    if (is.null(dot_args$bty)) {
      dot_args$bty <- "n"
    }
    
    if (is.null(dot_args$yaxt)) {
      dot_args$yaxt <- "n"
    }
    
    ##perform plot
    dot_args$xaxs <- "r"
    
    #invisible plot to get coordinate system
    do.call("plot", c(list(NULL),
                      dot_args))
    
    if (ignore.distances) {
      #calculate equal distances with the same range
      n <- length(x@positions)
      xcoord <-
        (min(x@positions) * ((n - 1):0) + max(x@positions) * (0:(n - 1))) / (n -
                                                                               1)
      if (is.null(dot_args$xaxt)) {
        segments(xcoord / scale, 0, x@positions / scale, -0.04, lty = 3)
      }
    } else{
      xcoord <- x@positions
    }
    
    #dashed vertical line at focal mrk
    if (!is.na(mrk)) {
      abline(v = xcoord[mrk] / scale,
             lty = 2,
             col = mrk.col)
    }
    
    for (i in seq_len(nrow(hh_subset@haplo))) {
      #draw haplo-lines
      y <- 1 - (i - 0.5) / nrow(hh_subset@haplo)
      lines(range(hh_subset@positions) / scale,
            rep(y, 2),
            col = linecol[ifelse(is.na(mrk), i, hh_subset@haplo[i, mrk]) %% length(linecol) +
                            1L],
            lwd = lwd)
      
      #add markers
      points(
        x = xcoord / scale,
        y = rep(y, ncol(hh_subset@haplo)),
        cex = cex,
        col = col[((hh_subset@haplo[i, ]) %% length(col)) + 1L],
        pch = pch[((hh_subset@haplo[i, ]) %% length(pch)) + 1L]
      )
      
      #add labels
      if (pos.lab.hap != "none") {
        if (pos.lab.hap == "left" || pos.lab.hap == "both") {
          text(
            x = dot_args$xlim[1],
            y = rep(1 - (i - 0.5) / nrow(hh_subset@haplo), 2),
            family = family.lab,
            labels = hap.names(hh_subset)[i],
            pos = 2,
            offset = offset.lab.hap,
            cex = cex.lab.hap,
            srt = srt.hap,
            xpd = TRUE
          )
        }
        if (pos.lab.hap == "right" || pos.lab.hap == "both") {
          text(
            x = dot_args$xlim[2],
            y = rep(1 - (i - 0.5) / nrow(hh_subset@haplo), 2),
            family = family.lab,
            labels = hap.names(hh_subset)[i],
            pos = 4,
            offset = offset.lab.hap,
            cex = cex.lab.hap,
            srt = srt.hap,
            xpd = TRUE
          )
        }
      }
    }
    
    if (pos.lab.mrk != "none") {
      text(
        x = xcoord / scale,
        y = rep(1, ncol(hh_subset@haplo)),
        family = family.lab,
        labels = mrk.names(hh_subset),
        pos = 3,
        offset = offset.lab.mrk,
        cex = cex.lab.mrk,
        srt = srt.mrk,
        xpd = TRUE
      )
    }
  }
