% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ranger.R
\name{ranger}
\alias{ranger}
\title{Ranger}
\usage{
ranger(formula = NULL, data = NULL, num.trees = 500, mtry = NULL,
  importance = "none", write.forest = TRUE, probability = FALSE,
  min.node.size = NULL, replace = TRUE, sample.fraction = ifelse(replace,
  1, 0.632), case.weights = NULL, splitrule = NULL, num.random.splits = 1,
  alpha = 0.5, minprop = 0.1, split.select.weights = NULL,
  always.split.variables = NULL, respect.unordered.factors = NULL,
  scale.permutation.importance = FALSE, keep.inbag = FALSE,
  holdout = FALSE, num.threads = NULL, save.memory = FALSE,
  verbose = TRUE, seed = NULL, dependent.variable.name = NULL,
  status.variable.name = NULL, classification = NULL)
}
\arguments{
\item{formula}{Object of class \code{formula} or \code{character} describing the model to fit. Interaction terms supported only for numerical variables.}

\item{data}{Training data of class \code{data.frame}, \code{matrix} or \code{gwaa.data} (GenABEL).}

\item{num.trees}{Number of trees.}

\item{mtry}{Number of variables to possibly split at in each node. Default is the (rounded down) square root of the number variables.}

\item{importance}{Variable importance mode, one of 'none', 'impurity', 'permutation'. The 'impurity' measure is the Gini index for classification and the variance of the responses for regression. For survival, only 'permutation' is available.}

\item{write.forest}{Save \code{ranger.forest} object, required for prediction. Set to \code{FALSE} to reduce memory usage if no prediction intended.}

\item{probability}{Grow a probability forest as in Malley et al. (2012).}

\item{min.node.size}{Minimal node size. Default 1 for classification, 5 for regression, 3 for survival, and 10 for probability.}

\item{replace}{Sample with replacement.}

\item{sample.fraction}{Fraction of observations to sample. Default is 1 for sampling with replacement and 0.632 for sampling without replacement.}

\item{case.weights}{Weights for sampling of training observations. Observations with larger weights will be selected with higher probability in the bootstrap (or subsampled) samples for the trees.}

\item{splitrule}{Splitting rule. For classification and probability estimation "gini" or "extratrees" with default "gini". For regression "variance", "extratrees" or "maxstat" with default "variance". For survival "logrank", "extratrees", "C" or "maxstat" with default "logrank".}

\item{num.random.splits}{For "extratrees" splitrule.: Number of random splits to consider for each candidate splitting variable.}

\item{alpha}{For "maxstat" splitrule: Significance threshold to allow splitting.}

\item{minprop}{For "maxstat" splitrule: Lower quantile of covariate distribtuion to be considered for splitting.}

\item{split.select.weights}{Numeric vector with weights between 0 and 1, representing the probability to select variables for splitting. Alternatively, a list of size num.trees, containing split select weight vectors for each tree can be used.}

\item{always.split.variables}{Character vector with variable names to be always selected in addition to the \code{mtry} variables tried for splitting.}

\item{respect.unordered.factors}{Handling of unordered factor covariates. One of 'ignore', 'order' and 'partition'. For the "extratrees" splitrule the default is "partition" for all other splitrules 'ignore'. Alternatively TRUE (='order') or FALSE (='ignore') can be used. See below for details.}

\item{scale.permutation.importance}{Scale permutation importance by standard error as in (Breiman 2001). Only applicable if permutation variable importance mode selected.}

\item{keep.inbag}{Save how often observations are in-bag in each tree.}

\item{holdout}{Hold-out mode. Hold-out all samples with case weight 0 and use these for variable importance and prediction error.}

\item{num.threads}{Number of threads. Default is number of CPUs available.}

\item{save.memory}{Use memory saving (but slower) splitting mode. No effect for survival and GWAS data. Warning: This option slows down the tree growing, use only if you encounter memory problems.}

\item{verbose}{Show computation status and estimated runtime.}

\item{seed}{Random seed. Default is \code{NULL}, which generates the seed from \code{R}.}

\item{dependent.variable.name}{Name of dependent variable, needed if no formula given. For survival forests this is the time variable.}

\item{status.variable.name}{Name of status variable, only applicable to survival data and needed if no formula given. Use 1 for event and 0 for censoring.}

\item{classification}{Only needed if data is a matrix. Set to \code{TRUE} to grow a classification forest.}
}
\value{
Object of class \code{ranger} with elements
  \item{\code{forest}}{Saved forest (If write.forest set to TRUE). Note that the variable IDs in the \code{split.varIDs} object do not necessarily represent the column number in R.}
  \item{\code{predictions}}{Predicted classes/values, based on out of bag samples (classification and regression only).}
  \item{\code{variable.importance}}{Variable importance for each independent variable.}
  \item{\code{prediction.error}}{Overall out of bag prediction error. For classification this is the fraction of missclassified samples, for regression the mean squared error and for survival one minus Harrell's c-index.}
  \item{\code{r.squared}}{R squared. Also called explained variance or coefficient of determination (regression only). Computed on out of bag data.}
  \item{\code{confusion.matrix}}{Contingency table for classes and predictions based on out of bag samples (classification only).}
  \item{\code{unique.death.times}}{Unique death times (survival only).}
  \item{\code{chf}}{Estimated cumulative hazard function for each sample (survival only).}
  \item{\code{survival}}{Estimated survival function for each sample (survival only).}
  \item{\code{call}}{Function call.}
  \item{\code{num.trees}}{Number of trees.}
  \item{\code{num.independent.variables}}{Number of independent variables.}
  \item{\code{mtry}}{Value of mtry used.}
  \item{\code{min.node.size}}{Value of minimal node size used.}
  \item{\code{treetype}}{Type of forest/tree. classification, regression or survival.}
  \item{\code{importance.mode}}{Importance mode used.}
  \item{\code{num.samples}}{Number of samples.}
  \item{\code{inbag.counts}}{Number of times the observations are in-bag in the trees.}
}
\description{
Ranger is a fast implementation of Random Forest (Breiman 2001) or recursive partitioning, particularly suited for high dimensional data.
Classification, regression, and survival forests are supported.
Classification and regression forests are implemented as in the original Random Forest (Breiman 2001), survival forests as in Random Survival Forests (Ishwaran et al. 2008).
}
\details{
The tree type is determined by the type of the dependent variable.
For factors classification trees are grown, for numeric values regression trees and for survival objects survival trees.
The Gini index is used as default splitting rule for classification.
For regression, the estimated response variances or maximally selected rank statistics (Wright et al. 2016) can be used.
For Survival the log-rank test, a C-index based splitting rule (Schmid et al. 2015) and maximally selected rank statistics (Wright et al. 2016) are available.
For all tree types, forests of extremely randomized trees (Geurts et al. 2006) can be grown.

With the \code{probability} option and factor dependent variable a probability forest is grown.
Here, the node impurity is used for splitting, as in classification forests.
Predictions are class probabilities for each sample.
In contrast to other implementations, each tree returns a probability estimate and these estimates are averaged for the forest probability estimate.
For details see Malley et al. (2012).

Note that for classification and regression nodes with size smaller than \code{min.node.size} can occur, as in original Random Forests.
For survival all nodes contain at \code{min.node.size} samples. 
Variables selected with \code{always.split.variables} are tried additionaly to the mtry variables randomly selected.
In \code{split.select.weights} variables weighted with 0 are never selected and variables with 1 are always selected. 
Weights do not need to sum up to 1, they will be normalized later. 
The weights are assigned to the variables in the order they appear in the formula or in the data if no formula is used.
Names of the \code{split.select.weights} vector are ignored.
The usage of \code{split.select.weights} can increase the computation times for large forests.

Unordered factor covariates can be handled in 3 different ways by using \code{respect.unordered.factors}: 
For 'ignore' all factors are regarded ordered, for 'partition' all possible 2-partitions are considered for splitting. 
For 'order' and 2-class classification the factor levels are ordered by their proportion falling in the second class, for regression by their mean response, as described in Hastie et al. (2009), chapter 9.2.4.
For multiclass classification and survival outcomes, 'order' is experimental and should be used with care.
The use of 'order' is recommended for 2-class classification and regression, as it computationally fast and can handle an unlimited number of factor levels. 
Note that the factors are only reordered once and not again in each split.

For a large number of variables and data frames as input data the formula interface can be slow or impossible to use.
Alternatively \code{dependent.variable.name} (and \code{status.variable.name} for survival) can be used.
Consider setting \code{save.memory = TRUE} if you encounter memory problems for very large datasets, but be aware that this option slows down the tree growing.

For GWAS data consider combining \code{ranger} with the \code{GenABEL} package. 
See the Examples section below for a demonstration using \code{Plink} data.
All SNPs in the \code{GenABEL} object will be used for splitting. 
To use only the SNPs without sex or other covariates from the phenotype file, use \code{0} on the right hand side of the formula. 
Note that missing values are treated as an extra category while splitting.

See \url{https://github.com/imbs-hl/ranger} for the development version.

With recent R versions, multithreading on Windows platforms should just work. 
If you compile yourself, the new RTools toolchain is required.
}
\examples{
require(ranger)

## Classification forest with default settings
ranger(Species ~ ., data = iris)

## Prediction
train.idx <- sample(nrow(iris), 2/3 * nrow(iris))
iris.train <- iris[train.idx, ]
iris.test <- iris[-train.idx, ]
rg.iris <- ranger(Species ~ ., data = iris.train, write.forest = TRUE)
pred.iris <- predict(rg.iris, dat = iris.test)
table(iris.test$Species, pred.iris$predictions)

## Variable importance
rg.iris <- ranger(Species ~ ., data = iris, importance = "impurity")
rg.iris$variable.importance

## Survival forest
require(survival)
rg.veteran <- ranger(Surv(time, status) ~ ., data = veteran)
plot(rg.veteran$unique.death.times, rg.veteran$survival[1,])

## Alternative interface
ranger(dependent.variable.name = "Species", data = iris)

\dontrun{
## Use GenABEL interface to read Plink data into R and grow a classification forest
## The ped and map files are not included
library(GenABEL)
convert.snp.ped("data.ped", "data.map", "data.raw")
dat.gwaa <- load.gwaa.data("data.pheno", "data.raw")
phdata(dat.gwaa)$trait <- factor(phdata(dat.gwaa)$trait)
ranger(trait ~ ., data = dat.gwaa)
}

}
\references{
\itemize{
  \item Wright, M. N. & Ziegler, A. (2017). ranger: A Fast Implementation of Random Forests for High Dimensional Data in C++ and R. J Stat Softw 77:1-17. \url{http://dx.doi.org/10.18637/jss.v077.i01}.
  \item Schmid, M., Wright, M. N. & Ziegler, A. (2016). On the use of Harrell's C for clinical risk prediction via random survival forests. Expert Syst Appl 63:450-459. \url{http://dx.doi.org/10.1016/j.eswa.2016.07.018}. 
  \item Wright, M. N., Dankowski, T. & Ziegler, A. (2017). Unbiased split variable selection for random survival forests using maximally selected rank statistics. Stat Med. \url{http://dx.doi.org/10.1002/sim.7212}.
  \item Breiman, L. (2001). Random forests. Mach Learn, 45(1), 5-32. \url{http://dx.doi.org/10.1023/A:1010933404324}. 
  \item Ishwaran, H., Kogalur, U. B., Blackstone, E. H., & Lauer, M. S. (2008). Random survival forests. Ann Appl Stat 2:841-860. \url{http://dx.doi.org/10.1097/JTO.0b013e318233d835}. 
  \item Malley, J. D., Kruppa, J., Dasgupta, A., Malley, K. G., & Ziegler, A. (2012). Probability machines: consistent probability estimation using nonparametric learning machines. Methods Inf Med 51:74-81. \url{http://dx.doi.org/10.3414/ME00-01-0052}.
  \item Hastie, T., Tibshirani, R., Friedman, J. (2009). The Elements of Statistical Learning. Springer, New York. 2nd edition.
  \item Geurts, P., Ernst, D., Wehenkel, L. (2006). Extremely randomized trees. Mach Learn 63:3-42. \url{http://dx.doi.org/10.1007/s10994-006-6226-1}. 
  }
}
\seealso{
\code{\link{predict.ranger}}
}
\author{
Marvin N. Wright
}
