## Register S3 methods for dplyr verbs to avoid having to export generics
.onLoad <- function(libname, pkgname) {
  register_s3_method("dplyr", "arrange", "ma_psychmeta")
  register_s3_method("dplyr", "arrange", "ma_table")

  register_s3_method("dplyr", "filter", "ma_psychmeta")
  register_s3_method("dplyr", "filter", "ma_table")

  register_s3_method("dplyr", "mutate", "ma_psychmeta")
  register_s3_method("dplyr", "mutate", "ma_table")

  register_s3_method("dplyr", "rename", "ma_psychmeta")
  register_s3_method("dplyr", "rename", "ma_table")

  register_s3_method("dplyr", "select", "ma_psychmeta")
  register_s3_method("dplyr", "select", "ma_table")


  invisible()
}

## Messages to be displayed when the user loads psychmeta:
#' @importFrom rlang .data
.onAttach <- function(libname, pkgname) {
    version <- read.dcf(file=system.file("DESCRIPTION", package=pkgname), fields="Version")
    packageStartupMessage(crayon::white("----------------------------------------------------- ", crayon::bold(paste(pkgname, "version", version)), " --"))
    packageStartupMessage("\nPlease report any bugs to ", crayon::italic("github.com/psychmeta/psychmeta/issues"), "\nor ", crayon::italic("issues@psychmeta.com"))
    packageStartupMessage("\nWe work hard to produce these open-source tools for the R community.  \nPlease cite psychmeta when you use it in your research: \n  Dahlke, J. A., & Wiernik, B. M. (2019). psychmeta: An R package for \n     psychometric meta-analysis. ",
                          crayon::italic("Applied Psychological Measurement, 43"), "(5), \n     415\u2013416. https://doi.org/10.1177/0146621618795933")
    packageStartupMessage("\nFind info about psychmeta on the web at ", crayon::italic("psychmeta.com"), " and ", crayon::italic("twitter.com/psychmetaR"))

    # Check if there is an internet connection. If there is, check whether the local version of psychmeta is up to date compared to the CRAN version.
    suppressWarnings(version_check <-
        try(unlist(strsplit(
             rawToChar(
                  curl::curl_fetch_memory("https://CRAN.R-project.org/package=psychmeta")$content
                  ),
             "\n")),
            silent = TRUE))
    if(!inherits(version_check, "try-error")) {

         check_version <- function(cran_version, sys_version){
              cran_v_char <- cran_version
              cran_v_num <- as.numeric(stringr::str_split(cran_v_char, "[.]")[[1]])
              sys_v_char <- stringr::str_split(sys_version, "[.]")[[1]]
              sys_v_num <- as.numeric(sys_v_char)

              if(length(cran_v_num) == 3) cran_v_num <- c(cran_v_num, "0")
              if(length(sys_v_num) == 3) sys_v_num <- c(sys_v_num, "0")

              .cran_v_num <- cran_v_num
              .sys_v_num <- sys_v_num
              desired_digits <- 4
              necessary_digits_cran <- desired_digits - nchar(cran_v_num)
              necessary_digits_sys <- desired_digits - nchar(sys_v_num)
              for(i in 1:length(sys_v_num)){
                   .cran_v_num[i] <- paste0(c(rep("0", necessary_digits_cran[i]), .cran_v_num[i]), collapse = "")
                   .sys_v_num[i] <- paste0(c(rep("0", necessary_digits_sys[i]), .sys_v_num[i]), collapse = "")
              }

              .cran_v_num <- paste(.cran_v_num, collapse = "")
              .sys_v_num <- paste(.sys_v_num, collapse = "")

              if(cran_v_char == sys_version){
                   out_of_date <- FALSE
                   ahead_of_cran <- FALSE
                   best_version <- c(Current = cran_version)
              }else{
                   best_version <- sort(c(CRAN = .cran_v_num, Local = .sys_v_num), decreasing = TRUE)[1]
                   out_of_date <- names(best_version) == "CRAN"
                   ahead_of_cran <- names(best_version) == "Local"
                   if(names(best_version) == "CRAN") best_version <- c(CRAN = cran_version)
                   if(names(best_version) == "Local") best_version <- c(Local = sys_version)
              }

              vcheck_devnum <- zapsmall(as.numeric(sys_v_num[4])) > 0

              as.data.frame(list(best_version = names(best_version),
                                 cran_version = cran_version,
                                 sys_version = sys_version,
                                 out_of_date = out_of_date,
                                 ahead_of_cran = ahead_of_cran,
                                 development = vcheck_devnum), stringsAsFactors = FALSE)
         }

         cran_v_char <- stringr::str_extract(version_check[which(stringr::str_detect(version_check, "Version:")) + 1], "\\d\\.\\d\\.\\d")
         if(length(cran_v_char) > 0){
           vcheck <- check_version(cran_version = cran_v_char, sys_version = version)
           
           use_symbols <- .support_unicode()
           
           packageStartupMessage(crayon::white("\n", paste0("-----------------------------------------------------", paste0(rep_len("-", nchar(paste(pkgname, "version", version)) - 13), collapse = ""), " "), crayon::bold("Version check"), " --"))
           if(vcheck$best_version == "CRAN"){
             version_message <- "Oh no! It looks like your copy of psychmeta is out of date!"
             if(use_symbols) version_message <- paste0(cli::symbol$cross, " ", version_message)
             packageStartupMessage(crayon::red(version_message))
             packageStartupMessage("No worries, it's easy to obtain the latest version - just run the following command: \n")
             packageStartupMessage('                       install.packages("psychmeta")')
           }else if(vcheck$best_version == "Current"){
             version_message <- "Yay! Your copy of psychmeta is up to date!"
             if(use_symbols) version_message <- paste0(cli::symbol$tick, " ", version_message)
             packageStartupMessage(crayon::green(version_message))
           }else if(vcheck$best_version == "Local"){
             version_message <- "Kudos! Your copy of psychmeta is more recent than the current CRAN release!"
             if(use_symbols) version_message <- paste0(cli::symbol$tick, " ", version_message)
             packageStartupMessage(crayon::green(version_message))
           }
           
         }
         
         sys_v_char <- stringr::str_split(version, "[.]")[[1]]
         sys_v_num <- as.numeric(sys_v_char)
         if(length(sys_v_num) == 3) sys_v_num <- c(sys_v_num, 0)
         if(sys_v_num[4] > 0)
           packageStartupMessage(paste0("NOTE: You are currently using an UNRELEASED development build (augmentation of release v", paste(sys_v_char[1:3], collapse = "."), ")"))   
    }
}

.support_unicode <- function(override = NULL) {
    if(!is.null(override)) override else (l10n_info()$`UTF-8` | if(!is.null(.Options$cli.unicode)) isTRUE(.Options$cli.unicode) else FALSE | nzchar(Sys.getenv("RSTUDIO_USER_IDENTITY")) )
}


#' Retrieve the NEWS file for the psychmeta package
#'
#' @description
#' This function gives a shortcut to the \code{utils::news(package = "psychmeta")} function and displays psychmeta's NEWS file, which contains version information, outlines additions and changes to the package, and describes other updates.
#'
#' @export
#'
#' @importFrom utils news
#'
#' @examples
#' psychmeta_news()
psychmeta_news <- function(){
     news(package = "psychmeta")
}

register_s3_method <- function(pkg, generic, class, fun = NULL) {
  stopifnot(is.character(pkg), length(pkg) == 1)
  envir <- asNamespace(pkg)

  stopifnot(is.character(generic), length(generic) == 1)
  stopifnot(is.character(class), length(class) == 1)
  if (is.null(fun)) {
    fun <- get(paste0(generic, ".", class), envir = parent.frame())
  }
  stopifnot(is.function(fun))


  if (pkg %in% loadedNamespaces()) {
    registerS3method(generic, class, fun, envir = envir)
  }

  # Always register hook in case package is later unloaded & reloaded
  setHook(
    packageEvent(pkg, "onLoad"),
    function(...) {
      registerS3method(generic, class, fun, envir = envir)
    }
  )
}

